/*****************************************************************************

GRAOUMF TRACKER 2

Copyright (c) 1996 - 2002 Laurent de Soras

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author : laurent@ohmforce.com
More information about this license : http://www.gnu.org/licenses/gpl.html

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include	<stdio.h>
#include	<stdlib.h>
#include	<string.h>
#include	<math.h>

#include	"base.h"
#include	"base.hpp"
#include	"base_ct.h"
#include	"log.h"
#include	"memory.h"
#include	"ReconstructionFilter.h"
#include	"splhandl.h"



/*\\\ CONSTANTES PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ CONSTANTES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#define	ReconstructionFilter_MAX_RESAMPLE_RATE	256



/*\\\ TYPES & STRUCTURES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ DEFINITION DES VARIABLES DE CLASSE PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ DEFINITION DES VARIABLES DE CLASSE PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ METHODES PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*==========================================================================*/
/*      Nom: (constructeur)                                                 */
/*      Description: Initialise                                             */
/*      Parametres en entree:                                               */
/*        - width: largeur du sinus cardinal utilise par le filtre (en      */
/*                 samples)                                                 */
/*==========================================================================*/

ReconstructionFilter::ReconstructionFilter (int width)
{
	_buffer_ptr = NULL;

	if (width <= 0)
	{
		LOG_printf ("ReconstructionFilter::ReconstructionFilter: Error: width parameter (%d) must be > 0.\n",
		            width);
		return;
	}
	_width = width;

	_buf_len = 1;
	while (_buf_len < _width + ReconstructionFilter_MAX_RESAMPLE_RATE)
	{
		_buf_len <<= 1;
	}

	_buffer_ptr = (double (*)[2]) MALLOC (_buf_len * sizeof (**_buffer_ptr) * 2);
	if (_buffer_ptr == NULL)
	{
		LOG_printf ("ReconstructionFilter::ReconstructionFilter: Error: couldn't allocate memory for buffer.\n");
		return;
	}

	clear_buffer ();
}



/*==========================================================================*/
/*      Nom: (destructeur)                                                  */
/*      Description: Detruit                                                */
/*==========================================================================*/

ReconstructionFilter::~ReconstructionFilter ()
{
	if (_buffer_ptr != NULL)
	{
		FREE (_buffer_ptr);
		_buffer_ptr = NULL;
	}
}



/*==========================================================================*/
/*      Nom: check_ok                                                       */
/*      Description: Verifie l'intergrite de l'objet.                       */
/*      Retour: 0 si l'objet parait correct.                                */
/*==========================================================================*/

signed int	ReconstructionFilter::check_ok (void) const
{
	if (_buffer_ptr == NULL)
	{
		LOG_printf ("ReconstructionFilter::check_ok: Error: _buffer_ptr is NULL.\n");
		return (-1);
	}

	MCHECK (_buffer_ptr);

	if (   _buffer_pos >= _buf_len
	    || _buffer_pos < 0)
	{
		LOG_printf ("ReconstructionFilter::check_ok: Error: _buffer_pos (%d) must be between 0 and %d.\n",
		            _buffer_pos, _buf_len - 1);
		return (-1);
	}

	return (0);
}



/*==========================================================================*/
/*      Nom: self_display                                                   */
/*      Description: Affiche "textuellement" le contenu courant de l'objet. */
/*==========================================================================*/

void	ReconstructionFilter::self_display (void) const
{

	/*** A faire ***/

}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	ReconstructionFilter::resample_and_mix (const void *s_buffer_ptr, SLWORD *d_buffer_ptr,
		                                        LWORD length, int s_stereo, int d_stereo,
		                                        SLWORD left_volume, SLWORD right_volume,
		                                        bool erase_flag, SPLH_BrokenCurveState broken_curve_state, ULWORD pos_frac,
		                                        double inv_resample_factor, int nbr_s_bits)
{
	double	sinc_step;
	double	sinc_pos;
	double	channel_amp [2];
	double	channel_amp_dif;
	double	multiplier;
	double	val [2];
	const int	s_inc [1+4] = { 0, 1, 2, 4, 4 };
	int		chn_cnt;
	int		nbr_s_bytes;
	int		buf_len;
	int		buf_pos;
	int		buf_cnt;
	double	sinc_val;
	double	new_val;
	double	d_pos;
	double	old_d_pos;
	double	temp_dbl;
	int		nbr_spl;
	int		sample_cnt;
	int		buf_mask;
	double	resample_factor;
	double	window_scale;
	double	window_pos;
	SLWORD	volume [2];
	SLWORD	first_new_value [2];
	double	offset [2];
	double	offset_inc [2];

	buf_mask = _buf_len - 1;
	resample_factor = 1.0 / inv_resample_factor;
	nbr_s_bytes = (nbr_s_bits + 7) >> 3;
	multiplier = (double)(1 << (24 - nbr_s_bits)) / 0x8000L;
	if (pos_frac == 0)
	{
		d_pos = 0;
	}
	else
	{
		d_pos = 1.0 - (double)pos_frac / 4294967296.0;
		d_pos *= resample_factor;
		s_buffer_ptr = (const void *)((const BYTE *)s_buffer_ptr + s_inc [nbr_s_bytes] * s_stereo);
	}

	if (resample_factor > 1)
	{
		sinc_step =	PI * inv_resample_factor;
		channel_amp [0] = multiplier;
	}
	else
	{
		sinc_step = PI;
		channel_amp [0] = resample_factor * multiplier;
	}
	channel_amp [1] = channel_amp [0];
	channel_amp [0] *= left_volume;
	channel_amp [1] *= right_volume;
	channel_amp_dif = channel_amp [1] / channel_amp [0];

	window_scale = PI / ((_width >> 1) * sinc_step);

	/* Cherche la premiere valeur des nouvelles donnees */
	if (s_stereo == 1)
	{
		if (nbr_s_bytes == 2)
		{
			first_new_value [0] = (SLWORD) (*((const SWORD *)s_buffer_ptr)) << 8;
		}
		else if (nbr_s_bytes == 1)
		{
			first_new_value [0] = (SLWORD) (*((const SBYTE *)s_buffer_ptr)) << 16;
		}
		else
		{
			first_new_value [0] = *((const SLWORD *)s_buffer_ptr);
		}
		first_new_value [1] = first_new_value [0];
	}
	else
	{
		if (nbr_s_bytes == 2)
		{
			first_new_value [0] = (SLWORD) ((const SWORD *)s_buffer_ptr) [0] << 8;
			first_new_value [1] = (SLWORD) ((const SWORD *)s_buffer_ptr) [1] << 8;
		}
		else if (nbr_s_bytes == 1)
		{
			first_new_value [0] = (SLWORD) ((const SBYTE *)s_buffer_ptr) [0] << 16;
			first_new_value [1] = (SLWORD) ((const SBYTE *)s_buffer_ptr) [1] << 16;
		}
		else
		{
			first_new_value [0] = ((const SLWORD *)s_buffer_ptr) [0];
			first_new_value [1] = ((const SLWORD *)s_buffer_ptr) [1];
		}
	}

/*______________________________________________
 *
 * Decliqueur
 *______________________________________________
 */

	volume [0] = left_volume;
	volume [1] = right_volume;

	for (chn_cnt = 0; chn_cnt < 2; chn_cnt ++)
	{
		if (broken_curve_state == SPLH_BrokenCurveState_MID)
		{
			offset [chn_cnt] = (  (double) _old_volume [chn_cnt] * _old_value [chn_cnt]
			                    - (double) volume [chn_cnt] * first_new_value [chn_cnt]) / 0x8000L;
		}
		else if (broken_curve_state == SPLH_BrokenCurveState_START)
		{
			offset [chn_cnt] = ((double) _old_volume [chn_cnt] * _old_value [chn_cnt]) / 0x8000L;
		}
		else
		{
			offset [chn_cnt] =   (double) (_old_volume [chn_cnt] - volume [chn_cnt])
			                   * _old_value [chn_cnt] / 0x8000L;
		}
		offset_inc [chn_cnt] = offset [chn_cnt] * resample_factor / length;
		_old_volume [chn_cnt] = volume [chn_cnt];
	}

/*______________________________________________
 *
 * Reechantillonnage
 *______________________________________________
 */

	old_d_pos = 0;
	for ( ; ; )
	{
		/* Ecrit les samples destination (les nouveaux de la boucle precedente,
		   ou les vieux de la frame d'avant qui n'avaient pas encore ete ecrits
		   parce qu'on n'en voulait pas. */
		nbr_spl = (int)(floor (d_pos) - floor (old_d_pos));
		for (sample_cnt = nbr_spl; sample_cnt > 0; sample_cnt --)
		{
			for (chn_cnt = 0; chn_cnt < d_stereo; chn_cnt ++)
			{
				if (erase_flag)
				{
					*d_buffer_ptr = (SLWORD) (_buffer_ptr [_buffer_pos] [chn_cnt]);
				}
				else
				{
					*d_buffer_ptr += (SLWORD) (_buffer_ptr [_buffer_pos] [chn_cnt]);
				}
				d_buffer_ptr ++;
			}

			/* Efface du buffer le sample qu'on vient de prendre car on n'en a
			   plus besoin. Cette case memoire du buffer sera utilisee par le
			   sample suivant donc doit etre intialisee a 0 maintenant. */
			for (chn_cnt = 0; chn_cnt < 2; chn_cnt ++)
			{
				_buffer_ptr [_buffer_pos] [chn_cnt] = 0;
			}

			/* Sample destination suivant */
			_buffer_pos = (_buffer_pos + 1) & buf_mask;
			length --;
			if (length <= 0)
			{
				_last_sample_trail = d_pos + (sample_cnt - 1 - (int)floor (old_d_pos) - nbr_spl);
				_last_sample_width = resample_factor;
				return;
			}
		}

		/* Preleve le(s) sample(s) source(s) */
		for (chn_cnt = 0; chn_cnt < s_stereo; chn_cnt ++)
		{
			if (nbr_s_bytes == 2)
			{
				_old_value [chn_cnt] = (SLWORD) (*((const SWORD *)s_buffer_ptr)) << 8;
				val [chn_cnt] = *((const SWORD *)s_buffer_ptr) * channel_amp [chn_cnt];
			}
			else if (nbr_s_bytes == 1)
			{
				_old_value [chn_cnt] = (SLWORD) (*((const SBYTE *)s_buffer_ptr)) << 16;
				val [chn_cnt] = *((const SBYTE *)s_buffer_ptr) * channel_amp [chn_cnt];
			}
			else
			{
				_old_value [chn_cnt] = *((const SLWORD *)s_buffer_ptr);
				val [chn_cnt] = *((const SLWORD *)s_buffer_ptr) * channel_amp [chn_cnt];
			}
			s_buffer_ptr = (const void *)((const BYTE *)s_buffer_ptr + s_inc [nbr_s_bytes]);
		}

		/* En cas de source mono, on repartit le sample sur les autres canaux */
		if (s_stereo == 1)
		{
			_old_value [1] = _old_value [0];
			val [1] = val [0] * channel_amp_dif;
		}

		/* Ajout des offsets destines au click removal */
		val [0] += offset [0];
		val [1] += offset [1];
		offset [0] -= offset_inc [0];
		offset [1] -= offset_inc [1];

		/* En cas de changement de sample (ou de position) par rapport au tick
			precedent, il faudra attenuer les premiers samples pour eviter un
			effet de superposition indesirable. */
		if (   broken_curve_state != SPLH_BrokenCurveState_NONE
		    && _last_sample_trail > 0)
		{
			double	attenuation;

			attenuation = (_last_sample_width - _last_sample_trail) / _last_sample_width;
			val [0] *= attenuation;
			val [1] *= attenuation;
			_last_sample_trail -= resample_factor;
		}

		/* En cas de destination mono, on mixe tous les canaux sur le premier */
		if (d_stereo == 1 && s_stereo > 1)
		{
			val [0] += val [1];
		}

		/* Reconstruction */
		sinc_pos =   (modf (d_pos, &temp_dbl) + (_width >> 1))
		           * (-sinc_step);
		buf_pos = _buffer_pos;
		buf_len = _width;
		do
		{
			buf_cnt = buf_len;
			if (buf_cnt + buf_pos >= _buf_len)
			{
				buf_cnt = _buf_len - buf_pos;
			}
			buf_len -= buf_cnt;
			do
			{
				/* Calcul de la valeur de la fonction du filtre (qui en est la
				   reponse impulsionnelle). Comme le filtre est un passe-bas, sa
				   reponse est un sinc () sur lequel on va appliquer une fenetre
				   pour le tronquer. */
				if (BASE_is_null (sinc_pos))
				{
					sinc_val = 1;
				}
				else
				{
					/* Sinus cardinal et fenetre de Blackman */
					window_pos = sinc_pos * window_scale;
					sinc_val =   (sin (sinc_pos) / sinc_pos)
					           * (  0.42
					              + 0.5 * cos (window_pos)
					              + 0.08 * cos (window_pos * 2));
				}

				for (chn_cnt = 0; chn_cnt < d_stereo; chn_cnt ++)
				{
					new_val = val [chn_cnt] * sinc_val;
					_buffer_ptr [buf_pos] [chn_cnt] += new_val;
					if (d_stereo == 1)
					{
						_buffer_ptr [buf_pos] [1] += new_val;
					}
				}

				sinc_pos += sinc_step;
				buf_pos ++;
				buf_cnt --;
			}
			while (buf_cnt > 0);
			buf_pos &= buf_mask;
		}
		while (buf_len > 0);

		/* Sample(s) destination suivant(s) */
		old_d_pos = d_pos;
		d_pos += resample_factor;
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	ReconstructionFilter::clear_buffer (void)
{
	if (_buffer_ptr != NULL)
	{
		memset (_buffer_ptr, 0, _buf_len * sizeof (**_buffer_ptr) * 2);
	}

	_buffer_pos = 0;
	_old_volume [0] = 0;
	_old_volume [1] = 0;
	_old_value [0] = 0;
	_old_value [1] = 0;
	_last_sample_trail = 0;
	_last_sample_width = 1;
}



/*\\\ METHODES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/
