/*****************************************************************************

GRAOUMF TRACKER 2

Copyright (c) 1996 - 2002 Laurent de Soras

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author : laurent@ohmforce.com
More information about this license : http://www.gnu.org/licenses/gpl.html

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include	<assert.h>
#include	<stdlib.h>

#include	"base.h"
#include	"base_ct.h"
#include	"d2d.h"
#include	"gt_limit.h"
#include	"gtracker.h"
#include	"log.h"
#include	"memory.h"
#include	"Mutex.h"
#include	"Pattern.h"
#include	"player.h"
#include	"Thread.h"
#include	"WaveForm.h"



/*\\\ CONSTANTES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#define	D2D_DEFAULT_D2D_CACHE_SIZE	(512*1024L)	/* Taille par defaut du cache en octets */



/*\\\ TYPES & STRUCTURES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PROTOTYPES DES FONCTIONS PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

void	D2D_cache_manager (void);

void	D2D_update_d2d_cache_cell (int track);
void	D2D_reserve_d2d_cache_cell (int track);



/*\\\ VARIABLES EXTERNES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/* Pointeur sur le handle du thread du gestionaire du cache. */
Thread	*D2D_thread_handle_ptr = NULL;



/*\\\ VARIABLES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/* Descripteur du cache des buffers de D2D */
D2D_BUFFER_ENTRY	D2D_buffer_descriptor [GTK_NBRVOICES_MAXI];

/* true indique au gestionnaire de cache qu'il doit quitter. Quand l'ordre a
   ete recu, le flag revient a false. */
volatile bool	D2D_cache_manager_quit_flag;



/*\\\ FONCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*==========================================================================*/
/*      Nom: D2D_init                                                       */
/*      Description: Initialise la gestion du cache du Direct-2-Disk.       */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	D2D_init (void)
{
	int		i;

	/* Initialise le tableau des descripteurs de cache de D2D */
	for (i = 0; i < GTK_NBRVOICES_MAXI; i ++)
	{
		D2D_buffer_descriptor [i].buffer_ptr [0] = NULL;
	}

	/* Installe le thread de gestion du cache */
	D2D_cache_manager_quit_flag = false;
	D2D_thread_handle_ptr = new Thread (D2D_cache_manager);
	if (D2D_thread_handle_ptr->spawn ())
	{
		LOG_printf ("D2D_init: Error: couldn't spawn a thread for D2D management.\n");
		return (-1);
	}

	return (0);
}



void	D2D_restore (void)
{
	int		i;

	if (D2D_thread_handle_ptr != NULL)
	{
		D2D_cache_manager_quit_flag = true;
		delete D2D_thread_handle_ptr;
		D2D_thread_handle_ptr = NULL;
	}

	/* Restore le tableau des descripteurs de cache de D2D */
	for (i = 0; i < GTK_NBRVOICES_MAXI; i ++)
	{
		if (D2D_buffer_descriptor [i].buffer_ptr [0] != NULL)
		{
			FREE (D2D_buffer_descriptor [i].buffer_ptr [0]);
			D2D_buffer_descriptor [i].buffer_ptr [0] = NULL;
		}
	}
}



/*==========================================================================*/
/*      Nom: D2D_get_buffer_info                                            */
/*      Description: Retourne les informations de D2D concernant une piste. */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste de sample concernee.                  */
/*      Retour: la structure d'informations sur les buffers.                */
/*==========================================================================*/

D2D_BUFFER_ENTRY	D2D_get_buffer_info (int track)
{
	return (D2D_buffer_descriptor [track]);
}



/*==========================================================================*/
/*      Nom: D2D_cache_manager                                              */
/*      Description: Thread qui s'occupe de surveiller les buffers dans le  */
/*                   cache et de les recharger si necessaire.               */
/*==========================================================================*/

void	D2D_cache_manager (void)
{
	for ( ; ; )
	{
		/* Gestion du cache */
		D2D_cache_manager_routine ();

		/* Regarde si on doit sortir */
		if (D2D_cache_manager_quit_flag)
		{
			break;
		}

		/* Met le thread en pause pendant quelques temps */

		/*** A faire: calculer un temps raisonnable ***/

		Thread::sleep (50);
	}

	D2D_cache_manager_quit_flag = false;
}



/*==========================================================================*/
/*      Nom: D2D_cache_manager_routine                                      */
/*      Description: Routine qui s'occupe de surveiller les buffers dans le */
/*                   cache et de les recharger si necessaire.               */
/*      Retour: 0 si tout s'est bien passe, ou -1 si une erreur a eu lieu   */
/*              (ces erreurs ne doivent pas etre prises en compte si le     */
/*              player fonctionne en temps reel).                           */
/*==========================================================================*/

signed int	D2D_cache_manager_routine (void)
{
	int		buffer_cnt;
	int		track;
	int		sample_mul;
	signed int	ret_val;
	BYTE		*buffer_ptr;
	SLWORD	file_pos;
	SLWORD	buffer_len;
	FILE		*file_ptr;
	D2D_BUFFER_ENTRY	*desc_ptr;
	PLAY_TRACK_INFO	*track_info_ptr;

	ret_val = 0;

	/* Signale qu'on met a jour les descripteurs et qu'il ne faut pas
		modifier les donnees des samples en ce moment. */
	GTK_mutex.wait ();

	/* Met a jour toutes les pistes */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_SPL]; track ++)
	{
		D2D_update_d2d_cache_cell (track);
	}

	/* C'est bon maintenant */
	GTK_mutex.signal ();

	/* Scanne toutes les entrees du cache */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_SPL]; track ++)
	{
		desc_ptr = D2D_buffer_descriptor + track;
		if (desc_ptr->buffer_ptr [0] == NULL)
		{
			continue;
		}

		/* Si les donnees ne correspondent pas a l'entree, on met a jour. */
		for (buffer_cnt = 0; buffer_cnt < 2; buffer_cnt ++)
		{
			if (! desc_ptr->buf_flag [buffer_cnt])
			{
				track_info_ptr =   PLAY_track_info
									  + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

				sample_mul =   track_info_ptr->mix.spl.resol
					          * track_info_ptr->mix.spl.tracks;
				file_pos =   desc_ptr->buf_pos [buffer_cnt];
				buffer_ptr = (BYTE *) desc_ptr->buffer_ptr [buffer_cnt];
				buffer_len = desc_ptr->buf_length [buffer_cnt];
				buffer_len = MIN (  buffer_len,
					                 track_info_ptr->mix.spl.reppos
					               + track_info_ptr->mix.spl.replen
					               - desc_ptr->buf_pos [buffer_cnt]);
				if (desc_ptr->buf_pos [buffer_cnt] < 0)
				{
					file_pos -= desc_ptr->buf_pos [buffer_cnt];
					buffer_len += desc_ptr->buf_pos [buffer_cnt];
				}
				file_pos *= sample_mul;
				file_pos += track_info_ptr->mix.spl.file_data_offset;
				buffer_len = MAX (buffer_len, 0);

				file_ptr = track_info_ptr->mix.spl.file_ptr;

				/* On se positionne dans le fichier */
				if (fseek (file_ptr, file_pos, SEEK_SET))
				{
					LOG_printf ("Warning: fseek() failed on D2D sample (track %d).\n", track);
					ret_val = -1;
				}

				else
				{
					/* On charge le buffer */
					if ((SLWORD) fread (buffer_ptr, sample_mul, buffer_len, file_ptr) != buffer_len)
					{
						LOG_printf ("Warning: couldn't load sample for D2D (track %d).\n", track);
						LOG_printf ("buffer_ptr = %08X, mul = %d, len = %08X, file_ptr = %08X\n",
							      (int)buffer_ptr, (int)sample_mul, (int)buffer_len, (int)file_ptr);
						ret_val = -1;
					}

					else
					{
						desc_ptr->buf_flag [buffer_cnt] = true;
					}
				}
			}
		}
	}

	return (ret_val);
}



/*==========================================================================*/
/*      Nom: D2D_update_d2d_cache_cell                                      */
/*      Description: Mets a jour les positions des buffers dans le sample   */
/*                   en fonction de l'endroit de replay. Tient compte des   */
/*                   autres pistes qui partagent le buffer, et en cree un   */
/*                   nouveau si la desynchronisation est trop importante.   */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste que l'on veut mettre a jour.          */
/*==========================================================================*/

void	D2D_update_d2d_cache_cell (int track)
{
	int		direction;
	int		buffer_cnt;
	int		buffer_to_keep_flags;
	int		first_time_offset;
	signed int	stop_direction;
	signed int	free_buffer;
	signed long	new_pos;
	long		current_pos;
	long		sample_length;
	long		buf_start [3];
	long		buf_end [3];
	long		stop_position;
	PLAY_TRACK_INFO	*track_info_ptr;
	D2D_BUFFER_ENTRY	*desc_ptr;

	track_info_ptr =   PLAY_track_info
						  + PLAY_track_info_list [Pattern_TYPE_SPL] [track];
	desc_ptr = D2D_buffer_descriptor + track;

/*______________________________________________
 *
 * Regarde s'il y a eu des changements de
 * sample ou d'etat de D2D
 *______________________________________________
 */

	/* La piste n'est pas (ou plus) une piste de D2D */
	if (! track_info_ptr->mix.spl.d2d_flag)
	{
		if (desc_ptr->buffer_ptr [0] != NULL)
		{
			D2D_free_d2d_cache_cell (track);
		}
		return;
	}

	/* La piste est en D2D, regarde s'il y a un descripteur */
	if (desc_ptr->buffer_ptr [0] != NULL)
	{
		/* Regarde si on a change de sample */
		if (track_info_ptr->mix.spl.file_ptr != desc_ptr->file_ptr)
		{
			D2D_free_d2d_cache_cell (track);
		}
	}

	/* Y a pas de desc, on en cree un */
	if (desc_ptr->buffer_ptr [0] == NULL)
	{
		D2D_reserve_d2d_cache_cell (track);

		/* La creation n'a pas marche (plus de place),
		   on attendra le prochain coup pour reessayer. */
		if (desc_ptr->buffer_ptr [0] == NULL)
		{
			return;
		}
	}

/*______________________________________________
 *
 * Met a jour les informations
 *______________________________________________
 */

	track_info_ptr =   PLAY_track_info
						  + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

	current_pos = track_info_ptr->mix.spl.curpos_int;
	direction = track_info_ptr->mix.spl.direction;
	sample_length = track_info_ptr->mix.spl.reppos + track_info_ptr->mix.spl.replen;
	if (sample_length == 0)
	{
		return;
	}

	/* Le sample est fini ? */
	if (   current_pos >= sample_length
		 && track_info_ptr->mix.spl.loopmode != WaveForm_LOOP_TYPE_PP)
	{
		return;
	}

	buf_start [0] = 0;
	buf_end   [0] = track_info_ptr->mix.spl.startbuf_len;
	buf_start [1] = desc_ptr->buf_pos [0];
	buf_end   [1] = desc_ptr->buf_pos [0] + desc_ptr->buf_length [0];
	buf_start [2] = desc_ptr->buf_pos [1];
	buf_end   [2] = desc_ptr->buf_pos [1] + desc_ptr->buf_length [1];
	buffer_to_keep_flags = 1 << 0;

	/* Trouve la position sur laquelle on arretera le scan. Apres avoir franchi
	   cette position, on saura que toutes les donnees utiles sont a present en
		memoire. */

	/* -1: on tient compte uniquement de la position de stop pour s'arreter. */
	stop_direction = -1;

	/* Pas de bouclage */
	if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_NONE)
	{
		/* Vers l'avant */
		if (direction == 0)
		{
			/* On s'arrete a la fin du sample */
			stop_position = sample_length;
		}

		/* Vers l'arriere */
		else
		{
			/* On s'arrete quand on a atteint le debut du sample */
			stop_position = 0;
		}
	}

	/* Bouclage */
	else
	{
		/* Le point courant est dans la boucle */
		if (current_pos >= track_info_ptr->mix.spl.reppos)
		{
			/* Boucle normale et on va en arriere */
			if (   direction == 1
			    && track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_FWD)
			{
				/* On s'arrete au debut du sample */
				stop_position = 0;
			}

			/* Sinon, on va vers l'avant ou en est en ping-pong loop */
			else
			{
				/* On s'arrete quand on est revenu au meme endroit,
				   dans la meme direction (pour le ping-pong loop). */
				stop_position = current_pos;
				stop_direction = direction;
			}
		}

		/* Le point courant est avant la boucle */
		else
		{
			/* A l'endroit: on s'arrete a la fin du sample */
			if (direction == 0)
			{
				stop_position = sample_length;
			}

			/* On va a l'envers: on s'arrete au debut */
			else
			{
				stop_position = 0;
			}
		}
	}

	/* Scanne le sample pour trouver si des parties utiles sont a charger en
	   memoire. On s'arrete quand tous les buffers sont remplis ou quand on a
	   atteint le point d'arret. */

	first_time_offset = 1;
	for ( ; ; )
	{
		free_buffer = -1;

		/* On repertorie tous les buffers qui sont deja en memoire
		   et qui nous seront utiles. */
		for (buffer_cnt = 0; buffer_cnt < 3; buffer_cnt ++)
		{
			/* On n'est pas dans ce buffer */
			if (   current_pos < buf_start [buffer_cnt]
				 || current_pos >= buf_end [buffer_cnt])
			{
				if ((buffer_to_keep_flags & (1 << buffer_cnt)) == 0)
				{
					/* Signale que ce buffer est libre, pour la partie dans
					   laquelle on devra charger un buffer. */
					free_buffer = buffer_cnt;
				}
				continue;
			}

			/* On essaie d'atteindre la fin du buffer */

			/* On va vers l'avant */
			if (direction == 0)
			{
				new_pos = buf_end [buffer_cnt];
				buffer_to_keep_flags |= 1 << buffer_cnt;

				/* On a depasse le point d'arret: on se tire */
				if (   stop_position >= current_pos + first_time_offset
				    && stop_position <= new_pos
				    && (   stop_direction == direction
				        || stop_direction == -1))
				{
					buffer_to_keep_flags = 0x07;
					break;
				}

				current_pos = new_pos;
				buffer_cnt = -1;	/* On recommencera a scanner tous les buffers */

				/* Si on depasse le sample */
				if (current_pos >= sample_length)
				{
					if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_FWD)
					{
						current_pos = track_info_ptr->mix.spl.reppos;
					}

					/* Ping-pong loop */
					else if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP)
					{
						direction = 1;
						current_pos = sample_length;
					}

					/* On sort du sample: ce cas ne devrait pas arriver */
					else
					{
						assert (false);
					}
				}
			}

			/* Vers l'arriere */
			else
			{
				new_pos = buf_start [buffer_cnt] - 1;
				buffer_to_keep_flags |= 1 << buffer_cnt;

				/* On a depasse le point d'arret: on se tire */
				if (   stop_position <= current_pos - first_time_offset
				    && stop_position >= new_pos
				    && (   stop_direction == direction
				        || stop_direction == -1))
				{
					buffer_to_keep_flags = 0x07;
					break;
				}

				current_pos = new_pos;
				buffer_cnt = -1;	/* On recommencera a scanner tous les buffers */

				/* On sort de la boucle ping-pong */
				if (   current_pos <= track_info_ptr->mix.spl.reppos
				    && track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP)
				{
					direction = 0;
					current_pos = track_info_ptr->mix.spl.reppos;
				}

				/* On sort du sample: ce cas ne devrait pas arriver */
				else if (current_pos <= 0)
				{
					assert (false);
				}
			}

			first_time_offset = 0;
		}

		/* On a fini de tout charger */
		if (buffer_to_keep_flags == 0x07 || free_buffer < 0)
		{
			break;
		}

		/* Remplit le buffer */
		if (direction == 0)
		{
			desc_ptr->buf_pos [free_buffer - 1] = current_pos;
			buf_start [free_buffer] = current_pos;
			buf_end [free_buffer] = current_pos + desc_ptr->buf_length [free_buffer - 1];
		}
		
		else
		{
			desc_ptr->buf_pos [free_buffer - 1] =   current_pos + 1
			                                      - desc_ptr->buf_length [free_buffer - 1];
			buf_start [free_buffer] = desc_ptr->buf_pos [free_buffer - 1];
			buf_end [free_buffer] = current_pos + 1;
		}

		desc_ptr->buf_flag [free_buffer - 1] = false;
		buffer_to_keep_flags |= 1 << free_buffer;
	}
}



/*==========================================================================*/
/*      Nom: D2D_reserve_d2d_cache_cell                                     */
/*      Description: Cherche un emplacement de cache pour le sample D2D     */
/*                   d'une voie donnee. L'emplacement occupe par le sample  */
/*                   precedent est libere, et un autre est recherche. S'il  */
/*                   y en a un qui convient dans le cache, il est utilise,  */
/*                   sinon un nouveau est cree. Si aucun emplacement n'est  */
/*                   libre, rien n'est fait.                                */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste qui doit s'offir une place dans le    */
/*                 cache.                                                   */
/*==========================================================================*/

void	D2D_reserve_d2d_cache_cell (int track)
{
	LWORD		resol;
	LWORD		buffer_size;
	void		*buffer_ptr;
	PLAY_TRACK_INFO	*track_info_ptr;
	D2D_BUFFER_ENTRY	*desc_ptr;

	/* Libere le buffer du sample precedent */
	if (D2D_buffer_descriptor [track].buffer_ptr [0] != NULL)
	{
		D2D_free_d2d_cache_cell (track);
	}

	/* Calcule la taille necessaire */
	track_info_ptr =   PLAY_track_info
						  + PLAY_track_info_list [Pattern_TYPE_SPL] [track];
	resol = track_info_ptr->mix.spl.resol * track_info_ptr->mix.spl.tracks;
	buffer_size = (  track_info_ptr->mix.spl.midbuf1_len
						+ track_info_ptr->mix.spl.midbuf2_len) * resol;

	buffer_ptr = MALLOC (buffer_size);

	/* Si on n'a pas reussi a trouver de la place, on s'en va sans rien dire
	   de plus */
	if (buffer_ptr == NULL)
	{
		return;
	}

	/* Construit le nouveau descripteur */
	desc_ptr = D2D_buffer_descriptor + track;
	desc_ptr->file_ptr       = track_info_ptr->mix.spl.file_ptr;
	desc_ptr->buffer_ptr [0] = buffer_ptr;
	desc_ptr->buffer_ptr [1] =   (BYTE *)buffer_ptr
										+ track_info_ptr->mix.spl.midbuf1_len * resol;
	desc_ptr->buf_length [0] = track_info_ptr->mix.spl.midbuf1_len;
	desc_ptr->buf_length [1] = track_info_ptr->mix.spl.midbuf2_len;
	desc_ptr->total_length   = buffer_size;
	desc_ptr->buf_pos [0]    = track_info_ptr->mix.spl.startbuf_len;
	desc_ptr->buf_pos [1]    = desc_ptr->buf_pos [0] + desc_ptr->buf_length [0];
	desc_ptr->buf_flag [0]   = false;
	desc_ptr->buf_flag [1]   = false;
}



/*==========================================================================*/
/*      Nom: D2D_free_d2d_cache_cell                                        */
/*      Description: Libere une cellule du cache qui etait reservee a la    */
/*                   piste parametre.                                       */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste dont le sample doit gicler du cache.  */
/*==========================================================================*/

void	D2D_free_d2d_cache_cell (int track)
{
	D2D_BUFFER_ENTRY	*desc_ptr;

	desc_ptr = D2D_buffer_descriptor + track;

	if (desc_ptr->buffer_ptr [0] != NULL)
	{
		FREE (desc_ptr->buffer_ptr [0]);
		desc_ptr->buffer_ptr [0] = NULL;
	}
}



/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/
