/*****************************************************************************

GRAOUMF TRACKER 2

Copyright (c) 1996 - 2002 Laurent de Soras

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author : laurent@ohmforce.com
More information about this license : http://www.gnu.org/licenses/gpl.html

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include <stdio.h>
#include	<stdlib.h>
#include <math.h>

#include "archi.h"
#include "base.h"
#include "base_ct.h"
#include "d2d.h"
#include	"filter.h"
#include "gt_limit.h"
#include "gtracker.h"
#include	"log.h"
#include "memory.h"
#include "mixer.h"
#include "mix_fx.h"
#include "os.h"
#include "player.h"
#include "splhandl.h"
#include	"WaveForm.h"



/*\\\ CONSTANTES ET MACROS PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/* Delay necessaire entre le mixage et le replay effectif (ms). On peut
prendre 100 ms pour Win 95/98 et 500 ms Win NT 4.0 */
#define	MIX_INIT_LATENCY_TIME_MS			500

/* Frequence relative maximum pour le buffer lineaire de sample. Au dela, le
sample n'est pas mixe car le buffer lineaire prendrait trop de place. */
#define	MIX_LINEAR_BUFFER_MAX_FREQ_RATIO	4

/*
	Macros d'inc/decrementation du pointeur source lors du reechantillonnage
	d'un sample 8 ou 16 bits. Parametres:
	SWORD/SBYTE		*s_buffer_ptr;
	ULWORD			pos_fraq;
	const int		freq_int;	Doit etre multiplie auparavant par stereo*(bits/8)
	const ULWORD	freq_fraq;
	const int		stereo;
*/

#define	MIX_STEP_8B(s_buffer_ptr,pos_frac,freq_int,freq_frac,stereo) \
	pos_frac -= freq_frac; 			\
	if ((ULWORD)(-(SLWORD)pos_frac) < freq_frac)	\
	{										\
		s_buffer_ptr -= stereo;		\
	}										\
	s_buffer_ptr -= freq_int;

#define	MIX_STEP_8F(s_buffer_ptr,pos_frac,freq_int,freq_frac,stereo) \
	pos_frac += freq_frac; 			\
	if (pos_frac < freq_frac)		\
	{										\
		s_buffer_ptr += stereo;		\
	}										\
	s_buffer_ptr += freq_int;

#define	MIX_STEP_16B(s_buffer_ptr,pos_frac,freq_int,freq_frac,stereo) \
	pos_frac -= freq_frac; 			\
	if ((ULWORD)(-(SLWORD)pos_frac) < freq_frac)	\
	{										\
		s_buffer_ptr = (SWORD *) ((BYTE *)s_buffer_ptr - stereo*2);	\
	}										\
	s_buffer_ptr = (SWORD *) ((BYTE *)s_buffer_ptr - freq_int);

#define	MIX_STEP_16F(s_buffer_ptr,pos_frac,freq_int,freq_frac,stereo) \
	pos_frac += freq_frac;			\
	if (pos_frac < freq_frac)		\
	{										\
		s_buffer_ptr = (SWORD *) ((BYTE *)s_buffer_ptr + stereo*2);	\
	}										\
	s_buffer_ptr = (SWORD *) ((BYTE *)s_buffer_ptr + freq_int);



/*\\\ TYPES & STRUCTURES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PROTOTYPES DES FONCTIONS PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

void	MIX_explore_fx_track (int fx_track_nbr, int global_fx_track_nbr, bool track_mixed_to_aou_flag [GTK_NBRVOICES_MAXI] [GTK_NBROUT_MAXI]);

void	MIX_set_new_sample_position (int track);
void	MIX_save_click_removal_info (void);
void	MIX_cr_set_new_values (PLAY_TRACK_INFO *track_info_ptr, const SLWORD *wet_buffer_ptr, const SLWORD *dry_buffer_ptr);
void	MIX_cr_remove_clicks (PLAY_TRACK_INFO *track_info_ptr, SLWORD *buffer_ptr);

void	MIX_fill_buffer_end_24 (SLWORD *d_buffer_ptr, int d_stereo, LWORD length,
									   const PLAY_TRACK_INFO *track_info_ptr, SLWORD left_volume,
									   SLWORD right_volume, bool erase_flag, int direction);
void	MIX_fill_buffer_end_16 (SLWORD *d_buffer_ptr, int d_stereo, LWORD length,
									   const PLAY_TRACK_INFO *track_info_ptr, int direction);
void	MIX_resample_and_mix_sample (int track, SLWORD *d_buffer_ptr, int d_stereo,
											  SLWORD left_volume, SLWORD right_volume, bool erase_flag);
void	MIX_mix_track (const SLWORD *s_buffer_ptr, SLWORD *d_buffer_ptr,
		               LWORD length, int s_stereo, int d_stereo,
		               SLWORD left_volume, SLWORD right_volume, bool erase_flag);
void	MIX_resample_16f_and_mix_track (const SWORD *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
												  int s_stereo, int d_stereo, SLWORD left_volume, SLWORD right_volume,
												  bool erase_flag, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_16b_and_mix_track (const SWORD *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
												  int s_stereo, int d_stereo, SLWORD left_volume, SLWORD right_volume,
												  bool erase_flag, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_16f_and_mix_track_interpol (const SWORD *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
												           int s_stereo, int d_stereo, SLWORD left_volume, SLWORD right_volume,
												           bool erase_flag, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_16f (const SWORD *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
								int s_stereo, int d_stereo, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_16b (const SWORD *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
								int s_stereo, int d_stereo, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_8f_and_mix_track (const SBYTE *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
												 int s_stereo, int d_stereo, SLWORD left_volume, SLWORD right_volume,
												 bool erase_flag, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_8b_and_mix_track (const SBYTE *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
												 int s_stereo, int d_stereo, SLWORD left_volume, SLWORD right_volume,
												 bool erase_flag, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_8f (const SBYTE *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
							  int s_stereo, int d_stereo, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
void	MIX_resample_8b (const SBYTE *s_buffer_ptr, SLWORD *d_buffer_ptr, LWORD length,
							  int s_stereo, int d_stereo, ULWORD pos_frac, int freq_int, ULWORD freq_frac);
signed int	MIX_make_sample_linear_16 (SWORD *buffer_ptr, int track, LWORD length);
void	MIX_get_next_linear_sample_block (int track, SLWORD old_position, int old_direction,
		                                  SLWORD *position_ptr, int *direction_ptr, void **block_ptr_ptr,
		                                  SLWORD *length_ptr, int *block_dir_ptr);



/*\\\ DEFINITION DES VARIABLES EXTERNES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

LWORD	MIX_replay_freq;		/* Frequence d'echantillonnage du mixer, fixee par le driver */
LWORD	MIX_max_frame_length = MIX_MAX_FRAME_LENGTH;	/* Longueur maximum d'une frame, en samples */
LWORD	MIX_latency_time_ms = MIX_INIT_LATENCY_TIME_MS;	/* Delay necessaire entre le mixage et le replay effectif (ms) */

/* Taille de la zone des buffers intermediaires, en samples */
LWORD	MIX_buffer_zone_size = MIX_INIT_BUFFER_ZONE_SIZE;

/* Longueur utilisee pour les calculs. Fixee par le gestionnaire de partition. */
LWORD	MIX_frame_length;

/* Adresse des buffers finaux, par piste. A initialiser */
SLWORD	*MIX_driver_out_buffer_ptr [GTK_NBROUT_MAXI];
SLWORD	*MIX_driver_in_buffer_ptr [GTK_NBRIN_MAXI];

/* Largeur de la fenetre du filtre de reconstruction */
int	MIX_recons_filter_width = MIX_INIT_RECONS_FLITER_WIDTH;



/*\\\ VARIABLES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/* Tableau de mixage */
/* A initialiser */
MIX_TRACK_ORDER	MIX_track_order [GTK_NBRVOICES_MAXI * GTK_NBROUT_MAXI];
int	MIX_total_nbr_tracks = 0;		/* Nombre de pistes presentes dans le tableau de mixage */

/* Nombre de buffers intermediaires 0 = mono, 1 = stereo */
int	MIX_nbr_intermediate_buffers [2] = { 0, 0 };

/* Adresse de la zone des buffers intermediaires 24 bits (mais 4 octets/spl) */
SLWORD	*MIX_buffer_zone_ptr = NULL;

/* Adresses de chaque buffer intermediaire de mixage */
SLWORD	*MIX_int_buffer_ptr [2] [GTK_NBRVOICES_MAXI * GTK_NBROUT_MAXI];

/* Buffers de mixage finaux */
SLWORD	MIX_driver_mix_data [(GTK_NBROUT_MAXI + GTK_NBRIN_MAXI) * MIX_MAX_FRAME_LENGTH];

/* Pistes triees lors de la construction du tableau de mixage */
bool	MIX_sorted_track_flag [PLAY_NBR_GLOBAL_TRACKS];
struct
{
	WORD		first;
	WORD		last;
} MIX_use_of_track [PLAY_NBR_GLOBAL_TRACKS];

/* Buffer mono/stereo servant a stoquer le sample sous forme lineaire 16 bits
   avant reechantillonnage. La taille fait quatre fois la taille maximum d'une
	frame plus 64 samples de securite. */
SWORD	MIX_linear_sample_buffer [(MIX_MAX_FRAME_LENGTH * MIX_LINEAR_BUFFER_MAX_FREQ_RATIO + 64) * 2];

/* Type de qualite de mixage */
int	MIX_sound_quality = MIX_SOUND_QUALITY_NORMAL;



/*\\\ FONCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom: MIX_init                                                       */
/*      Description: Initialisation des routines de mixage.                 */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	MIX_init (void)
{
	LOG_printf ("Mixer init:\n");

	/* Reserve la zone pour les buffers intermediaires */
	MIX_buffer_zone_ptr = (SLWORD *) MALLOC (MIX_buffer_zone_size * sizeof (SLWORD));
	if (MIX_buffer_zone_ptr == NULL)
	{
		LOG_printf ("MIX_init: Error: couldn't reserve memory for mix buffers.\n");
		return (-1);
	}

	LOG_printf ("\tOK.\n");

	return (0);
}



void	MIX_restore (void)
{
	if (MIX_buffer_zone_ptr != NULL)
	{
		FREE (MIX_buffer_zone_ptr);
		MIX_buffer_zone_ptr = NULL;
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

int	MIX_get_sound_quality (void)
{
	return (MIX_sound_quality);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

signed int	MIX_set_sound_quality (int quality)
{
	int		track;
	int		global_track_nbr;
	signed int	ret_val;
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	ret_val = 0;

	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_SPL]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_SPL] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;

		if (quality > MIX_SOUND_QUALITY_NORMAL)
		{
			if (track_info_ptr->mix.spl.recons_filter_ptr == NULL)
			{
				track_info_ptr->mix.spl.recons_filter_ptr = new ReconstructionFilter (MIX_recons_filter_width);
				if (track_info_ptr->mix.spl.recons_filter_ptr == NULL)
				{
					LOG_printf ("MIX_set_sound_quality: Warninig: couldn't create a reconstruction filter for Sample track # %d.\n",
					            track);
					ret_val = -1;
				}
				else if (track_info_ptr->mix.spl.recons_filter_ptr->check_ok ())
				{
					LOG_printf ("MIX_set_sound_quality: Warninig: couldn't initialize reconstruction filter for Sample track # %d.\n",
					            track);
					delete track_info_ptr->mix.spl.recons_filter_ptr;
					track_info_ptr->mix.spl.recons_filter_ptr = NULL;
					ret_val = -1;
				}
			}
		}

		else
		{
			if (track_info_ptr->mix.spl.recons_filter_ptr != NULL)
			{
				delete track_info_ptr->mix.spl.recons_filter_ptr;
				track_info_ptr->mix.spl.recons_filter_ptr = NULL;
			}
		}
	}

	MIX_sound_quality = quality;

	GTK_mutex.signal ();

	return (ret_val);
}



/*==========================================================================*/
/*      Nom: MIX_create_mix_buffers                                         */
/*      Description: Repartit les buffers finaux de mixage (in et out).     */
/*==========================================================================*/

void	MIX_create_mix_buffers (void)
{
	int		track;
	int		global_track_nbr;
	PLAY_TRACK_INFO	*track_info_ptr;

	for (track = 0; track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		if (track_info_ptr->stereo == 0)
		{
			MIX_driver_out_buffer_ptr [track] = MIX_driver_out_buffer_ptr [track - 1];
		}
		else
		{
			MIX_driver_out_buffer_ptr [track] =   MIX_driver_mix_data
			                                    + MIX_max_frame_length * track;
		}
	}
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_AIN]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_AIN] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		if (track_info_ptr->stereo == 0)
		{
			MIX_driver_in_buffer_ptr [track] = MIX_driver_in_buffer_ptr [track - 1];
		}
		else
		{
			MIX_driver_in_buffer_ptr [track] =   MIX_driver_mix_data
			                                   +   MIX_max_frame_length
			                                     * (track + GTK_NBROUT_MAXI);
		}
	}
}



/*==========================================================================*/
/*      Nom: MIX_create_mix_array                                           */
/*      Description: Cree le tableau de mixage en parcourant l'arbre forme  */
/*                   par le mixage successif des pistes.                    */
/*==========================================================================*/

void	MIX_create_mix_array (void)
{
	int		stereo;
	int		mix;
	int		used_buffer_type;
	int		aou_track;
	int		track_index;
	int		track_type;
	int		track_nbr;
	int		s_global_track_nbr;
	int		d_global_track_nbr;
	int		nbr_free_intermediate_buffers [2];
	int		free_intermediate_buffers [2] [GTK_NBRVOICES_MAXI * GTK_NBROUT_MAXI];	/* Pile */
	int		*free_int_buf_ptr [2];
	int		int_buf_for_global_track [PLAY_NBR_GLOBAL_TRACKS] [2];	/* 0 = wet, 1 = dry */
	bool		track_mixed_to_aou_flag [GTK_NBRVOICES_MAXI] [GTK_NBROUT_MAXI];
	MIX_TRACK_ORDER		*track_order_ptr;
	PLAY_TRACK_INFO		*s_track_info_ptr;
	PLAY_TRACK_INFO		*d_track_info_ptr;
	PLAY_TRACK_INFO		*aou_track_info_ptr;

/*______________________________________________
 *
 * Tri des pistes
 *______________________________________________
 */

/*
Principe:
	On scanne les pistes d'effets. Si cette piste utilise des piste qui
ne sont pas encore dans le tableau, on rentre dans ces pistes avant de
les placer. On finit par tomber dans des pistes de samples et d'Audio
In, que l'on place.

Optimisation (faite):
	A chaque fois qu'une piste sert pour la premiere fois, regarder si
elle doit etre mixee dans une piste Audio Out. Dans ce cas, mettre ce
mixage juste a cote de l'endroit ou elle est utilisee. Ca permet
d'economiser les buffers, particulierement pour les pistes de samples
qui servent de source a un effet et a l'Audio In.
*/

	/*** La construction du tableau fait comme si les
		  pistes pouvaient etre gardees en memoire. ***/

	MIX_total_nbr_tracks = 0;
	memset (MIX_sorted_track_flag, 0, PLAY_NBR_GLOBAL_TRACKS * sizeof (*MIX_sorted_track_flag));
	memset (track_mixed_to_aou_flag, 0, GTK_NBRVOICES_MAXI * GTK_NBROUT_MAXI * sizeof (**track_mixed_to_aou_flag));

	/* Commence par regarder dans toutes les pistes Audio Out */
	for (aou_track = 0; aou_track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; aou_track ++)
	{
		d_global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [aou_track];
		aou_track_info_ptr = PLAY_track_info + d_global_track_nbr;

		/* Repertorie toutes les pistes sources */
		for (track_index = 0; track_index < aou_track_info_ptr->mix.src.nbr_s_tracks; track_index ++)
		{
			track_type = aou_track_info_ptr->mix.src.track_conf [track_index].track_type;
			track_nbr  = aou_track_info_ptr->mix.src.track_conf [track_index].track_nbr;
			s_global_track_nbr = PLAY_track_info_list [track_type] [track_nbr];

			/* Si la piste a deja ete mixee auparavant, inutile de s'attarder */
			if (track_mixed_to_aou_flag [track_index] [aou_track])
			{
				continue;
			}

			/* Si la piste n'a pas deja ete repertoriee */
			if (! MIX_sorted_track_flag [s_global_track_nbr])
			{
				/* Si c'est une piste d'effets, on
					va egalement chercher ses sources */
				if (track_type == Pattern_TYPE_FX)
				{
					MIX_explore_fx_track (track_nbr, s_global_track_nbr, track_mixed_to_aou_flag);
				}

				/* Sinon, on la place dans le tableau */
				else
				{
					MIX_sorted_track_flag [s_global_track_nbr] = true;
					MIX_use_of_track [s_global_track_nbr].first = MIX_total_nbr_tracks;
				}
			}

			MIX_track_order [MIX_total_nbr_tracks].s_track_type = track_type;
			MIX_track_order [MIX_total_nbr_tracks].s_track_nbr = track_nbr;
			MIX_track_order [MIX_total_nbr_tracks].s_track_index = track_index;
			MIX_track_order [MIX_total_nbr_tracks].d_track_type = GTK_TRACK_TYPE_AOU;
			MIX_track_order [MIX_total_nbr_tracks].d_track_nbr = aou_track;
			track_mixed_to_aou_flag [track_index] [aou_track] = true;

			MIX_use_of_track [s_global_track_nbr].last = MIX_total_nbr_tracks;
			MIX_total_nbr_tracks ++;
		}

		/* Si la piste courante est stereo, la piste suivante doit etre
			ignoree donc on la saute. */
		aou_track += aou_track_info_ptr->stereo - 1;
	}

/*______________________________________________
 *
 * Attribution des buffers intermediaires
 *______________________________________________
 */

	MIX_nbr_intermediate_buffers [0] = 0;
	MIX_nbr_intermediate_buffers [1] = 0;
	nbr_free_intermediate_buffers [0] = 0;
	nbr_free_intermediate_buffers [1] = 0;
	free_int_buf_ptr [0] = free_intermediate_buffers [0] + GTK_NBRVOICES_MAXI * GTK_NBROUT_MAXI;
	free_int_buf_ptr [1] = free_intermediate_buffers [1] + GTK_NBRVOICES_MAXI * GTK_NBROUT_MAXI;
	track_order_ptr = MIX_track_order;

	/* On parcourt chaque mixage */
	for (mix = 0; mix < MIX_total_nbr_tracks; mix ++)
	{
		s_global_track_nbr = PLAY_track_info_list [track_order_ptr->s_track_type]
																[track_order_ptr->s_track_nbr];
		s_track_info_ptr = PLAY_track_info + s_global_track_nbr;

		d_global_track_nbr = PLAY_track_info_list [track_order_ptr->d_track_type]
																[track_order_ptr->d_track_nbr];
		d_track_info_ptr = PLAY_track_info + d_global_track_nbr;

		/* Piste source */
		used_buffer_type = 0;
		switch (track_order_ptr->s_track_type)
		{
		case	Pattern_TYPE_SPL:
			stereo = s_track_info_ptr->stereo - 1;
			break;
		case	Pattern_TYPE_FX:
			used_buffer_type = d_track_info_ptr->mix.src.track_conf [track_order_ptr->s_track_index].wet_flag ? 0 : 1;
			if (used_buffer_type == 1)
			{
				stereo = 1;	/* Le buffer dry est toujours stereo */
			}
			else
			{
				stereo = s_track_info_ptr->stereo - 1;
			}
			break;
		}

		/* Si la piste source est une Audio In, (ou que c'est une piste
			de samples utilisee qu'une seule fois directement pour la
			sortie Audio Out), pas besoin de buffer intermediaire. */
		if (   track_order_ptr->s_track_type == Pattern_TYPE_AIN
			 /*|| (   track_order_ptr->s_track_type == Pattern_TYPE_SPL
		        && track_order_ptr->d_track_type == GTK_TRACK_TYPE_AOU
		        &&    MIX_use_of_track [s_global_track_nbr].last
		           == MIX_use_of_track [s_global_track_nbr].first)*/)
		{
			track_order_ptr->s_buf_nbr = -1;
		}

		/* Sinon la piste est utilisee plusieurs fois. */
		else
		{
			/* Si c'est la premiere fois qu'on l'utilise, on cree un buffer.
			   Ca ne peut pas etre une piste FX donc on ne fait pas de buffer Wet. */
			if (mix == MIX_use_of_track [s_global_track_nbr].first)
			{
				/* S'il y a des buffers dans la pile des buffers deja utilises */
				if (nbr_free_intermediate_buffers [stereo] > 0)
				{
					nbr_free_intermediate_buffers [stereo] --;
					int_buf_for_global_track [s_global_track_nbr] [0] = *((free_int_buf_ptr [stereo]) ++);
				}
				/* Sinon on en prend un nouveau */
				else
				{
					int_buf_for_global_track [s_global_track_nbr] [0] = MIX_nbr_intermediate_buffers [stereo];
					MIX_nbr_intermediate_buffers [stereo] ++;
				}
			}

			track_order_ptr->s_buf_nbr = int_buf_for_global_track [s_global_track_nbr] [used_buffer_type];

			/* Si c'est la derniere fois qu'on utilise ce buffer,
				il faut le liberer. */
			if (mix == MIX_use_of_track [s_global_track_nbr].last)
			{
				*(--(free_int_buf_ptr [stereo])) = int_buf_for_global_track [s_global_track_nbr] [used_buffer_type];
				nbr_free_intermediate_buffers [stereo] ++;

				/* 2 buffers pour les effets (dry et wet) */
				if (track_order_ptr->s_track_type == Pattern_TYPE_FX)
				{
					if (used_buffer_type == 0)	// Libere aussi le dry
					{
						*(--(free_int_buf_ptr [1])) = int_buf_for_global_track [s_global_track_nbr] [1];
						nbr_free_intermediate_buffers [1] ++;
					}
					else								// Libere aussi le wet
					{
						*(--(free_int_buf_ptr [s_track_info_ptr->stereo - 1])) =
							int_buf_for_global_track [s_global_track_nbr] [0];
						nbr_free_intermediate_buffers [s_track_info_ptr->stereo - 1] ++;
					}
				}
			}
		}

		/* Piste destination */
		used_buffer_type = 0;
		switch (track_order_ptr->d_track_type)
		{
		case	Pattern_TYPE_FX:
			stereo = d_track_info_ptr->stereo - 1;
			used_buffer_type = 1;
			break;
		case	GTK_TRACK_TYPE_AOU:
			stereo = d_track_info_ptr->stereo - 1;
			break;
		}

		/* Si la piste destination est une Audio Out,
		   pas besoin de buffer intermediaire. */
		if (track_order_ptr->d_track_type == GTK_TRACK_TYPE_AOU)
		{
			track_order_ptr->d_buf_nbr = -1;
		}

		/* Sinon, il faut utiliser un buffer intermediaire. */
		else
		{
			/* Si c'est la premiere fois qu'on l'utilise, il faut la creer. */
			if (mix == MIX_use_of_track [d_global_track_nbr].first)
			{
				/* S'il y a des buffers dans la pile des buffers deja utilises */
				if (nbr_free_intermediate_buffers [stereo] > 0)
				{
					nbr_free_intermediate_buffers [stereo] --;
					int_buf_for_global_track [d_global_track_nbr] [0] = *((free_int_buf_ptr [stereo]) ++);
				}
				/* Sinon on en prend un nouveau */
				else
				{
					int_buf_for_global_track [d_global_track_nbr] [0] = MIX_nbr_intermediate_buffers [stereo];
					MIX_nbr_intermediate_buffers [stereo] ++;
				}

				/* Il y a un 2eme buffer pour les effets (dry, toujours stereo) */
				if (track_order_ptr->d_track_type == Pattern_TYPE_FX)
				{
					/* S'il y a des buffers dans la pile des buffers deja utilises */
					if (nbr_free_intermediate_buffers [1] > 0)
					{
						nbr_free_intermediate_buffers [1] --;
						int_buf_for_global_track [d_global_track_nbr] [1] = *((free_int_buf_ptr [1]) ++);
					}
					/* Sinon on en prend un nouveau */
					else
					{
						int_buf_for_global_track [d_global_track_nbr] [1] = MIX_nbr_intermediate_buffers [1];
						MIX_nbr_intermediate_buffers [1] ++;
					}
				}
			}

			track_order_ptr->d_buf_nbr = int_buf_for_global_track [d_global_track_nbr] [used_buffer_type];

			/* Si la piste destination est un effet et que l'index de la piste
			   source etait le dernier, on donne les numeros de buffers dry & wet
			   afin de pouvoir appliquer l'effet de l'un a l'autre quand le temps
			   sera venu (bref, quand on en sera au meme point, mais pendant le
			   mixage). */
			if (   track_order_ptr->d_track_type == Pattern_TYPE_FX
				 && track_order_ptr->s_track_index >= d_track_info_ptr->mix.src.nbr_s_tracks - 1)
			{
				track_order_ptr->wet_buf_nbr = int_buf_for_global_track [d_global_track_nbr] [0];
				track_order_ptr->dry_buf_nbr = int_buf_for_global_track [d_global_track_nbr] [1];
			}

			/* Si c'est la derniere fois que le buffer est utilise, on le
				met sur la pile des buffers libres. Ce cas peut arriver si
				une piste d'effet par exemple n'est pas redirigee vers une
				autre piste (AOU ou FX). */
			/*** Ce cas peut-il vraiment se produire ??? ***/
			if (mix == MIX_use_of_track [d_global_track_nbr].last)
			{
				*(--(free_int_buf_ptr [stereo])) = int_buf_for_global_track [d_global_track_nbr] [0];
				nbr_free_intermediate_buffers [stereo] ++;

				/* 2 buffers pour les effets (dry et wet) */
				if (track_order_ptr->s_track_type == Pattern_TYPE_FX)
				{
					*(--(free_int_buf_ptr [1])) = int_buf_for_global_track [d_global_track_nbr] [1];
					nbr_free_intermediate_buffers [1] ++;
				}
			}
		}

		track_order_ptr ++;
	}
}



/*==========================================================================*/
/*      Nom: MIX_explore_fx_track                                           */
/*      Description: Cherche recursivement les sources d'une piste d'effets */
/*                   pour les placer dans le tableau de mixage avant la     */
/*                   piste elle-meme.                                       */
/*      Parametres en entree:                                               */
/*        - track_nbr: numero de la piste d'effets.                         */
/*        - global_track_nbr: numero de piste generalise qui correspond a   */
/*                            track_nbr.                                    */
/*==========================================================================*/

void	MIX_explore_fx_track (int fx_track_nbr, int global_fx_track_nbr, bool track_mixed_to_aou_flag [GTK_NBRVOICES_MAXI] [GTK_NBROUT_MAXI])
{
	int		track_type;
	int		track_nbr;
	int		track_type_2;
	int		track_nbr_2;
	int		track_index;
	int		track_index_2;
	int		global_track_nbr;
	int		global_track_nbr_2;
	int		aou_track;
	PLAY_TRACK_INFO	*fx_track_info_ptr;
	PLAY_TRACK_INFO	*aou_track_info_ptr;

	fx_track_info_ptr = PLAY_track_info + global_fx_track_nbr;

	/* Repertorie toutes les pistes sources */
	for (track_index = 0; track_index < fx_track_info_ptr->mix.src.nbr_s_tracks; track_index ++)
	{
		track_type = fx_track_info_ptr->mix.src.track_conf [track_index].track_type;
		track_nbr  = fx_track_info_ptr->mix.src.track_conf [track_index].track_nbr;
		global_track_nbr = PLAY_track_info_list [track_type] [track_nbr];

		/* Si la piste n'a pas deja ete repertoriee */
		if (! MIX_sorted_track_flag [global_track_nbr])
		{
			/* Si c'est une piste d'effets, on
				va egalement chercher ses sources */
			if (track_type == Pattern_TYPE_FX)
			{
				MIX_explore_fx_track (track_nbr, global_track_nbr, track_mixed_to_aou_flag);
			}

			/* Sinon, on la repertorie */
			else
			{
				MIX_sorted_track_flag [global_track_nbr] = true;
				MIX_use_of_track [global_track_nbr].first = MIX_total_nbr_tracks;
			}

			/* Si la piste source doit etre mixee a un buffer Audio Out, il
				faut la repertorier ici (si ce n'est pas encore fait). */
			for (aou_track = 0; aou_track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; aou_track ++)
			{
				global_track_nbr_2 = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [aou_track];
				aou_track_info_ptr = PLAY_track_info + global_track_nbr_2;

				for (track_index_2 = 0; track_index_2 < aou_track_info_ptr->mix.src.nbr_s_tracks; track_index_2 ++)
				{
					track_type_2 = aou_track_info_ptr->mix.src.track_conf [track_index_2].track_type;
					track_nbr_2  = aou_track_info_ptr->mix.src.track_conf [track_index_2].track_nbr;

					/* C'est la piste qu'on vient de faire ? */
					if (   track_type_2 == track_type
						 && track_nbr_2 == track_nbr)
					{
						MIX_track_order [MIX_total_nbr_tracks].s_track_type = track_type_2;
						MIX_track_order [MIX_total_nbr_tracks].s_track_nbr = track_nbr_2;
						MIX_track_order [MIX_total_nbr_tracks].s_track_index = track_index_2;
						MIX_track_order [MIX_total_nbr_tracks].d_track_type = GTK_TRACK_TYPE_AOU;
						MIX_track_order [MIX_total_nbr_tracks].d_track_nbr = aou_track;
						MIX_total_nbr_tracks ++;
						track_mixed_to_aou_flag [track_index_2] [aou_track] = true;
						break;
					}
				}

				/* Si la piste courante est stereo, la piste suivante doit etre
					ignoree donc on la saute. */
				aou_track += aou_track_info_ptr->stereo - 1;
			}
		}

		MIX_track_order [MIX_total_nbr_tracks].s_track_type = track_type;
		MIX_track_order [MIX_total_nbr_tracks].s_track_nbr = track_nbr;
		MIX_track_order [MIX_total_nbr_tracks].s_track_index = track_index;
		MIX_track_order [MIX_total_nbr_tracks].d_track_type = Pattern_TYPE_FX;
		MIX_track_order [MIX_total_nbr_tracks].d_track_nbr = fx_track_nbr;

		/* On marque la premiere utilisation de la
			piste d'effet sur la premiere track source. */
		if (track_index == 0)
		{
			MIX_sorted_track_flag [global_fx_track_nbr] = true;
			MIX_use_of_track [global_fx_track_nbr].first = MIX_total_nbr_tracks;
		}

		/* Si la piste source doit etre mixee a un buffer Audio Out, il
			faut la repertorier ici (si ce n'est pas encore fait). */
		for (aou_track = 0; aou_track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; aou_track ++)
		{
			global_track_nbr_2 = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [aou_track];
			aou_track_info_ptr = PLAY_track_info + global_track_nbr_2;

			for (track_index_2 = 0; track_index_2 < aou_track_info_ptr->mix.src.nbr_s_tracks; track_index_2 ++)
			{
				track_type_2 = aou_track_info_ptr->mix.src.track_conf [track_index_2].track_type;
				track_nbr_2  = aou_track_info_ptr->mix.src.track_conf [track_index_2].track_nbr;

				/* C'est la piste qu'on vient de faire ? */
				if (   track_type_2 == track_type
					 && track_nbr_2 == track_nbr)
				{
					/* La piste a deja ete faite dans ce buffer d'Audio Out,
						on peut passer au suivant. */
					if (track_mixed_to_aou_flag [track_index_2] [aou_track])
					{
						break;
					}

					/* La piste n'est pas encore mixee, on la met la. */
					else
					{
						MIX_total_nbr_tracks ++;
						MIX_track_order [MIX_total_nbr_tracks].s_track_type = track_type_2;
						MIX_track_order [MIX_total_nbr_tracks].s_track_nbr = track_nbr_2;
						MIX_track_order [MIX_total_nbr_tracks].s_track_index = track_index_2;
						MIX_track_order [MIX_total_nbr_tracks].d_track_type = GTK_TRACK_TYPE_AOU;
						MIX_track_order [MIX_total_nbr_tracks].d_track_nbr = aou_track;
						track_mixed_to_aou_flag [track_index_2] [aou_track] = true;
						break;
					}
				}
			}

			/* Si la piste courante est stereo, la piste suivante doit etre
				ignoree donc on la saute. */
			aou_track += aou_track_info_ptr->stereo - 1;
		}

		MIX_use_of_track [global_track_nbr].last = MIX_total_nbr_tracks;
		MIX_total_nbr_tracks ++;
	}
}



/*==========================================================================*/
/*      Nom: MIX_do_mix                                                     */
/*      Description: Routine principale de mixage. Doit etre appelee dans   */
/*                   la routine d'interruption du driver apres la routine   */
/*                   de gestion de partition.                               */
/*==========================================================================*/

void	MIX_do_mix (void)
{
	int		s_track_type;
	int		d_track_type;
	int		stereo;
	int		s_stereo;
	int		d_stereo;
	int		buffer;
	int		track_index;
	int		mix_cnt;
	int		s_global_track_nbr;
	int		d_global_track_nbr;
	int		track;
	bool		wet_flag;
	bool		erase_flag;
	bool		mute_flag;
	void		*s_buffer_ptr;
	SLWORD	*d_buffer_ptr;
	LWORD		mix_volume;
	LWORD		mix_panning;
	LWORD		left_mix_volume;
	LWORD		right_mix_volume;
	LWORD		temp;
	LWORD		temp_2;
	MIX_TRACK_ORDER	*track_order_ptr;
	PLAY_TRACK_INFO	*s_track_info_ptr;
	PLAY_TRACK_INFO	*d_track_info_ptr;

	GTK_mutex.wait ();

	/* Mise a jour du tableau de mixage des voies */
	MIX_create_mix_array ();

/*______________________________________________
 *
 * Recalcule les adresses de chaque buffer
 *______________________________________________
 */

	{
		SLWORD	*buffer_ptr = MIX_buffer_zone_ptr;
		LWORD	buffer_length = MIX_frame_length;		/* 24 bits -> 4 octets */

		for (stereo = 0; stereo < 2; stereo ++)
		{
			for (buffer = 0; buffer < MIX_nbr_intermediate_buffers [stereo]; buffer ++)
			{
				MIX_int_buffer_ptr [stereo] [buffer] = buffer_ptr;
				buffer_ptr += buffer_length;
			}
			buffer_length *= 2;		/* Passage en stereo */
		}
	}

	/* Parcourt les mixages un par un */
	track_order_ptr = MIX_track_order;
	for (mix_cnt = 0; mix_cnt < MIX_total_nbr_tracks; mix_cnt ++)
	{

/*______________________________________________
 *
 * Preparation au mixage
 *______________________________________________
 */

		track_index = track_order_ptr->s_track_index;
		erase_flag = (track_index == 0);
		s_track_type = track_order_ptr->s_track_type;
		d_track_type = track_order_ptr->d_track_type;

		/* Piste source */
		s_global_track_nbr = PLAY_track_info_list [s_track_type]
																[track_order_ptr->s_track_nbr];
		s_track_info_ptr = PLAY_track_info + s_global_track_nbr;

		/* Piste destination */
		d_global_track_nbr = PLAY_track_info_list [d_track_type]
																[track_order_ptr->d_track_nbr];
		d_track_info_ptr = PLAY_track_info + d_global_track_nbr;

		/* Calcule le volume de mixage et la balance selon les types des pistes,
		   ainsi que le flag de mute. */
		mute_flag = s_track_info_ptr->wet_mute_flag;
		switch (s_track_type)
		{
		/* Sample -> */
		case	Pattern_TYPE_SPL:
			mix_volume = (LWORD)s_track_info_ptr->outvol;
			mix_panning = s_track_info_ptr->outpan;
			s_stereo = s_track_info_ptr->stereo;
			break;

		/* Audio In -> */
		case	Pattern_TYPE_AIN:
			mix_volume = s_track_info_ptr->outvol;
			mix_panning = s_track_info_ptr->outpan;
			s_stereo = s_track_info_ptr->stereo;
			break;

		/* FX -> */
		case	Pattern_TYPE_FX:
			wet_flag = d_track_info_ptr->mix.src.track_conf [track_index].wet_flag;
			s_stereo = s_track_info_ptr->stereo;
			
			/* FX (wet) -> */
			if (wet_flag)
			{
				mix_volume = s_track_info_ptr->outvol;
				mix_panning = s_track_info_ptr->outpan;
			}
			/* FX (dry) -> */
			else
			{
				mute_flag = s_track_info_ptr->dry_mute_flag;
				mix_volume = s_track_info_ptr->dry_outvol;
				mix_panning = s_track_info_ptr->dry_outpan;
				s_stereo = 2;
			}
			break;
		}

		switch (d_track_type)
		{
		/* -> FX */
		case	Pattern_TYPE_FX:
			mix_volume *= d_track_info_ptr->mix.src.track_conf [track_index].inpvol;
			mix_volume >>= 8;
			temp = d_track_info_ptr->mix.src.track_conf [track_index].inppan;
			temp_2 = temp;
			if (temp > 0x8000L)
			{
				temp = 0x10000L - temp;
			}
			mix_panning -= 0x8000L;
			mix_panning *= temp;
			mix_panning /= 0x8000L;
			mix_panning += temp_2;
			d_stereo = 2;
			break;

		/* -> Audio Out */
		case	GTK_TRACK_TYPE_AOU:
			mix_volume *= d_track_info_ptr->outvol;
			mix_volume >>= 8;
			d_stereo = d_track_info_ptr->stereo;
			break;
		}

		/* En cas de source stereo, il faut doubler les volumes de mixage pour
		   conserver la quantite de samples avant et apres mixage. Par contre si
		   la source est un sample, on n'applique pas cette regle (moyenne des
		   cannaux). */
		if (s_stereo == 2 && s_track_type != Pattern_TYPE_SPL)
		{
			mix_volume <<= 1;
		}

		/* Volume gauche */
		left_mix_volume = (  (ULWORD)(mix_volume & 0xFFFFL)
								 * (ULWORD)(0x10000L - mix_panning)) >> 16;
		left_mix_volume +=   (mix_volume / 65536) * (0x10000L - mix_panning);

		/* Volume droit */
		right_mix_volume = (  (ULWORD)(mix_volume & 0xFFFFL)
								  * (ULWORD)mix_panning) >> 16;
		right_mix_volume +=   (mix_volume / 65536) * mix_panning;

		/* Stoquage pour le decliqueur */
		d_track_info_ptr->mix.src.track_conf [track_index].cr.old_vol [0]
			= d_track_info_ptr->mix.src.track_conf [track_index].cr.new_vol [0];
		d_track_info_ptr->mix.src.track_conf [track_index].cr.old_vol [1]
			= d_track_info_ptr->mix.src.track_conf [track_index].cr.new_vol [1];
		d_track_info_ptr->mix.src.track_conf [track_index].cr.new_vol [0] = left_mix_volume;
		d_track_info_ptr->mix.src.track_conf [track_index].cr.new_vol [1] = right_mix_volume;

		/* Recupere l'adresse du buffer source */
		if (track_order_ptr->s_buf_nbr >= 0)
		{
			s_buffer_ptr = MIX_int_buffer_ptr [s_stereo - 1] [track_order_ptr->s_buf_nbr];
		}

		/* Adresse du buffer d'Audio In */
		else if (s_track_type == Pattern_TYPE_AIN)
		{
			s_buffer_ptr = MIX_driver_out_buffer_ptr [track_order_ptr->s_track_nbr];
		}

		/* Recupere l'adresse du buffer destination */
		if (track_order_ptr->d_buf_nbr >= 0)
		{
			d_buffer_ptr = MIX_int_buffer_ptr [d_stereo - 1] [track_order_ptr->d_buf_nbr];
		}

		/* Adresse du buffer Audio Out */
		else
		{
			d_buffer_ptr = MIX_driver_out_buffer_ptr [track_order_ptr->d_track_nbr];
		}

/*______________________________________________
 *
 * Mixage
 *______________________________________________
 */

		/* Si la piste source est un sample */
		if (s_track_type == Pattern_TYPE_SPL)
		{
			/* Si la source est un buffer et pas le sample lui-meme */
			if (track_order_ptr->s_buf_nbr >= 0)
			{
				/* Si le sample est utilise pour la premiere fois */
				if (MIX_use_of_track [s_global_track_nbr].first >= mix_cnt)
				{
					/*	Reechantillonne le sample dans le buffer source en
						tenant compte de la resolution du sample, et en
						laissant volume et stereo tels quels. */
					/*** Voir si on peut pas optimiser un peu au
					     lieu de reutiliser la routine generale ***/
					if (s_stereo == 2)
					{
						MIX_resample_and_mix_sample (track_order_ptr->s_track_nbr,
															  (SLWORD *)s_buffer_ptr, s_stereo,
															  0x10000L, 0x10000L, true);
					}
					else
					{
						MIX_resample_and_mix_sample (track_order_ptr->s_track_nbr,
															  (SLWORD *)s_buffer_ptr, s_stereo,
															  0x08000L, 0x08000L, true);
					}

					/* Applique le filtre passe-bas si necessaire */
					if (s_track_info_ptr->score.lpf.filter_flag)
					{
						FILT_do_iir_filtering ((const SLWORD *)s_buffer_ptr,
						                       (SLWORD *)s_buffer_ptr,
						                       s_stereo, s_stereo,
						                       s_track_info_ptr->mix.spl.coef_x,
						                       s_track_info_ptr->mix.spl.coef_y, 3,
						                       s_track_info_ptr->mix.spl.buffer_x,
						                       s_track_info_ptr->mix.spl.buffer_y,
						                       s_track_info_ptr->mix.spl.buffer_pos, 4,
						                       MIX_frame_length);
					}

					/* Initialise le decliqueur avec les nouvelles valeurs */
					MIX_cr_set_new_values (s_track_info_ptr, (const SLWORD *)s_buffer_ptr, NULL);
				}

				/* Mixe le buffer source sur le buffer destination, en tenant
					compte de la stereo de chaque buffer. */
				MIX_mix_track ((const SLWORD *)s_buffer_ptr, d_buffer_ptr, MIX_frame_length,
									s_stereo, d_stereo,
									left_mix_volume, right_mix_volume, erase_flag);
			}

			/* La piste source est le sample lui-meme */
			else
			{
				/*	Reechantillonne et mixe le sample source au buffer
					destination, en tenant compte de la stereo de
					chaque buffer et de la resolution du sample. */
				MIX_resample_and_mix_sample (track_order_ptr->s_track_nbr,
				                             d_buffer_ptr, d_stereo,
													  left_mix_volume, right_mix_volume, erase_flag);
			}
		}

		/* La piste source n'est pas un sample */
		else
		{
			if (mute_flag)
			{
				if (erase_flag)
				{
					memset (d_buffer_ptr, 0, MIX_frame_length * d_stereo * sizeof (*d_buffer_ptr));
				}
			}

			else
			{
				/* Mixe le buffer source sur le buffer destination, en tenant
					compte de la stereo de chaque buffer. */
				MIX_mix_track ((const SLWORD *) s_buffer_ptr, d_buffer_ptr, MIX_frame_length,
									s_stereo, d_stereo,
									left_mix_volume, right_mix_volume, erase_flag);
			}
		}

		/* Dernier index de la piste source */
		if (track_index >= d_track_info_ptr->mix.src.nbr_s_tracks - 1)
		{
			/* Decliqueur */
			MIX_cr_remove_clicks (d_track_info_ptr, d_buffer_ptr);

			/* La piste destination est un effet */
			if (d_track_type == Pattern_TYPE_FX)
			{
				/* Appliquer l'effet sur le buffer wet */
				MIXFX_do_fx (d_track_info_ptr,
				             MIX_int_buffer_ptr [d_stereo - 1]
				                                [track_order_ptr->dry_buf_nbr],
				             MIX_int_buffer_ptr [d_track_info_ptr->stereo - 1]
				                                [track_order_ptr->wet_buf_nbr]);

				/* Initialise le decliqueur avec les nouvelles valeurs */
				MIX_cr_set_new_values (d_track_info_ptr,
				                       MIX_int_buffer_ptr [d_track_info_ptr->stereo - 1]
				                                          [track_order_ptr->wet_buf_nbr],
				                       MIX_int_buffer_ptr [d_stereo - 1]
				                                          [track_order_ptr->dry_buf_nbr]);
			}
		}

		track_order_ptr ++;	/* Mixage suivant */
	}

	/* Fait avancer les samples */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_SPL]; track ++)
	{
		MIX_set_new_sample_position (track);
	}

	/* Memorise les donnees utile au decliqueur pour le tick suivant */
	MIX_save_click_removal_info ();

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: MIX_set_new_sample_position                                    */
/*      Description: Actualise la position d'un sample.                     */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste de sample qu'on veut mettre a jour.   */
/*==========================================================================*/

void	MIX_set_new_sample_position (int track)
{
	LWORD		sample_length;
	int		direction;
	int		loop_type;
	double	temp_dbl;
	double	freq_dbl;
	SLWORD	curpos_int;
	double	curpos_dbl;
	SLWORD	newpos_int;
	ULWORD	newpos_frac;
	double	newpos_dbl;
	SLWORD	reppos;
	LWORD		replen;
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

	reppos = track_info_ptr->mix.spl.reppos;
	replen = track_info_ptr->mix.spl.replen;
	sample_length = reppos + replen;
	curpos_int = track_info_ptr->mix.spl.curpos_int;
	loop_type = track_info_ptr->mix.spl.loopmode;
	if (   (   curpos_int >= sample_length
	        || curpos_int < 0)
		 && loop_type != WaveForm_LOOP_TYPE_PP)
	{
		return;
	}

	direction = track_info_ptr->mix.spl.direction;
	freq_dbl =   PLAY_convert_period_to_rel_freq (track_info_ptr->mix.spl.outperiod)
	           * track_info_ptr->mix.spl.freq / MIX_replay_freq;
	curpos_dbl = curpos_int + track_info_ptr->mix.spl.curpos_frac * (1.0/4294967296.0);

	/* Calcule la nouvelle position du sample */
	if (direction == 0)
	{
		newpos_dbl = curpos_dbl + MIX_frame_length * freq_dbl;
	}
	else
	{
		newpos_dbl = curpos_dbl - MIX_frame_length * freq_dbl;
	}

	/* Si on est en ping-pong, marche arriere et qu'on depasse
		de la boucle a gauche, on se remet en marche avant pour
		traiter le bouclage. */
	if (   loop_type == WaveForm_LOOP_TYPE_PP
		 && direction != 0
		 && newpos_dbl <= reppos - 1)
	{
		direction = 0;
		newpos_dbl = (reppos * 2 - 1) - newpos_dbl;
	}

	newpos_frac = (ULWORD) (modf (newpos_dbl, &temp_dbl) * 4294967296.0);
	newpos_int = (LWORD) newpos_dbl - (newpos_dbl < 0);

	/* Si on depasse du sample a droite, donc qu'on
		est dirige vers l'avant */
	if (newpos_int >= sample_length)
	{
		/* Bouclage normal */
		if (loop_type == WaveForm_LOOP_TYPE_FWD)
		{
			newpos_int -= sample_length;
			newpos_int %= replen;
			newpos_int += reppos;
		}

		/* Ping-pong loop */
		else if (loop_type == WaveForm_LOOP_TYPE_PP)
		{
			newpos_int -= reppos;
			newpos_int %= replen * 2;
			if (newpos_int >= replen)
			{
				direction = 1;
				newpos_int = (replen * 2 - 1) - newpos_int;
				if (newpos_frac != 0)
				{
					newpos_int --;
					newpos_frac = -(SLWORD)newpos_frac;
				}
			}
			newpos_int += reppos;
		}
	}

	track_info_ptr->mix.spl.curpos_int = newpos_int;
	track_info_ptr->mix.spl.curpos_frac = newpos_frac;
	track_info_ptr->mix.spl.direction = direction;
}



/*==========================================================================*/
/*      Nom: MIX_save_click_removal_info                                    */
/*      Description: Sauve les informations utiles au decliqueur pour le    */
/*                   tick suivant.                                          */
/*==========================================================================*/

void	MIX_save_click_removal_info (void)
{
	int		track_type_cnt;
	int		track;
	const int	track_type [3] =
	{
		Pattern_TYPE_SPL, Pattern_TYPE_FX, Pattern_TYPE_AIN
	};
	PLAY_TRACK_INFO	*track_info_ptr;

	for (track_type_cnt = 0; track_type_cnt < 3; track_type_cnt ++)
	{
		for (track = 0; track < GTK_nbr_tracks [track_type [track_type_cnt]]; track ++)
		{
			track_info_ptr =   PLAY_track_info
			                 + PLAY_track_info_list [track_type [track_type_cnt]] [track];
			track_info_ptr->mix.spl.cr.broken_curve_state = SPLH_BrokenCurveState_NONE;

			// Bourrin.
			track_info_ptr->mix.spl.cr.old_last_value [0] [0] = track_info_ptr->mix.spl.cr.last_value [0] [0];
			track_info_ptr->mix.spl.cr.old_last_value [0] [1] = track_info_ptr->mix.spl.cr.last_value [0] [1];
			track_info_ptr->mix.spl.cr.old_last_value [1] [0] = track_info_ptr->mix.spl.cr.last_value [1] [0];
			track_info_ptr->mix.spl.cr.old_last_value [1] [1] = track_info_ptr->mix.spl.cr.last_value [1] [1];
		}
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	MIX_cr_set_new_values (PLAY_TRACK_INFO *track_info_ptr, const SLWORD *wet_buffer_ptr, const SLWORD *dry_buffer_ptr)
{
	int		buf_cnt;
	int		stereo;
	const SLWORD	*buffer_start_ptr;
	const SLWORD	*buffer_end_ptr;

	buf_cnt = 0;
	buffer_start_ptr = wet_buffer_ptr;
	stereo = track_info_ptr->stereo;
	do
	{
		buffer_end_ptr = buffer_start_ptr + (MIX_frame_length - 1) * stereo;

		track_info_ptr->mix.spl.cr.first_value [buf_cnt] [0] = *buffer_start_ptr;
		track_info_ptr->mix.spl.cr.last_value [buf_cnt] [0] = *buffer_end_ptr;
		
		if (stereo == 2)
		{
			track_info_ptr->mix.spl.cr.first_value [buf_cnt] [1] = buffer_start_ptr [1];
			track_info_ptr->mix.spl.cr.last_value [buf_cnt] [1] = buffer_end_ptr [1];
		}
		else
		{
			track_info_ptr->mix.spl.cr.first_value [buf_cnt] [1] = *buffer_start_ptr;
			track_info_ptr->mix.spl.cr.last_value [buf_cnt] [1] = *buffer_end_ptr;
		}
		
		buffer_start_ptr = dry_buffer_ptr;
		stereo = 2;		// Buffer Dry toujours stereo
		buf_cnt ++;
	}
	while (buf_cnt < 2 && buffer_start_ptr != NULL);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	MIX_cr_remove_clicks (PLAY_TRACK_INFO *track_info_ptr, SLWORD *buffer_ptr)
{
	SLWORD	offset [2];
	SLWORD	left_offset;
	SLWORD	right_offset;
	SLWORD	left_offset_step;
	SLWORD	right_offset_step;
	int		src_cnt;
	int		track_type;
	int		track;
	int		buf_type;
	int		chn_cnt;
	long		length;
	bool		broken_test_flag;
	PLAY_TRACK_INFO	*s_track_info_ptr;

	offset [0] = 0;
	offset [1] = 0;

	for (src_cnt = 0; src_cnt < track_info_ptr->mix.src.nbr_s_tracks; src_cnt ++)
	{
		/* Cherche les infos sur la piste source */
		track_type = track_info_ptr->mix.src.track_conf [src_cnt].track_type;
		track = track_info_ptr->mix.src.track_conf [src_cnt].track_nbr;
		buf_type = 0;
		if (track_type == Pattern_TYPE_FX)
		{
			buf_type = track_info_ptr->mix.src.track_conf [src_cnt].wet_flag ? 0 : 1;
		}
		s_track_info_ptr =   PLAY_track_info
		                   + PLAY_track_info_list [track_type] [track];

		/* Si on utilise un filtre de reconstruction sur une piste de samples,
		   pas besoin de prendre les infos car on a deja un decliqueur dans le
			filtre. */
		broken_test_flag = true;
		if (track_type == Pattern_TYPE_SPL)
		{
			if (s_track_info_ptr->mix.spl.recons_filter_ptr != NULL)
			{
				broken_test_flag = false;
			}
		}

		/* Recherche l'offset pour chaque canal */
		for (chn_cnt = 0; chn_cnt < 2; chn_cnt ++)
		{
			if (   s_track_info_ptr->mix.spl.cr.broken_curve_state != SPLH_BrokenCurveState_NONE
			    && broken_test_flag)
			{
				if (s_track_info_ptr->mix.spl.cr.broken_curve_state == SPLH_BrokenCurveState_START)
				{
					offset [chn_cnt] += (SLWORD)
						(((SQWORD) track_info_ptr->mix.src.track_conf [src_cnt].cr.old_vol [chn_cnt]
						* s_track_info_ptr->mix.spl.cr.old_last_value [buf_type] [chn_cnt]) >> 16);
				}
				else
				{
					offset [chn_cnt] += (SLWORD)
						(((SQWORD) track_info_ptr->mix.src.track_conf [src_cnt].cr.old_vol [chn_cnt]
						* s_track_info_ptr->mix.spl.cr.old_last_value [buf_type] [chn_cnt]
						- (SQWORD) track_info_ptr->mix.src.track_conf [src_cnt].cr.new_vol [chn_cnt]
						* s_track_info_ptr->mix.spl.cr.first_value [buf_type] [chn_cnt]) >> 16);
				}
			}
			else
			{
				offset [chn_cnt] += (SLWORD)
					(((SQWORD) (track_info_ptr->mix.src.track_conf [src_cnt].cr.old_vol [chn_cnt]
					- track_info_ptr->mix.src.track_conf [src_cnt].cr.new_vol [chn_cnt])
					* s_track_info_ptr->mix.spl.cr.old_last_value [buf_type] [chn_cnt]) >> 16);
			}
		}
	}

	/* Applique les offsets */
	length = MIX_frame_length;
	left_offset = offset [0];
	right_offset = offset [1];

	if (   left_offset != 0
	    && right_offset != 0)
	{
		if (   track_info_ptr->stereo == 2
		    || track_info_ptr->track_type == Pattern_TYPE_FX)	// Le buffer Dry est toujours stereo
		{
			left_offset_step = (SLWORD) floor ((double)left_offset / length + 0.5);
			right_offset_step = (SLWORD) floor ((double)right_offset / length + 0.5);
			do
			{
				*buffer_ptr++ += left_offset;
				*buffer_ptr++ += right_offset;
				left_offset -= left_offset_step;
				right_offset -= right_offset_step;
				length --;
			}
			while (length > 0);
		}

		else
		{
			left_offset += right_offset;
			left_offset_step = left_offset / length;
			do
			{
				*buffer_ptr++ += left_offset;
				left_offset -= left_offset_step;
				length --;
			}
			while (length > 0);
		}
	}
}



/*==========================================================================*/
/*      Nom: MIX_resample_and_mix_sample                                    */
/*      Description: Reechantillonne et mixe un sample quelconque dans un   */
/*                   buffer 24 bits mono ou stereo.                         */
/*      Parametres en entree:                                               */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*      Parametres en entree/sortie:                                        */
/*        - track: numero de la piste de sample.                            */
/*        - d_buffer_ptr: pointeur sur le buffer 24 bits destination.       */
/*==========================================================================*/

void	MIX_resample_and_mix_sample (int track, SLWORD *d_buffer_ptr, int d_stereo, SLWORD left_volume, SLWORD right_volume, bool erase_flag)
{
	LWORD		length;
	LWORD		sample_length;
	int		direction;
	int		freq_int;
	ULWORD	freq_frac;
	int		per_int;
	ULWORD	per_frac;
	double	per_dbl;
	double	freq_dbl;
	double	temp_dbl;
	SLWORD	curpos_int;
	ULWORD	curpos_frac;
	ULWORD	linear_curpos_frac;
	double	curpos_dbl;
	SLWORD	reppos;
	LWORD		replen;
	LWORD		block_size_int;
	double	block_size_dbl;
	SLWORD	limite;
	int		loop_type;
	int		resol;
	int		stereo;
	int		bytes_per_sample;
	void		*sample_ptr;
	bool		silence_flag;
	SWORD		*lin_buffer_ptr;
	long		lin_buffer_len;
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

	length = MIX_frame_length;

	/* Piste mute: on met du vide et on se tire */
	if (track_info_ptr->wet_mute_flag)
	{
		if (erase_flag)
		{
			memset (d_buffer_ptr, 0, length * d_stereo * sizeof (*d_buffer_ptr));
		}
		return;
	}

	loop_type = track_info_ptr->mix.spl.loopmode;
	direction = track_info_ptr->mix.spl.direction;
	reppos = track_info_ptr->mix.spl.reppos;
	replen = track_info_ptr->mix.spl.replen;
	sample_length = reppos + replen;
	resol = track_info_ptr->mix.spl.resol;
	stereo = track_info_ptr->mix.spl.tracks;
	bytes_per_sample = resol * stereo;
	curpos_int = track_info_ptr->mix.spl.curpos_int;
	curpos_frac = track_info_ptr->mix.spl.curpos_frac;
	curpos_dbl = curpos_int + curpos_frac * (1.0/4294967296.0);
	sample_ptr =   (BYTE *)track_info_ptr->mix.spl.sample_ptr
					 +   (curpos_int + (direction != 0 && curpos_frac != 0))
						* bytes_per_sample;

	/* Si le sample est termine, on efface et on se tire derechef */
	if (   curpos_int >= sample_length
		 && loop_type != WaveForm_LOOP_TYPE_PP)
	{
		if (erase_flag)
		{
			memset (d_buffer_ptr, 0, length * d_stereo * sizeof (*d_buffer_ptr));
		}
		return;
	}

	/* Calcul de la periode relative */
	per_dbl =   PLAY_convert_period_to_rel_per (track_info_ptr->mix.spl.outperiod)
	          * MIX_replay_freq / track_info_ptr->mix.spl.freq;
	freq_dbl = 1.0 / per_dbl;
	per_frac = (ULWORD) (modf (per_dbl, &temp_dbl) * 4294967296.0);
	per_int = (int) per_dbl;

	/* Calcul de la frequence relative */
	freq_frac = (ULWORD) (modf (freq_dbl, &temp_dbl) * 4294967296.0);
	freq_int = (int) freq_dbl;

	/* Calcul de la position fractionnaire dans le buffer lineaire, qui est
	   toujours lu vers l'avant */
	linear_curpos_frac = curpos_frac;
	if (direction != 0)
	{
		linear_curpos_frac = -(SLWORD)curpos_frac;
	}

/*______________________________________________
 *
 * Direct-To-Disk
 *______________________________________________
 */

	if (track_info_ptr->mix.spl.d2d_flag)
	{
		/* Comme on est oblige de lineariser le sample , on impose une
		   contrainte de frequence maximum sur le sample. */
		if (freq_int < MIX_LINEAR_BUFFER_MAX_FREQ_RATIO)
		{
			/* Linearisation du sample */
			MIX_make_sample_linear_16 (MIX_linear_sample_buffer, track, (LWORD)(length * freq_dbl) + 3);

			/* Reechantillonnage par filtre de reconstruction */
			if (track_info_ptr->mix.spl.recons_filter_ptr != NULL)
			{
				track_info_ptr->mix.spl.recons_filter_ptr->resample_and_mix
				(
					MIX_linear_sample_buffer, d_buffer_ptr, length, stereo, d_stereo,
					left_volume, right_volume, erase_flag, track_info_ptr->mix.spl.cr.broken_curve_state,
					linear_curpos_frac, freq_dbl, 16
				);
			}

			/* Reechantillonnage avec interpolation */
			else if (track_info_ptr->mix.spl.interpol_flag)
			{
				MIX_resample_16f_and_mix_track_interpol (MIX_linear_sample_buffer,
																	  d_buffer_ptr, length, stereo,
																	  d_stereo, left_volume, right_volume, erase_flag,
																	  linear_curpos_frac, freq_int, freq_frac);
			}

			/* Reechantillonnage sans interpolation */
			else
			{
				MIX_resample_16f_and_mix_track (MIX_linear_sample_buffer,
														  d_buffer_ptr, length, stereo,
														  d_stereo, left_volume, right_volume, erase_flag,
														  linear_curpos_frac, freq_int, freq_frac);
			}
		}

		/* Si la frequence est trop haute, on met du blanc si on est en temps reel */
		else
		{
			silence_flag = true;

			if (track_info_ptr->mix.spl.recons_filter_ptr != NULL)
			{
				lin_buffer_len = (long)(length * freq_dbl) + 3;
				lin_buffer_ptr = (SWORD *) MALLOC (lin_buffer_len * stereo * sizeof (*lin_buffer_ptr));
				if (lin_buffer_ptr != NULL)
				{
					silence_flag = false;
					MIX_make_sample_linear_16 (lin_buffer_ptr, track, (LWORD)lin_buffer_len);
					track_info_ptr->mix.spl.recons_filter_ptr->resample_and_mix
					(
						lin_buffer_ptr, d_buffer_ptr, length, stereo, d_stereo,
						left_volume, right_volume, erase_flag, track_info_ptr->mix.spl.cr.broken_curve_state,
						linear_curpos_frac, freq_dbl, 16
					);
					FREE (lin_buffer_ptr);
				}

			}

			if (erase_flag && silence_flag)
			{
				memset (d_buffer_ptr, 0, length * d_stereo * sizeof (*d_buffer_ptr));
			}
		}
	}

/*______________________________________________
 *
 * Sample en memoire
 *______________________________________________
 */

	else
	{
		/* Reechantillonnage par filtre de reconstruction */
		if (track_info_ptr->mix.spl.recons_filter_ptr != NULL)
		{
			if (freq_int < MIX_LINEAR_BUFFER_MAX_FREQ_RATIO)
			{
				lin_buffer_len = (long)(length * freq_dbl) + 3;
				MIX_make_sample_linear_16 (MIX_linear_sample_buffer, track, (LWORD)lin_buffer_len);
				track_info_ptr->mix.spl.recons_filter_ptr->resample_and_mix
				(
					MIX_linear_sample_buffer, d_buffer_ptr, length, stereo, d_stereo,
					left_volume, right_volume, erase_flag, track_info_ptr->mix.spl.cr.broken_curve_state,
					linear_curpos_frac, freq_dbl, 16
				);
			}
			else
			{
				silence_flag = true;

				lin_buffer_len = (long)(length * freq_dbl) + 3;
				lin_buffer_ptr = (SWORD *) MALLOC (lin_buffer_len * stereo * sizeof (*lin_buffer_ptr));
				if (lin_buffer_ptr != NULL)
				{
					silence_flag = false;
					MIX_make_sample_linear_16 (lin_buffer_ptr, track, (LWORD)lin_buffer_len);
					track_info_ptr->mix.spl.recons_filter_ptr->resample_and_mix
					(
						lin_buffer_ptr, d_buffer_ptr, length, stereo, d_stereo,
						left_volume, right_volume, erase_flag, track_info_ptr->mix.spl.cr.broken_curve_state,
						linear_curpos_frac, freq_dbl, 16
					);
					FREE (lin_buffer_ptr);
				}

				if (erase_flag && silence_flag)
				{
					memset (d_buffer_ptr, 0, length * d_stereo * sizeof (*d_buffer_ptr));
				}
			}
		}

		/* Avec interpolation */
		else if (   track_info_ptr->mix.spl.interpol_flag
		         && freq_int < MIX_LINEAR_BUFFER_MAX_FREQ_RATIO)
		{
			/* Linearise le sample et le convertit en 16 bits */
			/* Pourquoi + 3 ? +1 pour l'arrondi par exces, +1 pour
			   l'interpolation, et +1 par securite (sinon, manque de
			   sample remarque sur la note D1) */
			MIX_make_sample_linear_16 (MIX_linear_sample_buffer, track, (LWORD)(length * freq_dbl) + 3);

			/* Reechantillonne */
			MIX_resample_16f_and_mix_track_interpol (MIX_linear_sample_buffer,
												              d_buffer_ptr, length, stereo,
												              d_stereo, left_volume, right_volume, erase_flag,
																  linear_curpos_frac, freq_int, freq_frac);
		}

		/* Pas d'interpolation */
		else
		{
			while (length > 0)
			{
				/*--------------------*/
				/*--- Vers l'avant ---*/
				/*--------------------*/

				if (direction == 0)
				{
					/* Si on ne depasse pas du sample */
					if (  (LWORD)(curpos_dbl + (length - 1) * freq_dbl)
						 < sample_length)
					{
						/* On transfert le sample */
						if (resol == 1)
						{
							MIX_resample_8f_and_mix_track ((const SBYTE *)sample_ptr,
																	 d_buffer_ptr, length, stereo,
																	 d_stereo, left_volume, right_volume, erase_flag,
																	 curpos_frac, freq_int, freq_frac);
						}
						else
						{
							MIX_resample_16f_and_mix_track ((const SWORD *)sample_ptr,
																	  d_buffer_ptr, length, stereo,
																	  d_stereo, left_volume, right_volume, erase_flag,
																	  curpos_frac, freq_int, freq_frac);
						}
						break;
					}

					/* Si on depasse du sample */
					block_size_dbl = (sample_length - curpos_dbl) * per_dbl;
					#ifdef MACHINE_TYPE_PC
						block_size_int = (LWORD) (block_size_dbl + (4294967295.0/4294967296.0));
					#else
						block_size_int = (LWORD) ceil (block_size_dbl);
					#endif
					length -= block_size_int;
					if (resol == 1)
					{
						MIX_resample_8f_and_mix_track ((const SBYTE *)sample_ptr,
																 d_buffer_ptr, block_size_int, stereo,
																 d_stereo, left_volume, right_volume, erase_flag,
																 curpos_frac, freq_int, freq_frac);
					}
					else
					{
						MIX_resample_16f_and_mix_track ((const SWORD *)sample_ptr,
																  d_buffer_ptr, block_size_int, stereo,
																  d_stereo, left_volume, right_volume, erase_flag,
																  curpos_frac, freq_int, freq_frac);
					}
					d_buffer_ptr += block_size_int * d_stereo;

					/* Si aucun bouclage, on termine avec du remplissage */
					if (loop_type == WaveForm_LOOP_TYPE_NONE)
					{
						MIX_fill_buffer_end_24 (d_buffer_ptr, d_stereo, length, track_info_ptr,
													   left_volume, right_volume, erase_flag, 0);
						break;
					}

					/* Nouvelle position, referencee cette fois-ci a partir du
						debut de la boucle ! */
					curpos_dbl += block_size_int * freq_dbl;
					curpos_dbl -= sample_length;
					curpos_frac = (ULWORD) (modf (curpos_dbl, &temp_dbl) * 4294967296.0);
					curpos_int = (LWORD) curpos_dbl;
					sample_ptr =   (BYTE *)track_info_ptr->mix.spl.loopbuf_ptr
									 + curpos_int * bytes_per_sample;

					/* Si on ne depasse pas du buffer de bouclage */
					if (  (LWORD)(curpos_dbl + (length - 1) * freq_dbl)
						 < track_info_ptr->mix.spl.loopbuf_len)
					{
						/* On transfert le sample */
						if (resol == 1)
						{
							MIX_resample_8f_and_mix_track ((const SBYTE *)sample_ptr,
																	 d_buffer_ptr, length, stereo,
																	 d_stereo, left_volume, right_volume, erase_flag,
																	 curpos_frac, freq_int, freq_frac);
						}
						else
						{
							MIX_resample_16f_and_mix_track ((const SWORD *)sample_ptr,
																	  d_buffer_ptr, length, stereo,
																	  d_stereo, left_volume, right_volume, erase_flag,
																	  curpos_frac, freq_int, freq_frac);
						}
						break;
					}

					/* Si on depasse du buffer de bouclage */
					block_size_dbl = (track_info_ptr->mix.spl.loopbuf_len - curpos_dbl) * per_dbl;
					#ifdef MACHINE_TYPE_PC
						block_size_int = (LWORD) (block_size_dbl + (4294967295.0/4294967296.0));
					#else
						block_size_int = (LWORD) ceil (block_size_dbl);
					#endif
					length -= block_size_int;
					if (resol == 1)
					{
						MIX_resample_8f_and_mix_track ((const SBYTE *)sample_ptr,
																 d_buffer_ptr, block_size_int, stereo,
																 d_stereo, left_volume, right_volume, erase_flag,
																 curpos_frac, freq_int, freq_frac);
					}
					else
					{
						MIX_resample_16f_and_mix_track ((const SWORD *)sample_ptr,
																  d_buffer_ptr, block_size_int, stereo,
																  d_stereo, left_volume, right_volume, erase_flag,
																  curpos_frac, freq_int, freq_frac);
					}
					d_buffer_ptr += block_size_int * d_stereo;

					/* Calcule la nouvelle position et la nouvelle direction
						en cas de ping-pong loop. */
					if (loop_type == WaveForm_LOOP_TYPE_PP)
					{
						curpos_dbl = fmod (curpos_dbl + block_size_int * freq_dbl, replen * 2);

						/* Avant */
						if (curpos_dbl >= replen)
						{
							curpos_dbl -= replen;
						}

						/* Arriere */
						else
						{
							curpos_dbl = (replen - 1) - curpos_dbl;
							direction = 1;
						}
					}

					/* Calcule la nouvelle position bouclage normal */
					else
					{
						curpos_dbl = fmod (curpos_dbl + block_size_int * freq_dbl, replen);
					}

					curpos_dbl += reppos;
					curpos_frac = (ULWORD) (modf (curpos_dbl, &temp_dbl) * 4294967296.0);
					curpos_int = (LWORD) curpos_dbl - (curpos_dbl < 0);
					sample_ptr =   (BYTE *)track_info_ptr->mix.spl.sample_ptr
									 +   (curpos_int + (direction != 0 && curpos_frac != 0))
										* bytes_per_sample;
				}

				/*----------------------*/
				/*--- Vers l'arriere ---*/
				/*----------------------*/

				else
				{
					/* Ping-pong loop: limite au debut de la boucle */
					if (loop_type == WaveForm_LOOP_TYPE_PP)
					{
						limite = reppos;
					}

					/* Bouclage normal ou aucun bouclage: limite au debut du sample */
					else
					{
						limite = 0;
					}

					/* Si on ne depasse pas de la limite */
					/* Le +1 est dans la partie flottante a cause de l'arrondi du
						cast qui est vers 0 et non -oo (et floor ne marche pas sous
						interruption). 0 ou - est donc toujours hors limite. */
					if ((LWORD)(curpos_dbl + 1 - (length - 1) * freq_dbl) > limite)
					{
						if (resol == 1)
						{
							MIX_resample_8b_and_mix_track ((const SBYTE *)sample_ptr,
																	 d_buffer_ptr, length, stereo,
																	 d_stereo, left_volume, right_volume, erase_flag,
																	 curpos_frac, freq_int, freq_frac);
						}
						else
						{
							MIX_resample_16b_and_mix_track ((const SWORD *)sample_ptr,
																	  d_buffer_ptr, length, stereo,
																	  d_stereo, left_volume, right_volume, erase_flag,
																	  curpos_frac, freq_int, freq_frac);
						}
						break;
					}

					/* Si on depasse de la limite */
					block_size_dbl = (curpos_dbl + 1 - limite) * per_dbl;
					#ifdef MACHINE_TYPE_PC
						block_size_int = (LWORD) (block_size_dbl + (4294967295.0/4294967296.0));
					#else
						block_size_int = (LWORD) ceil (block_size_dbl);
					#endif
					length -= block_size_int;
					if (resol == 1)
					{
						MIX_resample_8b_and_mix_track ((const SBYTE *)sample_ptr,
																 d_buffer_ptr, block_size_int, stereo,
																 d_stereo, left_volume, right_volume, erase_flag,
																 curpos_frac, freq_int, freq_frac);
					}
					else
					{
						MIX_resample_16b_and_mix_track ((const SWORD *)sample_ptr,
																  d_buffer_ptr, block_size_int, stereo,
																  d_stereo, left_volume, right_volume, erase_flag,
																  curpos_frac, freq_int, freq_frac);
					}
					d_buffer_ptr += block_size_int * d_stereo;

					/* S'il n'y a pas de ping-pong loop (c-a-d rien ou normal),
						on termine avec du remplissage. */
					if (loop_type != WaveForm_LOOP_TYPE_PP)
					{
						MIX_fill_buffer_end_24 (d_buffer_ptr, d_stereo, length, track_info_ptr,
													   left_volume, right_volume, erase_flag, 1);
						break;
					}

					/* Sinon on calcule la nouvelle position
						et on passe a l'endroit. */
					curpos_dbl -= block_size_int * freq_dbl;
					curpos_dbl = (reppos * 2 - 1) - curpos_dbl;
					curpos_frac = (ULWORD) (modf (curpos_dbl, &temp_dbl) * 4294967296.0);
					curpos_int = (LWORD) curpos_dbl;
					sample_ptr =   (BYTE *)track_info_ptr->mix.spl.sample_ptr
									 + curpos_int * bytes_per_sample;
					direction = 0;
				}
			}
		}
	}
}



/*==========================================================================*/
/*      Nom: MIX_fill_buffer_end_24                                         */
/*      Description: Remplit la fin du buffer 24 bits quand un sample est   */
/*                   termine.                                               */
/*      Parametres en entree:                                               */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - length: longueur a mixer, en samples.                           */
/*        - track_info_ptr: pointeur sur les information de la piste du     */
/*                          sample.                                         */
/*        - left_volume: volume gauche 0...10000...?                        */
/*        - right_volume: volume droit 0...10000...?                        */
/*        - erase_flag: true indique qu'on doit mettre directement les      */
/*                      donnees dans le buffer destination au lieu de les   */
/*                      mixer avec son ancien contenu.                      */
/*        - direction: 0 si on avance vers l'avant, 1 vers l'arriere.       */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: pointeur sur le buffer 24 bits destination.       */
/*==========================================================================*/

void	MIX_fill_buffer_end_24 (SLWORD *d_buffer_ptr,
		                        int d_stereo,
		                        LWORD length,
		                        const PLAY_TRACK_INFO *track_info_ptr,
		                        SLWORD left_volume,
		                        SLWORD right_volume,
		                        bool erase_flag,
		                        int direction)
{
	SLWORD	left_sample;
	SLWORD	right_sample;
	void		*sample_ptr;

	/* Volumes sur 12 bits */
	left_volume /= 16;
	right_volume /= 16;

	if (direction == 0)
	{
		sample_ptr = track_info_ptr->mix.spl.loopbuf_ptr;
	}
	else
	{
		sample_ptr = track_info_ptr->mix.spl.sample_ptr;
	}

	/* Motif 16 bits */
	if (track_info_ptr->mix.spl.resol == 2)
	{
		left_sample = *((SWORD *)sample_ptr);
		if (track_info_ptr->mix.spl.tracks == 1)
		{
			right_sample = left_sample;
		}
		else
		{
			right_sample = *((SWORD *)sample_ptr + 1);
		}
	}

	/* Motif 8 bits */
	else
	{
		left_sample = *((SBYTE *)sample_ptr) << 8;
		if (track_info_ptr->mix.spl.tracks == 1)
		{
			right_sample = left_sample;
		}
		else
		{
			right_sample = *((SBYTE *)sample_ptr + 1) << 8;
		}
	}

	/* Remplissage stereo */
	if (d_stereo == 2)
	{
		left_sample = (left_sample * left_volume) / 16;
		right_sample = (right_sample * right_volume) / 16;
		if (erase_flag)
		{
			for ( ; length > 0; length --)
			{
				*d_buffer_ptr++ = left_sample;
				*d_buffer_ptr++ = right_sample;
				left_sample = left_sample * 3 / 4;
				right_sample = right_sample * 3 / 4;
			}
		}

		else
		{
			for ( ; length > 0; length --)
			{
				*d_buffer_ptr++ += left_sample;
				*d_buffer_ptr++ += right_sample;
				left_sample = left_sample * 3 / 4;
				right_sample = right_sample * 3 / 4;
			}
		}
	}

	/* Remplissage mono */
	else
	{
		left_sample = (left_sample * left_volume + right_sample * right_volume) / 32;
		if (erase_flag)
		{
			for ( ; length > 0; length --)
			{
				*d_buffer_ptr++ = left_sample;
				left_sample = left_sample * 3 / 4;
			}
		}

		else
		{
			for ( ; length > 0; length --)
			{
				*d_buffer_ptr++ += left_sample;
				left_sample = left_sample * 3 / 4;
			}
		}
	}
}



/*==========================================================================*/
/*      Nom: MIX_fill_buffer_end_16                                         */
/*      Description: Remplit la fin du buffer 16 bits quand un sample est   */
/*                   termine.                                               */
/*      Parametres en entree:                                               */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - length: longueur a mixer, en samples.                           */
/*        - track_info_ptr: pointeur sur les information de la piste du     */
/*                          sample.                                         */
/*        - direction: 0 si on avance vers l'avant, 1 vers l'arriere.       */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: pointeur sur le buffer 24 bits destination.       */
/*==========================================================================*/

void	MIX_fill_buffer_end_16 (SWORD *d_buffer_ptr,
		                        int d_stereo,
		                        LWORD length,
		                        const PLAY_TRACK_INFO *track_info_ptr,
		                        int direction)
{
	SLWORD	left_sample;
	SLWORD	right_sample;
	void		*sample_ptr;

	if (direction == 0)
	{
		sample_ptr = track_info_ptr->mix.spl.loopbuf_ptr;
	}
	else
	{
		sample_ptr = track_info_ptr->mix.spl.sample_ptr;
	}

	/* Motif 16 bits */
	if (track_info_ptr->mix.spl.resol == 2)
	{
		left_sample = *((SWORD *)sample_ptr);
		if (track_info_ptr->mix.spl.tracks == 1)
		{
			right_sample = left_sample;
		}
		else
		{
			right_sample = *((SWORD *)sample_ptr + 1);
		}
	}

	/* Motif 8 bits */
	else
	{
		left_sample = *((SBYTE *)sample_ptr) << 8;
		if (track_info_ptr->mix.spl.tracks == 1)
		{
			right_sample = left_sample;
		}
		else
		{
			right_sample = *((SBYTE *)sample_ptr + 1) << 8;
		}
	}

	/* Remplissage stereo */
	if (d_stereo == 2)
	{
		for ( ; length > 0; length --)
		{
			*d_buffer_ptr++ = (SWORD)left_sample;
			*d_buffer_ptr++ = (SWORD)right_sample;
			left_sample = left_sample * 3 / 4;
			right_sample = right_sample * 3 / 4;
		}
	}

	/* Remplissage mono */
	else
	{
		left_sample = (left_sample + right_sample) / 2;
		for ( ; length > 0; length --)
		{
			*d_buffer_ptr++ = (SWORD)left_sample;
			left_sample = left_sample * 3 / 4;
		}
	}
}



/*==========================================================================*/
/*      Nom: MIX_mix_track                                                  */
/*      Description: Mixe un buffer 24 bits sur un autre, avec modification */
/*                   du volume. La stereo de chaque buffer est prise en     */
/*                   compte, ainsi que le mix/ecrasement. Les overflows ne  */
/*                   sont pas controles.                                    */
/*      Parametres en entree:                                               */
/*        - s_buffer_ptr: adresse du buffer source.                         */
/*        - length: longueur a mixer, en samples.                           */
/*        - s_stereo: nombre de voies du buffer source (1 ou 2).            */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - left_volume: volume de mixage gauche (0...10000...?).           */
/*        - right_volume: volume de mixage droit (0...10000...?).           */
/*        - erase_flag: true indique qu'on doit mettre directement les      */
/*                      donnees dans le buffer destination au lieu de les   */
/*                      mixer avec son ancien contenu.                      */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: adresse du buffer destination.                    */
/*==========================================================================*/

void	MIX_mix_track (const SLWORD *s_buffer_ptr,
							SLWORD *d_buffer_ptr,
							LWORD length,
							int s_stereo,
							int d_stereo,
							SLWORD left_volume,
							SLWORD right_volume,
							bool erase_flag)
{
	SLWORD		val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 2) + (d_stereo << 1) + erase_flag)
	{
	/* Mono -> Mono, mixage */
	case	(1<<2) + (1<<1) + 0:
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ += BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, left_volume);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Mono, ecrasement */
	case	(1<<2) + (1<<1) + 1:
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ = BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, left_volume);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, mixage */
	case	(1<<2) + (2<<1) + 0:
		do
		{
			val = *s_buffer_ptr++;
			*d_buffer_ptr++ += BASE_MUL_32S_32S_M32S (val, left_volume);
			*d_buffer_ptr++ += BASE_MUL_32S_32S_M32S (val, right_volume);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, ecrasement */
	case	(1<<2) + (2<<1) + 1:
		do
		{
			val = *s_buffer_ptr++;
			*d_buffer_ptr++ = BASE_MUL_32S_32S_M32S (val, left_volume);
			*d_buffer_ptr++ = BASE_MUL_32S_32S_M32S (val, right_volume);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, mixage */
	case	(2<<2) + (1<<1) + 0:
		do
		{
			val = BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, left_volume);
			val += BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, right_volume);
			*d_buffer_ptr++ += val;
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, ecrasement */
	case	(2<<2) + (1<<1) + 1:
		do
		{
			val = BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, left_volume);
			val += BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, right_volume);
			*d_buffer_ptr++ = val;
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, mixage */
	case	(2<<2) + (2<<1) + 0:
		do
		{
			*d_buffer_ptr++ += BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, left_volume);
			*d_buffer_ptr++ += BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, right_volume);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<2) + (2<<1) + 1:
		do
		{
			*d_buffer_ptr++ = BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, left_volume);
			*d_buffer_ptr++ = BASE_MUL_32S_32S_M32S (*s_buffer_ptr++, right_volume);
			length --;
		}
		while (length > 0);
		break;
	}
}



/*==========================================================================*/
/*      Nom: MIX_resample_16f_and_mix_track                                 */
/*           MIX_resample_16b_and_mix_track                                 */
/*      Description: Reechantillonne et mixe un buffer 16 bits sur un autre */
/*                   buffer 24 bits, avec modification du volume. La stereo */
/*                   de chaque buffer est prise en compte, ainsi que le     */
/*                   mix/ecrasement. Les overflows ne sont pas controles.   */
/*                   _16f_ va a l'endroit, et _16b_ a l'envers.             */
/*      Parametres en entree:                                               */
/*        - s_buffer_ptr: adresse du buffer source.                         */
/*        - length: longueur a mixer, en samples.                           */
/*        - s_stereo: nombre de voies du buffer source (1 ou 2).            */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - left_volume: volume de mixage gauche (0...10000...?).           */
/*        - right_volume: volume de mixage droit (0...10000...?).           */
/*        - erase_flag: true indique qu'on doit mettre directement les      */
/*                      donnees dans le buffer destination au lieu de les   */
/*                      mixer avec son ancien contenu.                      */
/*        - pos_frac: position fractionnaire du reechantillonnage (sur un   */
/*                    mot long).                                            */
/*        - freq_int: frequence relative de reechantillonnage, partie       */
/*                    entiere.                                              */
/*        - freq_frac: frequence relative de reechantillonnage, partie      */
/*                     fractionnaire (sur un mot long).                     */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: adresse du buffer destination.                    */
/*==========================================================================*/

void	MIX_resample_16f_and_mix_track (const SWORD *s_buffer_ptr,
												  SLWORD *d_buffer_ptr,
												  LWORD length,
												  int s_stereo,
												  int d_stereo,
												  SLWORD left_volume,
												  SLWORD right_volume,
												  bool erase_flag,
												  ULWORD pos_frac,
												  int freq_int,
												  ULWORD freq_frac)
{
	SLWORD		val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	/* Volumes sur 12 bits */
	left_volume /= 16;
	right_volume /= 16;

	switch ((s_stereo << 2) + (d_stereo << 1) + erase_flag)
	{
	/* Mono -> Mono, mixage */
	case	(1<<2) + (1<<1) + 0:
		freq_int *= 2;
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ += ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Mono, ecrasement */
	case	(1<<2) + (1<<1) + 1:
		freq_int *= 2;
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ = ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, mixage */
	case	(1<<2) + (2<<1) + 0:
		freq_int *= 2;
		do
		{
			val = (SLWORD) (*s_buffer_ptr);
			*d_buffer_ptr++ += (val * left_volume) / 16;
			*d_buffer_ptr++ += (val * right_volume) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, ecrasement */
	case	(1<<2) + (2<<1) + 1:
		freq_int *= 2;
		do
		{
			val = (SLWORD) (*s_buffer_ptr);
			*d_buffer_ptr++ = (val * left_volume) / 16;
			*d_buffer_ptr++ = (val * right_volume) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, mixage */
	case	(2<<2) + (1<<1) + 0:
		freq_int *= 4;
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * left_volume;
			val += (SLWORD)(*(s_buffer_ptr + 1)) * right_volume;
			*d_buffer_ptr++ += val / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, ecrasement */
	case	(2<<2) + (1<<1) + 1:
		freq_int *= 4;
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * left_volume;
			val += (SLWORD)(*(s_buffer_ptr + 1)) * right_volume;
			*d_buffer_ptr++ = val / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, mixage */
	case	(2<<2) + (2<<1) + 0:
		freq_int *= 4;
		do
		{
			*d_buffer_ptr++ += ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			*d_buffer_ptr++ += ((SLWORD)(*(s_buffer_ptr + 1)) * right_volume) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<2) + (2<<1) + 1:
		freq_int *= 4;
		do
		{
			*d_buffer_ptr++ = ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			*d_buffer_ptr++ = ((SLWORD)(*(s_buffer_ptr + 1)) * right_volume) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



void	MIX_resample_16b_and_mix_track (const SWORD *s_buffer_ptr,
												  SLWORD *d_buffer_ptr,
												  LWORD length,
												  int s_stereo,
												  int d_stereo,
												  SLWORD left_volume,
												  SLWORD right_volume,
												  bool erase_flag,
												  ULWORD pos_frac,
												  int freq_int,
												  ULWORD freq_frac)
{
	SLWORD		val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	/* Volumes sur 12 bits */
	left_volume /= 16;
	right_volume /= 16;

	switch ((s_stereo << 2) + (d_stereo << 1) + erase_flag)
	{
	/* Mono -> Mono, mixage */
	case	(1<<2) + (1<<1) + 0:
		freq_int *= 2;
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ += ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Mono, ecrasement */
	case	(1<<2) + (1<<1) + 1:
		freq_int *= 2;
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ = ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, mixage */
	case	(1<<2) + (2<<1) + 0:
		freq_int *= 2;
		do
		{
			val = (SLWORD) (*s_buffer_ptr);
			*d_buffer_ptr++ += (val * left_volume) / 16;
			*d_buffer_ptr++ += (val * right_volume) / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, ecrasement */
	case	(1<<2) + (2<<1) + 1:
		freq_int *= 2;
		do
		{
			val = (SLWORD) (*s_buffer_ptr);
			*d_buffer_ptr++ = (val * left_volume) / 16;
			*d_buffer_ptr++ = (val * right_volume) / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, mixage */
	case	(2<<2) + (1<<1) + 0:
		freq_int *= 4;
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * left_volume;
			val += (SLWORD)(*(s_buffer_ptr + 1)) * right_volume;
			*d_buffer_ptr++ += val / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, ecrasement */
	case	(2<<2) + (1<<1) + 1:
		freq_int *= 4;
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * left_volume;
			val += (SLWORD)(*(s_buffer_ptr + 1)) * right_volume;
			*d_buffer_ptr++ = val / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, mixage */
	case	(2<<2) + (2<<1) + 0:
		freq_int *= 4;
		do
		{
			*d_buffer_ptr++ += ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			*d_buffer_ptr++ += ((SLWORD)(*(s_buffer_ptr + 1)) * right_volume) / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<2) + (2<<1) + 1:
		freq_int *= 4;
		do
		{
			*d_buffer_ptr++ = ((SLWORD)(*s_buffer_ptr) * left_volume) / 16;
			*d_buffer_ptr++ = ((SLWORD)(*(s_buffer_ptr + 1)) * right_volume) / 16;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



/*==========================================================================*/
/*      Nom: MIX_resample_16f_and_mix_track_interpol                        */
/*      Description: Reechantillonne et mixe un buffer 16 bits sur un autre */
/*                   buffer 24 bits, avec modification du volume. La stereo */
/*                   de chaque buffer est prise en compte, ainsi que le     */
/*                   mix/ecrasement. Les overflows ne sont pas controles.   */
/*                   Il y a interpolation lors du reechantillonnage.        */
/*                   Le sample source doit comporter un echantillon de plus */
/*                   que la longueur annoncee a cause de l'interpolation.   */
/*      Parametres en entree:                                               */
/*        - s_buffer_ptr: adresse du buffer source.                         */
/*        - length: longueur a mixer, en samples.                           */
/*        - s_stereo: nombre de voies du buffer source (1 ou 2).            */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - left_volume: volume de mixage gauche (0...10000...?).           */
/*        - right_volume: volume de mixage droit (0...10000...?).           */
/*        - erase_flag: true indique qu'on doit mettre directement les      */
/*                      donnees dans le buffer destination au lieu de les   */
/*                      mixer avec son ancien contenu.                      */
/*        - pos_frac: position fractionnaire du reechantillonnage (sur un   */
/*                    mot long).                                            */
/*        - freq_int: frequence relative de reechantillonnage, partie       */
/*                    entiere.                                              */
/*        - freq_frac: frequence relative de reechantillonnage, partie      */
/*                     fractionnaire (sur un mot long).                     */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: adresse du buffer destination.                    */
/*==========================================================================*/

void	MIX_resample_16f_and_mix_track_interpol (const SWORD *s_buffer_ptr,
												           SLWORD *d_buffer_ptr,
												           LWORD length,
												           int s_stereo,
												           int d_stereo,
												           SLWORD left_volume,
												           SLWORD right_volume,
												           bool erase_flag,
												           ULWORD pos_frac,
												           int freq_int,
												           ULWORD freq_frac)
{
	SLWORD		pos_frac_2;
	SLWORD		val_1;
	SLWORD		val_2;
	SLWORD		s1_vol;
	SLWORD		s2_vol;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	/* Volumes sur 12 bits */
	left_volume /= 16;
	right_volume /= 16;

	switch ((s_stereo << 2) + (d_stereo << 1) + erase_flag)
	{
	/* Mono -> Mono, mixage */
	case	(1<<2) + (1<<1) + 0:
		freq_int *= 2;
		left_volume += right_volume;
		do
		{
			pos_frac_2 = pos_frac >> 24;	/* Pos frac sur 8 bits */
			s2_vol = (pos_frac_2 * left_volume) / 256;	/* Volume sur base 12 bits */
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ += (  *s_buffer_ptr * s1_vol
			                    + *(s_buffer_ptr + 1) * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Mono, ecrasement */
	case	(1<<2) + (1<<1) + 1:
		freq_int *= 2;
		left_volume += right_volume;
		do
		{
			pos_frac_2 = pos_frac >> 24;
			s2_vol = (pos_frac_2 * left_volume) / 256;
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ = (  *s_buffer_ptr * s1_vol
			                   + *(s_buffer_ptr + 1) * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, mixage */
	case	(1<<2) + (2<<1) + 0:
		freq_int *= 2;
		do
		{
			val_1 = *s_buffer_ptr;
			val_2 = *(s_buffer_ptr + 1);
			pos_frac_2 = pos_frac >> 24;
			s2_vol = (pos_frac_2 * left_volume) / 256;
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ += (val_1 * s1_vol + val_2 * s2_vol) / 16;
			s2_vol = (pos_frac_2 * right_volume) / 256;
			s1_vol = right_volume - s2_vol;
			*d_buffer_ptr++ += (val_1 * s1_vol + val_2 * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, ecrasement */
	case	(1<<2) + (2<<1) + 1:
		freq_int *= 2;
		do
		{
			val_1 = *s_buffer_ptr;
			val_2 = *(s_buffer_ptr + 1);
			pos_frac_2 = pos_frac >> 24;
			s2_vol = (pos_frac_2 * left_volume) / 256;
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ = (val_1 * s1_vol + val_2 * s2_vol) / 16;
			s2_vol = (pos_frac_2 * right_volume) / 256;
			s1_vol = right_volume - s2_vol;
			*d_buffer_ptr++ = (val_1 * s1_vol + val_2 * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, mixage */
	case	(2<<2) + (1<<1) + 0:
		freq_int *= 4;
		do
		{
			val_1 = *s_buffer_ptr * left_volume;
			val_1 += *(s_buffer_ptr + 1) * right_volume;
			val_2 = *(s_buffer_ptr + 2) * left_volume;
			val_2 += *(s_buffer_ptr + 3) * right_volume;
			pos_frac_2 = pos_frac >> 24;	/* Pos frac sur 8 bits */
			s2_vol = (pos_frac_2 * left_volume) / 256;	/* Volume sur base 11 bits */
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ += (val_1 * s1_vol + val_2 * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, ecrasement */
	case	(2<<2) + (1<<1) + 1:
		freq_int *= 4;
		do
		{
			val_1 = *s_buffer_ptr * left_volume;
			val_1 += *(s_buffer_ptr + 1) * right_volume;
			val_2 = *(s_buffer_ptr + 2) * left_volume;
			val_2 += *(s_buffer_ptr + 3) * right_volume;
			pos_frac_2 = pos_frac >> 24;	/* Pos frac sur 8 bits */
			s2_vol = (pos_frac_2 * left_volume) / 256;	/* Volume sur base 11 bits */
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ = (val_1 * s1_vol + val_2 * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, mixage */
	case	(2<<2) + (2<<1) + 0:
		freq_int *= 4;
		do
		{
			pos_frac_2 = pos_frac >> 24;
			s2_vol = (pos_frac_2 * left_volume) / 256;
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ += (*s_buffer_ptr * s1_vol + *(s_buffer_ptr + 2) * s2_vol) / 16;
			s2_vol = (pos_frac_2 * right_volume) / 256;
			s1_vol = right_volume - s2_vol;
			*d_buffer_ptr++ += (*(s_buffer_ptr + 1) * s1_vol + *(s_buffer_ptr + 3) * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<2) + (2<<1) + 1:
		freq_int *= 4;
		do
		{
			pos_frac_2 = pos_frac >> 24;
			s2_vol = (pos_frac_2 * left_volume) / 256;
			s1_vol = left_volume - s2_vol;
			*d_buffer_ptr++ = (*s_buffer_ptr * s1_vol + *(s_buffer_ptr + 2) * s2_vol) / 16;
			s2_vol = (pos_frac_2 * right_volume) / 256;
			s1_vol = right_volume - s2_vol;
			*d_buffer_ptr++ = (*(s_buffer_ptr + 1) * s1_vol + *(s_buffer_ptr + 3) * s2_vol) / 16;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



/*==========================================================================*/
/*      Nom: MIX_resample_16f                                               */
/*           MIX_resample_16b                                               */
/*      Description: Reechantillonne un buffer 16 bits sur un autre buffer  */
/*                   24 bits, sans modification du volume. La stereo        */
/*                   de chaque buffer est prise en compte.                  */
/*                   _16f va a l'endroit, et _16b a l'envers.               */
/*      Parametres en entree:                                               */
/*        - s_buffer_ptr: adresse du buffer source.                         */
/*        - length: longueur a mixer, en samples.                           */
/*        - s_stereo: nombre de voies du buffer source (1 ou 2).            */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - pos_frac: position fractionnaire du reechantillonnage (sur un   */
/*                    mot long).                                            */
/*        - freq_int: frequence relative de reechantillonnage, partie       */
/*                    entiere.                                              */
/*        - freq_frac: frequence relative de reechantillonnage, partie      */
/*                     fractionnaire (sur un mot long).                     */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: adresse du buffer destination.                    */
/*==========================================================================*/

void	MIX_resample_16f (const SWORD *s_buffer_ptr,
								SLWORD *d_buffer_ptr,
								LWORD length,
								int s_stereo,
								int d_stereo,
								ULWORD pos_frac,
								int freq_int,
								ULWORD freq_frac)
{
	SLWORD	val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 1) + d_stereo)
	{
	/* Mono -> Mono */
	case	(1<<1) + 1:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 256;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo */
	case	(1<<1) + 2:
		freq_int *= 2;
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * 256;
			*d_buffer_ptr++ = val;
			*d_buffer_ptr++ = val;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono */
	case	(2<<1) + 1:
		freq_int *= 4;
		do
		{
			val = *s_buffer_ptr;
			val += *(s_buffer_ptr + 1);
			*d_buffer_ptr++ = val * 128;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<1) + 2:
		freq_int *= 4;
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 256;
			*d_buffer_ptr++ = (SLWORD)(*(s_buffer_ptr + 1)) * 256;
			MIX_STEP_16F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



void	MIX_resample_16b (const SWORD *s_buffer_ptr,
								SLWORD *d_buffer_ptr,
								LWORD length,
								int s_stereo,
								int d_stereo,
								ULWORD pos_frac,
								int freq_int,
								ULWORD freq_frac)
{
	SLWORD		val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 1) + d_stereo)
	{
	/* Mono -> Mono */
	case	(1<<1) + 1:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 256;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo */
	case	(1<<1) + 2:
		freq_int *= 2;
		do
		{
			val = *s_buffer_ptr;
			*d_buffer_ptr++ = val * 256;
			*d_buffer_ptr++ = val * 256;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono */
	case	(2<<1) + 1:
		freq_int *= 4;
		do
		{
			val = *s_buffer_ptr;
			val += *(s_buffer_ptr + 1);
			*d_buffer_ptr++ = val * 128;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<1) + 2:
		freq_int *= 4;
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 256;
			*d_buffer_ptr++ = (SLWORD)(*(s_buffer_ptr + 1)) * 256;
			MIX_STEP_16B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



/*==========================================================================*/
/*      Nom: MIX_resample_8f_and_mix_track                                  */
/*           MIX_resample_8b_and_mix_track                                  */
/*      Description: Reechantillonne et mixe un buffer 8 bits sur un        */
/*                   buffer 24 bits, avec modification du volume. La stereo */
/*                   de chaque buffer est prise en compte, ainsi que le     */
/*                   mix/ecrasement. Les overflows ne sont pas controles.   */
/*                   _8f_ va a l'endroit, et _8b_ a l'envers.               */
/*      Parametres en entree:                                               */
/*        - s_buffer_ptr: adresse du buffer source.                         */
/*        - length: longueur a mixer, en samples.                           */
/*        - s_stereo: nombre de voies du buffer source (1 ou 2).            */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - left_volume: volume de mixage gauche (0...10000...?).           */
/*        - right_volume: volume de mixage droit (0...10000...?).           */
/*        - erase_flag: true indique qu'on doit mettre directement les      */
/*                      donnees dans le buffer destination au lieu de les   */
/*                      mixer avec son ancien contenu.                      */
/*        - pos_frac: position fractionnaire du reechantillonnage (sur un   */
/*                    mot long).                                            */
/*        - freq_int: frequence relative de reechantillonnage, partie       */
/*                    entiere.                                              */
/*        - freq_frac: frequence relative de reechantillonnage, partie      */
/*                     fractionnaire (sur un mot long).                     */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: adresse du buffer destination.                    */
/*==========================================================================*/

void	MIX_resample_8f_and_mix_track (const SBYTE *s_buffer_ptr,
												 SLWORD *d_buffer_ptr,
												 LWORD length,
												 int s_stereo,
												 int d_stereo,
												 SLWORD left_volume,
												 SLWORD right_volume,
												 bool erase_flag,
												 ULWORD pos_frac,
												 int freq_int,
												 ULWORD freq_frac)
{
	SLWORD	val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 2) + (d_stereo << 1) + erase_flag)
	{
	/* Mono -> Mono, mixage */
	case	(1<<2) + (1<<1) + 0:
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ += *s_buffer_ptr * left_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Mono, ecrasement */
	case	(1<<2) + (1<<1) + 1:
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ = *s_buffer_ptr * left_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, mixage */
	case	(1<<2) + (2<<1) + 0:
		do
		{
			val = *s_buffer_ptr;
			*d_buffer_ptr++ += val * left_volume;
			*d_buffer_ptr++ += val * right_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, ecrasement */
	case	(1<<2) + (2<<1) + 1:
		do
		{
			val = *s_buffer_ptr;
			*d_buffer_ptr++ = val * left_volume;
			*d_buffer_ptr++ = val * right_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, mixage */
	case	(2<<2) + (1<<1) + 0:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ += *s_buffer_ptr * left_volume + *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, ecrasement */
	case	(2<<2) + (1<<1) + 1:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = *s_buffer_ptr * left_volume + *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, mixage */
	case	(2<<2) + (2<<1) + 0:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ += *s_buffer_ptr * left_volume;
			*d_buffer_ptr++ += *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<2) + (2<<1) + 1:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = *s_buffer_ptr * left_volume;
			*d_buffer_ptr++ = *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



void	MIX_resample_8b_and_mix_track (const SBYTE *s_buffer_ptr,
												 SLWORD *d_buffer_ptr,
												 LWORD length,
												 int s_stereo,
												 int d_stereo,
												 SLWORD left_volume,
												 SLWORD right_volume,
												 bool erase_flag,
												 ULWORD pos_frac,
												 int freq_int,
												 ULWORD freq_frac)
{
	SLWORD	val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 2) + (d_stereo << 1) + erase_flag)
	{
	/* Mono -> Mono, mixage */
	case	(1<<2) + (1<<1) + 0:
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ += *s_buffer_ptr * left_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Mono, ecrasement */
	case	(1<<2) + (1<<1) + 1:
		left_volume += right_volume;
		do
		{
			*d_buffer_ptr++ = *s_buffer_ptr * left_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, mixage */
	case	(1<<2) + (2<<1) + 0:
		do
		{
			val = *s_buffer_ptr;
			*d_buffer_ptr++ += val * left_volume;
			*d_buffer_ptr++ += val * right_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo, ecrasement */
	case	(1<<2) + (2<<1) + 1:
		do
		{
			val = *s_buffer_ptr;
			*d_buffer_ptr++ = val * left_volume;
			*d_buffer_ptr++ = val * right_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, mixage */
	case	(2<<2) + (1<<1) + 0:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ += *s_buffer_ptr * left_volume + *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono, ecrasement */
	case	(2<<2) + (1<<1) + 1:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = *s_buffer_ptr * left_volume + *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, mixage */
	case	(2<<2) + (2<<1) + 0:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ += *s_buffer_ptr * left_volume;
			*d_buffer_ptr++ += *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<2) + (2<<1) + 1:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = *s_buffer_ptr * left_volume;
			*d_buffer_ptr++ = *(s_buffer_ptr + 1) * right_volume;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



/*==========================================================================*/
/*      Nom: MIX_resample_8f                                                */
/*           MIX_resample_8b                                                */
/*      Description: Reechantillonne un buffer 8 bits sur un autre buffer   */
/*                   24 bits, sans modification du volume. La stereo        */
/*                   de chaque buffer est prise en compte.                  */
/*                   _8f va a l'endroit, et _8b a l'envers.                 */
/*      Parametres en entree:                                               */
/*        - s_buffer_ptr: adresse du buffer source.                         */
/*        - length: longueur a mixer, en samples.                           */
/*        - s_stereo: nombre de voies du buffer source (1 ou 2).            */
/*        - d_stereo: nombre de voies du buffer destination (1 ou 2).       */
/*        - pos_frac: position fractionnaire du reechantillonnage (sur un   */
/*                    mot long).                                            */
/*        - freq_int: frequence relative de reechantillonnage, partie       */
/*                    entiere.                                              */
/*        - freq_frac: frequence relative de reechantillonnage, partie      */
/*                     fractionnaire (sur un mot long).                     */
/*      Parametres en entree/sortie:                                        */
/*        - d_buffer_ptr: adresse du buffer destination.                    */
/*==========================================================================*/

void	MIX_resample_8f (const SBYTE *s_buffer_ptr,
							  SLWORD *d_buffer_ptr,
							  LWORD length,
							  int s_stereo,
							  int d_stereo,
							  ULWORD pos_frac,
							  int freq_int,
							  ULWORD freq_frac)
{
	SLWORD	val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 1) + d_stereo)
	{
	/* Mono -> Mono */
	case	(1<<1) + 1:
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 65536;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo */
	case	(1<<1) + 2:
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * 65536;
			*d_buffer_ptr++ = val;
			*d_buffer_ptr++ = val;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono */
	case	(2<<1) + 1:
		freq_int *= 2;
		do
		{
			val = *s_buffer_ptr;
			val += *(s_buffer_ptr + 1);
			*d_buffer_ptr++ = val * 65536;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<1) + 2:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 65536;
			*d_buffer_ptr++ = (SLWORD)(*(s_buffer_ptr + 1)) * 65536;
			MIX_STEP_8F (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



void	MIX_resample_8b (const SBYTE *s_buffer_ptr,
							  SLWORD *d_buffer_ptr,
							  LWORD length,
							  int s_stereo,
							  int d_stereo,
							  ULWORD pos_frac,
							  int freq_int,
							  ULWORD freq_frac)
{
	SLWORD	val;

	/* Test preliminaire */
	if (length <= 0)
	{
		return;
	}

	switch ((s_stereo << 1) + d_stereo)
	{
	/* Mono -> Mono */
	case	(1<<1) + 1:
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 65536;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Mono -> Stereo */
	case	(1<<1) + 2:
		do
		{
			val = (SLWORD)(*s_buffer_ptr) * 65536;
			*d_buffer_ptr++ = val;
			*d_buffer_ptr++ = val;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 1);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Mono */
	case	(2<<1) + 1:
		freq_int *= 2;
		do
		{
			val = *s_buffer_ptr;
			val += *(s_buffer_ptr + 1);
			*d_buffer_ptr++ = val * 65536;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;

	/* Stereo -> Stereo, ecrasement */
	case	(2<<1) + 2:
		freq_int *= 2;
		do
		{
			*d_buffer_ptr++ = (SLWORD)(*s_buffer_ptr) * 65536;
			*d_buffer_ptr++ = (SLWORD)(*(s_buffer_ptr + 1)) * 65536;
			MIX_STEP_8B (s_buffer_ptr, pos_frac, freq_int, freq_frac, 2);
			length --;
		}
		while (length > 0);
		break;
	}
}



/*==========================================================================*/
/*      Nom: MIX_make_sample_linear_16                                      */
/*      Description: Copie une partie d'un sample dans un buffer en         */
/*                   developpant si necessaire la boucle. Le buffer est     */
/*                   toujours 16 bits (conversion du sample si necessaire). */
/*      Parametres en entree:                                               */
/*        - track: Numerod de la piste qui joue le sample.                  */
/*        - length: longueur du developpement, en samples.                  */
/*      Parametres en sortie:                                               */
/*        - buffer_ptr: adresse du buffer ou doit etre developpe le sample. */
/*      Parametres en entree/sortie:                                        */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	MIX_make_sample_linear_16 (SWORD *buffer_ptr, int track, LWORD length)
{
	int		current_dir;
	int		new_dir;
	SLWORD	current_pos;
	SLWORD	new_pos;
	int		buffer_dir;
	int		nbr_tracks;
	int		resol;
	int		sample_mul;
	SLWORD		block_len;
	void		*sample_ptr;
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

	nbr_tracks = track_info_ptr->mix.spl.tracks;
	resol = track_info_ptr->mix.spl.resol;
	sample_mul = nbr_tracks * resol;
	current_dir = track_info_ptr->mix.spl.direction;
	current_pos = track_info_ptr->mix.spl.curpos_int;
	if (   current_dir != 0
	    && track_info_ptr->mix.spl.curpos_frac != 0)
	{
		current_pos ++;
	}

	while (length > 0)
	{
		MIX_get_next_linear_sample_block (track, current_pos, current_dir,
		                                  &new_pos, &new_dir, &sample_ptr,
		                                  &block_len, &buffer_dir);
		/* Sample fini */
		if (block_len == -1)
		{
			MIX_fill_buffer_end_16 (buffer_ptr, track_info_ptr->mix.spl.tracks,
						               length, track_info_ptr, current_dir);
			break;
		}

		/* Donnees indisponibles */
		else if (block_len == -2)
		{
			memset (buffer_ptr, 0, length * nbr_tracks * 2);
			break;
		}

		block_len = MIN (block_len, length);

		/* Vers l'arriere */
		if (buffer_dir != 0)
		{
			SPLH_copy_mem_invert_convert_sample_2_16 (buffer_ptr, sample_ptr,
			                                          block_len, nbr_tracks, resol);
		}

		/* Vers l'avant */
		else
		{
			SPLH_copy_mem_convert_sample_2_16 (buffer_ptr, sample_ptr,
			                                   block_len, nbr_tracks, resol);
		}

		current_pos = new_pos;
		current_dir = new_dir;
		length -= block_len;
		buffer_ptr += block_len * nbr_tracks;
	}

	return (0);
}



/*==========================================================================*/
/*      Nom: MIX_get_next_linear_sample_block                               */
/*      Description: Recherche le plus grand bloc continu de sample dispo   */
/*                   a partir de la position et de la direction donnee.     */
/*                   Donne l'adresse de ce bloc, sa longueur et les         */
/*                   nouvelles directions et positions apres le bloc.       */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste qui joue le sample.                   */
/*        - old_position: ancienne position dans le sample, en samples.     */
/*                        -1 signifie que le sample est termine (si la      */
/*                           direction est vers l'arriere).                 */
/*        - old_direction: ancienne direction (0 = fwd, 1 = backwd).        */
/*      Parametres en sortie:                                               */
/*        - position_ptr: pointeur sur la position dans le sample, en       */
/*                        samples.                                          */
/*                        -1 signifie que le sample est termine (si la      */
/*                           direction est vers l'arriere).                 */
/*        - direction_ptr: pointeur sur la direction (0 = fwd, 1 = backwd)  */
/*        - block_ptr_ptr: pointeur sur l'adresse du bloc lineaire trouve.  */
/*                         Si la direction est vers l'arriere, cette        */
/*                         variable pointe sur le dernier sample du block.  */
/*        - length_ptr: pointeur sur la longueur en sample du bloc trouve.  */
/*                      -1 indique que le sample est fini,                  */
/*                      -2 indique que les donnees ne sont pas disponibles  */
/*                         (pour le D2D).                                   */
/*        - block_dir_ptr: indique la direction dans laquelle doit etre lu  */
/*                         le bloc trouve.                                  */
/*==========================================================================*/

void	MIX_get_next_linear_sample_block (int track,
		                                  SLWORD old_position,
		                                  int old_direction,
		                                  SLWORD *position_ptr,
		                                  int *direction_ptr,
		                                  void **block_ptr_ptr,
		                                  SLWORD *length_ptr,
		                                  int *block_dir_ptr)
{
	int		sample_mul;
	int		buffer;
	LWORD		sample_length;
	LWORD		new_length;
	LWORD		buf_start;
	LWORD		buf_length;
	D2D_BUFFER_ENTRY	buffer_info;
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

	sample_mul =   track_info_ptr->mix.spl.tracks
	             * track_info_ptr->mix.spl.resol;
	sample_length =   track_info_ptr->mix.spl.reppos
	                + track_info_ptr->mix.spl.replen;

/*______________________________________________
 *
 * Direct-2-Disk
 *______________________________________________
 */

	if (track_info_ptr->mix.spl.d2d_flag)
	{
		buffer_info = D2D_get_buffer_info (track);

		/* On est dans le buffer de bouclage */
		if (   (   old_direction == 0
		        && old_position == track_info_ptr->mix.spl.reppos
				  && track_info_ptr->mix.spl.loopmode != WaveForm_LOOP_TYPE_PP)
		    || (   old_direction == 1
		        && old_position == sample_length - 1
		        && track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP))
		{
			*block_ptr_ptr = track_info_ptr->mix.spl.loopbuf_ptr;
			*length_ptr = track_info_ptr->mix.spl.loopbuf_len;
			*block_dir_ptr = 0;

			/* Arriere */
			if (old_direction != 0)
			{
				*position_ptr =   track_info_ptr->mix.spl.loopbuf_len
				                % (track_info_ptr->mix.spl.replen * 2);
				
				/* Nouvelle direction vers l'arriere */
				if (*position_ptr < track_info_ptr->mix.spl.replen)
				{
					*position_ptr =   track_info_ptr->mix.spl.reppos
						             + track_info_ptr->mix.spl.replen
						             - *position_ptr - 1;
					*direction_ptr = 1;
				}
				
				/* Nouvelle direction vers l'avant */
				else
				{
					*position_ptr +=   track_info_ptr->mix.spl.reppos
						              - track_info_ptr->mix.spl.replen;
					*direction_ptr = 0;
				}
			}

			/* Avant */
			else
			{
				*direction_ptr = 0;

				/* Bouclage normal */
				if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_FWD)
				{
					*position_ptr =   track_info_ptr->mix.spl.reppos
										 + (  track_info_ptr->mix.spl.loopbuf_len
											 % track_info_ptr->mix.spl.replen);
				}

				/* Pas de bouclage */
				else
				{
					*position_ptr = track_info_ptr->mix.spl.reppos;
					*length_ptr = -1;
				}
			}
		}

		else
		{
			/* On est dans le buffer de debut */
			if (old_position < track_info_ptr->mix.spl.startbuf_len)
			{
				*block_ptr_ptr =   (BYTE *)track_info_ptr->mix.spl.startbuf_ptr
									  + old_position * sample_mul;
				buf_start = 0;
				buf_length = track_info_ptr->mix.spl.startbuf_len;
			}

			/* On est dans un des deux buffers de D2D */
			else
			{
				/* Si les buffers de D2D existent reellement */
				if (buffer_info.buffer_ptr [0] != NULL)
				{
					for (buffer = 0; buffer < 2; buffer ++)
					{
						if (buffer_info.buf_flag [buffer]
							 && old_position >= buffer_info.buf_pos [buffer]
							 && old_position <   buffer_info.buf_pos [buffer]
													 + buffer_info.buf_length [buffer])
						{
							break;
						}
					}
				}

				/* Le buffer de D2D n'est pas encore cree (manque de memoire
				   par exemple) */
				else
				{
					buffer = 2;
				}

				/* On est bien dans un des buffers */
				if (buffer < 2)
				{
					*block_ptr_ptr =   (BYTE *)buffer_info.buffer_ptr [buffer]
						              +   (old_position - buffer_info.buf_pos [buffer])
						                * sample_mul;
					buf_start =  buffer_info.buf_pos [buffer];
					buf_length = buffer_info.buf_length [buffer];
				}

				/* Les donnees qu'on recherche ne sont pas disponibles en RAM */
				else
				{
					*length_ptr = -2;
					*block_ptr_ptr = NULL;
					*direction_ptr = old_direction;
					*position_ptr = old_position;
					*block_dir_ptr = 0;
				}
			}

			/* Ajuste le reste des donnees */
			if (*block_ptr_ptr != NULL)
			{
				*direction_ptr = old_direction;

				/* Arriere */
				if (old_direction != 0)
				{
					new_length = old_position - buf_start + 1;
					new_length = MIN (new_length, old_position + 1);
					*block_dir_ptr = 1;
					*position_ptr = old_position - new_length;

					/* On sort de la boucle ? */
					if (   track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP
					    && old_position >= track_info_ptr->mix.spl.reppos
						 && *position_ptr <= track_info_ptr->mix.spl.reppos)
					{
						*position_ptr = track_info_ptr->mix.spl.reppos;
						*direction_ptr= 0;
						new_length = old_position - track_info_ptr->mix.spl.reppos + 1;
					}
				}

				/* Avant */
				else
				{
					new_length = buf_start + buf_length - old_position;
					new_length = MIN (new_length, sample_length - old_position);
					*block_dir_ptr = 0;
					*position_ptr = old_position + new_length;

					/* On depasse ? */
					if (*position_ptr + new_length >= sample_length)
					{
						if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP)
						{
							*direction_ptr = 1;
							*position_ptr = sample_length - 1;
						}

						else
						{
							*position_ptr = track_info_ptr->mix.spl.reppos;
						}
					}
				}

				*length_ptr = new_length;
			}
		}
	}

/*______________________________________________
 *
 * Memoire
 *______________________________________________
 */

	else
	{
		/* Vers l'avant */
		if (old_direction == 0)
		{
			*block_dir_ptr = 0;

			/* Buffer de bouclage */
			if (   old_position == track_info_ptr->mix.spl.reppos
			    && (   track_info_ptr->mix.spl.replen < track_info_ptr->mix.spl.loopbuf_len
			        || track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_NONE)
				 && track_info_ptr->mix.spl.loopmode != WaveForm_LOOP_TYPE_PP)
			{
				*block_ptr_ptr = track_info_ptr->mix.spl.loopbuf_ptr;
				*length_ptr = track_info_ptr->mix.spl.loopbuf_len;
				*direction_ptr = 0;

				/* Bouclage normal */
				if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_FWD)
				{
					*position_ptr =   track_info_ptr->mix.spl.reppos
										 + (  track_info_ptr->mix.spl.loopbuf_len
											 % track_info_ptr->mix.spl.replen);
				}

				/* Pas de bouclage */
				else
				{
					*position_ptr = track_info_ptr->mix.spl.reppos;
					*length_ptr = -1;
				}
			}

			/* Sample normal */
			else
			{
				*length_ptr = sample_length - old_position;
				*block_ptr_ptr =   (BYTE *)track_info_ptr->mix.spl.sample_ptr
				                 + old_position * sample_mul;

				/* Ping-pong loop */
				if (track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP)
				{
					*position_ptr = sample_length - 1;
					*direction_ptr = 1;
				}

				/* Bouclage normal ou pas de bouclage */
				else
				{
					*position_ptr = track_info_ptr->mix.spl.reppos;
					*direction_ptr = 0;
				}
			}
		}

		/* Vers l'arriere */
		else
		{
			/* Buffer de bouclage */
			if (   old_position == sample_length - 1
			    && track_info_ptr->mix.spl.replen < track_info_ptr->mix.spl.loopbuf_len
				 && track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP)
			{
				*block_dir_ptr = 0;
				*block_ptr_ptr = track_info_ptr->mix.spl.loopbuf_ptr;
				*length_ptr = track_info_ptr->mix.spl.loopbuf_len;
				*position_ptr =   track_info_ptr->mix.spl.loopbuf_len
				                % (track_info_ptr->mix.spl.replen * 2);
				
				/* Nouvelle direction vers l'arriere */
				if (*position_ptr < track_info_ptr->mix.spl.replen)
				{
					*position_ptr =   track_info_ptr->mix.spl.reppos
						             + track_info_ptr->mix.spl.replen
						             - *position_ptr - 1;
					*direction_ptr = 1;
				}
				
				/* Nouvelle direction vers l'avant */
				else
				{
					*position_ptr +=   track_info_ptr->mix.spl.reppos
						              - track_info_ptr->mix.spl.replen;
					*direction_ptr = 0;
				}
			}

			/* Sample normal */
			else
			{
				/* Loop et on est dans la boucle (ping-pong) */
				if (   track_info_ptr->mix.spl.loopmode == WaveForm_LOOP_TYPE_PP
				    && old_position >= track_info_ptr->mix.spl.reppos)
				{
					*block_ptr_ptr =   (BYTE *)track_info_ptr->mix.spl.sample_ptr
					                 + old_position * sample_mul;
					*length_ptr = old_position - track_info_ptr->mix.spl.reppos + 1;
					*position_ptr = track_info_ptr->mix.spl.reppos;
					*direction_ptr = 0;
					*block_dir_ptr = 1;
				}

				/* Sample termine */
				else if (old_position < 0)
				{
					*position_ptr = -1;
					*length_ptr = -1;
					*block_ptr_ptr = track_info_ptr->mix.spl.sample_ptr;
					*direction_ptr = 1;
					*block_dir_ptr = 0;
				}

				/* Avant de la boucle ou pas de ping-pong loop du tout */
				else
				{
					*length_ptr = old_position + 1;
					*block_ptr_ptr =   (BYTE *)track_info_ptr->mix.spl.sample_ptr
					                 + old_position * sample_mul;
					*position_ptr = -1;
					*direction_ptr = 1;
					*block_dir_ptr = 1;
				}
			}
		}
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

