/*****************************************************************************

GRAOUMF TRACKER 2

Copyright (c) 1996 - 2002 Laurent de Soras

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

Contact the author : laurent@ohmforce.com
More information about this license : http://www.gnu.org/licenses/gpl.html

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include	<assert.h>
#include	<stdio.h>
#include <math.h>

#include "archi.h"
#include "base.h"
#include "base_ct.h"
#include	"config.h"
#include	"ConfigKey.h"
#include	"ConfigKeyValue.h"
#include "d2d.h"
#include "gtracker.h"
#include "inst.h"
#include	"List.h"
#include	"log.h"
#include "mixer.h"
#include	"Mutex.h"
#include "os.h"
#include "play_cmd.h"
#include "play_env.h"
#include "play_fx.h"
#include "player.h"
#include "samp.h"
#include "Sample.h"
#include	"SimpleCalculator.h"
#include "song.h"
#include	"splhandl.h"
#include	"String.h"
#include	"Thread.h"
#include	"WaveForm.h"



/*\\\ CONSTANTES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/* Type de forme d'ondes dans la table */
#define	PLAY_WAVEFORM_SIN				0
#define	PLAY_WAVEFORM_SQUARE			1
#define	PLAY_WAVEFORM_RAMPDOWN		2
#define	PLAY_WAVEFORM_RAMPUP			3



/*\\\ TYPES & STRUCTURES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PROTOTYPES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

signed int	PLAY_set_replay_format (void);
void	PLAY_player_thread_routine (void);

void	PLAY_copy_line_to_track_info (void);

void	PLAY_handle_effects_1st_tick (void);
void	PLAY_handle_effects_all_ticks (void);

void	PLAY_compute_final_parameters (void);
void	PLAY_compute_final_parameters_track (int track_type, int track);

void	PLAY_reset_all_tracks (void);
void	PLAY_reset_track_partition_info (PLAY_TRACK_INFO *track_info_ptr);
void	PLAY_reset_click_removal (PLAY_TRACK_INFO *track_info_ptr);

signed int	PLAY_set_nbr_source_tracks (int track_type, int track_nbr, int nbr_src);



/*\\\ VARIABLES EXTERNES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/* Indique l'etat d'interpolation par defaut des pistes */
bool	PLAY_default_interpolation_flag = true;

/* Tableau des numeros de PLAY_track_info [] pour chaque type de piste */
/* A initialiser */
int	PLAY_track_info_list [GTK_NBR_MIX_TRACK_TYPE] [GTK_NBRVOICES_MAXI];

/* Blocs d'informations de chaque piste */
/* A initialiser */
PLAY_TRACK_INFO	PLAY_track_info [PLAY_NBR_GLOBAL_TRACKS];

/* Bloc d'info pour la partition */
/* A initialiser */
PLAY_SCORE_INFO	PLAY_score_info;

/* Table des periodes. De C#-2 a G+8 en 16emes de ton, +/- 3/2 tons de securite */
float	PLAY_note_period_table [(3+(GTK_NBRNOTES_MAXI-1)+3)*8];
/* Table des periodes pour chaque finetune, indexee a partir de 0 */
float	*PLAY_note_period_table_ref = PLAY_note_period_table + 3*8;

/* Table d'onde de periode 64. -1 = -0xFF, 1 = 0xFF */
SWORD	PLAY_sin_table [4] [64] =
{
	/* Sinus */
	{
		 0x00,  0x18,  0x31,  0x4A,  0x61,  0x78,  0x8D,  0xA1,
		 0xB4,  0xC5,  0xD4,  0xE0,  0xEB,  0xF4,  0xFA,  0xFD,
		 0xFF,  0xFD,  0xFA,  0xF4,  0xEB,  0xE0,  0xD4,  0xC5,
		 0xB4,  0xA1,  0x8D,  0x78,  0x61,  0x4A,  0x31,  0x18,
		-0x00, -0x18, -0x31, -0x4A, -0x61, -0x78, -0x8D, -0xA1,
		-0xB4, -0xC5, -0xD4, -0xE0, -0xEB, -0xF4, -0xFA, -0xFD,
		-0xFF, -0xFD, -0xFA, -0xF4, -0xEB, -0xE0, -0xD4, -0xC5,
		-0xB4, -0xA1, -0x8D, -0x78, -0x61, -0x4A, -0x31, -0x18
	},

	/* Square */
	{
		 0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,
		 0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,
		 0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,
		 0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,  0xFF,
		-0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF,
		-0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF,
		-0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF,
		-0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF, -0xFF
	},

	/* Rampdown */
	{
		 0xFF,  0xF8,  0xF0,  0xE8,  0xE0,  0xD8,  0xD0,  0xC8,
		 0xC0,  0xB8,  0xB0,  0xA8,  0xA0,  0x98,  0x90,  0x88,
		 0x80,  0x78,  0x70,  0x68,  0x60,  0x58,  0x50,  0x48,
		 0x40,  0x38,  0x30,  0x28,  0x20,  0x18,  0x10,  0x08,
		-0x00, -0x08, -0x10, -0x18, -0x20, -0x28, -0x30, -0x38,
		-0x40, -0x48, -0x50, -0x58, -0x60, -0x68, -0x70, -0x78,
		-0x80, -0x88, -0x90, -0x98, -0xA0, -0xA8, -0xB0, -0xB8,
		-0xC0, -0xC8, -0xD0, -0xD8, -0xE0, -0xE8, -0xF0, -0xF8
	},

	/* Rampup */
	{
		-0xF8, -0xF0, -0xE8, -0xE0, -0xD8, -0xD0, -0xC8, -0xC0,
		-0xB8, -0xB0, -0xA8, -0xA0, -0x98, -0x90, -0x88, -0x80,
		-0x78, -0x70, -0x68, -0x60, -0x58, -0x50, -0x48, -0x40,
		-0x38, -0x30, -0x28, -0x20, -0x18, -0x10, -0x08,  0x00,
		 0x08,  0x10,  0x18,  0x20,  0x28,  0x30,  0x38,  0x40,
		 0x48,  0x50,  0x58,  0x60,  0x68,  0x70,  0x78,  0x80,
		 0x88,  0x90,  0x98,  0xA0,  0xA8,  0xB0,  0xB8,  0xC0,
		 0xC8,  0xD0,  0xD8,  0xE0,  0xE8,  0xF0,  0xF8,  0xFF
	}
};

/* Lignes de pattern pour le replay de notes instantanne */
volatile UBYTE	PLAY_new_spl_notes [GTK_NBRVOICES_MAXI * MODS_GT2_SPL_NOTE_LEN];
volatile UBYTE	PLAY_new_ain_notes [GTK_NBRVOICES_MAXI * MODS_GT2_AIN_NOTE_LEN];
volatile UBYTE	PLAY_new_fx_notes [GTK_NBRVOICES_MAXI * MODS_GT2_FX_NOTE_LEN];
volatile UBYTE	PLAY_new_mid_notes [GTK_NBRVOICES_MAXI * MODS_GT2_MID_NOTE_LEN];
volatile UBYTE	PLAY_new_line_flag [Pattern_NBR_TYPES];	/* Les flags indiquent qu'une des lignes a ete remplie */
volatile UBYTE	PLAY_new_note_flag [Pattern_NBR_TYPES] [GTK_NBRVOICES_MAXI];	/* Les flags indiquent les nouvelles notes pour chaque ligne */



/*\\\ VARIABLES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

Thread	PLAY_player_thread (PLAY_player_thread_routine);
signed int	PLAY_player_thread_state = 0;	// Indique l'etat du thread. 0 = en cours d'init, 1 = ok, -1 = erreur d'init
bool	PLAY_player_started_flag = false;	/* Indique que le player est demarre */

/* Tables exp/log a initialiser */
UWORD	PLAY_instr_vol_table_exp [0x801];	/* exp(x): volumes instruments */
UWORD	PLAY_instr_vol_table_log [0x801];	/* ln(x): volumes instruments */
UWORD	PLAY_track_vol_table_exp [0x100];	/* exp(x): volumes pistes */
UWORD	PLAY_track_vol_table_log [0x100];	/* ln(x): volumes pistes */

volatile signed int	PLAY_current_linepos = 0;	/* Donnee valide en mode Pattern et Song */
volatile signed int	PLAY_current_songpos = 0;	/* Donnee valide en mode Pattern */

/* Nouveau mode de replay */
volatile int	PLAY_new_play_mode = PLAY_MODE_STOP;
volatile	bool	PLAY_new_play_mode_flag = false;
volatile bool	PLAY_new_play_mode_cont_flag = false;
volatile signed int	PLAY_new_linepos = 0;
volatile signed int	PLAY_new_songpos = 0;

/* Tableau indiquant si une voie de sortie a clippe ou pas.
   S'il n'y a pas clipping, la variable vaut 0 ou toute autre valeur positive
   representable sur 24 bits signes. En cas de clipping, la valeur indique la
	valeur absolue du plus haut pic atteint. */
SLWORD	PLAY_clipping_level [GTK_NBROUT_MAXI];

/* Memorisation du niveau de clipping pour toutes les voies. Ce niveau n'est
   remis a 0 que par l'utilisateur exterieur (interface) */
volatile SLWORD	PLAY_max_clipping_level = 0;

// Duree ecoulee depuis le debut du playback
volatile long	PLAY_elapsed_time = 0;	// Samples



/*\\\ FONCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

/*==========================================================================*/
/*      Nom: PLAY_read_conf                                                 */
/*      Description: Lit les informations concernant le player dans le      */
/*                   fichier de configuration.                              */
/*      Parametres en entree:                                               */
/*        key_list_ptr: pointeur sur la liste des cles correspondant a la   */
/*                      cle "/player" dans le fichier de configuration.     */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	PLAY_read_conf (List *key_list_ptr)
{
	int				int_val;
	ConfigKey		*key_ptr;
	ConfigKeyValue	*key_val_ptr;
	String			key_name;
	String			key_val;

	while (! key_list_ptr->is_empty ())
	{
		/* Recupere la cle */
		key_ptr = (ConfigKey *) key_list_ptr->get (0);

		/* Recupere son nom */
		key_name = key_ptr->get_name ();
		key_val_ptr = key_ptr->get_value ();

		/* Cle anonyme: interdit */
		if (BASE_compare_string (key_name, "") == 0)
		{
			LOG_printf ("PLAY_read_conf: Warning: (in \"module_editing\") unexpected unassigned value.\n");
		}

		/* Latency */
		else if (BASE_compare_string (key_name, "latency") == 0)
		{
			if (CFG_get_int (key_val_ptr, "player/latency", int_val))
			{
				int_val = MAX (int_val, 0);
				if (OS_sound_driver.set_latency (int_val))
				{
					LOG_printf ("PLAY_read_conf: Warning: (in \"module_editing/latency\") bad parameter value.\n");
				}
			}
		}

		/* Cle inconnue */
		else
		{
			LOG_printf ("PLAY_read_conf: Warning: (in \"module_editing\") syntax error \"%s\".\n",
			            (const char *) key_name);
		}

		key_list_ptr = &key_list_ptr->next ();
	}

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_first_init                                                */
/*      Description: Initialise les differentes tables utiles pour le       */
/*                   player, ainsi que la zone de cache de D2D.             */
/*      Retour: 0 si tout va bien.                                          */
/*==========================================================================*/

signed int	PLAY_first_init (void)
{
	int		i;
	double	period;

	LOG_printf ("PLAY_first_init:\n");

	/* Table exp(x): volumes instruments [0...800] -> [0...800] */
	PLAY_instr_vol_table_exp [0] = 0;
	for (i = 1; i <= 0x800; i ++)
	{
		PLAY_instr_vol_table_exp [i] = (UWORD) floor (pow (2.0, static_cast <double> (i) / 0x100 + 3) + 0.5);
	}

	/* Table ln(x): volumes instruments [0...800] -> [0...800] */
	PLAY_instr_vol_table_log [0] = 0;
	for (i = 1; i <= 0x800; i ++)
	{
		PLAY_instr_vol_table_log [i] = MAX ((UWORD) floor ((log (static_cast <double> (i)) / LN2 - 3) * 0x100 + 0.5), 0U);
	}

	/* Table exp(x): volumes pistes [0...100[ -> [8000...10000[ */
	for (i = 0; i < 0x100; i ++)
	{
		PLAY_track_vol_table_exp [i] = (UWORD) floor (pow (2.0, static_cast <double> (i) / 0x100) * 0x8000 + 0.5);
	}

	/* Table ln(x): volumes pistes [0...100[ -> [0...100[ */
	for (i = 0; i < 0x100; i ++)
	{
		PLAY_track_vol_table_log [i] = (UWORD) floor ((log (static_cast <double> (i + 0x100)) - log (static_cast <double> (0x100))) * 0x100 / LN2 + 0.5);
	}

	/* Table des periodes.
		De C#-2 a G+8 en 16emes de ton, +/- 3/2 tons de securite */
	for (i = 0; i < (3+(GTK_NBRNOTES_MAXI-1)+3)*8; i ++)
	{
		period = pow (2.0, static_cast <double> ((Sample_REF_C2+3)*8 - i) / (12*8)) * PLAY_REF_PERIOD;
		period = MIN (period, PLAY_MAX_PERIOD);
		period = MAX (period, PLAY_MIN_PERIOD);
		PLAY_note_period_table [i] = (float) period;
	}

	if (PLAYFX_init ())
	{
		LOG_printf ("PLAY_first_init: Error: couldn't initialize effect subroutine jump tables.\n");
		return (-1);	/* Erreur lors de l'init des variables des effets */
	}

	LOG_printf ("\tOK.\n");

	return (0);
}



void	PLAY_first_restore (void)
{
	PLAYFX_restore ();
}



/*==========================================================================*/
/*      Nom: PLAY_second_init                                               */
/*      Description: Demarre le player (routine sous interruption).         */
/*                   Le driver doit deja etre installe.                     */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	PLAY_second_init (void)
{
	double	tick_len;
	int		nbr_frames;
	int		track;
	int		track_type;
	int		global_track_nbr;
	PLAY_TRACK_INFO	*track_info_ptr;

	LOG_printf ("PLAY_second_init:\n");

	tick_len = (double)(MIX_replay_freq << 1) / ((LWORD)PLAY_INIT_TEMPO * 5);
	nbr_frames = (int) ceil ((tick_len + 1) / MIX_MAX_FRAME_LENGTH);

	/* Initialisation de la stucture de la song */
	PLAY_score_info.tick_len_int = (LWORD)floor (tick_len);
	PLAY_score_info.tick_len_frac =
		(ULWORD)((tick_len - floor (tick_len)) * 4294967296.0);
	PLAY_score_info.frame_pos_int = 0;
	PLAY_score_info.frame_pos_frac = 0;
	PLAY_score_info.frame_len = (LWORD) (tick_len / nbr_frames);

	PLAY_score_info.tempo_int = PLAY_INIT_TEMPO;
	PLAY_score_info.tempo_frac = 0;
	PLAY_score_info.frame_counter = 0;
	PLAY_score_info.nbr_frames = nbr_frames;
	PLAY_score_info.time_high = PLAY_INIT_TIME_HIGH;
	PLAY_score_info.time_low = PLAY_INIT_TIME_LOW;

	PLAY_score_info.tick_counter = 0;
	PLAY_score_info.speed = PLAY_INIT_SPEED;
	PLAY_score_info.pattern_delay = 0;
	PLAY_score_info.current_line = 0;
	PLAY_score_info.current_pos = 0;
	PLAY_score_info.next_line = 0;
	PLAY_score_info.next_pos = 0;
	PLAY_score_info.next_flag = false;
	PLAY_set_linear_period_flag (PLAY_INIT_LINEAR_PERIOD_FLAG);

	PLAY_score_info.current_pattern = 0;

	PLAY_score_info.current_play_mode = PLAY_MODE_STOP;

	PLAY_elapsed_time = 0;

	/* Affectation des numeros de piste absolus */
	global_track_nbr = 0;
	for (track_type = 0; track_type < GTK_NBR_MIX_TRACK_TYPE; track_type ++)
	{
		for (track = 0; track < GTK_nbr_tracks [track_type]; track ++)
		{
			PLAY_track_info_list [track_type] [track] = global_track_nbr;
			track_info_ptr = PLAY_track_info + global_track_nbr;
			track_info_ptr->track = track;
			track_info_ptr->track_type = track_type;
			PLAY_set_track_name (track_type, track, "");
			global_track_nbr ++;
		}
	}

	PLAY_reset_mix_parameters ();

	if (PLAY_set_replay_format ())
	{
		LOG_printf ("PLAY_second_init: Error: couldn't set replay format.\n");
		return (-1);
	}

	if (PLAY_start_player ())
	{
		LOG_printf ("PLAY_second_init: Error: couldn't start player.\n");
		return (-1);
	}

	LOG_printf ("\tOK.\n");

	return (0);
}



void	PLAY_second_restore (void)
{
	if (PLAY_player_started_flag)
	{
		PLAY_set_play_mode (PLAY_MODE_STOP_ALL, -1, -1, false);
		PLAY_clear_all_effects ();
		PLAY_stop_player ();
	}
}



signed int	PLAY_set_replay_format (void)
{
	SoundDriver_CONFIG	config;
	int		out_stereo [GTK_NBROUT_MAXI];
	int		in_stereo [GTK_NBRIN_MAXI];
	int		track;
	int		global_track_nbr;
	PLAY_TRACK_INFO	*track_info_ptr;

	config.sample_freq = 44100;
	config.nbr_buffers = 2;
	config.nbr_out = GTK_nbr_tracks [GTK_TRACK_TYPE_AOU];
	config.nbr_in = GTK_nbr_tracks [Pattern_TYPE_AIN];
	config.out_stereo_ptr = out_stereo;
	config.in_stereo_ptr = in_stereo;
	for (track = 0; track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		out_stereo [track] = track_info_ptr->stereo;
	}
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_AIN]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_AIN] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		out_stereo [track] = track_info_ptr->stereo;
	}

	/* Fait evaluer la config */
	if (OS_sound_driver.set_config (&config))
	{
		LOG_printf ("PLAY_set_replay_format: Error: couldn't set configuration.\n");
		return (-1);
	}

	/* Reajuste les config des pistes en fonction de la config
		reelle renvoyee par le driver. */

	/*** A faire ***/

	/* Fabrique les buffers de mixage */
	MIX_create_mix_buffers ();

	return (0);
}



void	PLAY_player_thread_routine (void)
{
	int		track;
	SLWORD	clip_level;
	SLWORD	clip_level_array [GTK_NBROUT_MAXI] = { 0 };
	QWORD		tick_start_time = 0;
	QWORD		tick_duration = 0;
	QWORD		current_time;
	Thread_Message	message;

	/* On se fixe le niveau de priorite le plus haut */
	if (PLAY_player_thread.set_priority (Thread_PRIORITY_TIME_CRITICAL))
	{
		LOG_printf ("PLAY_player_thread_routine: Warning: couldn't set priority for player thread.\n");
	}

	if (OS_sound_driver.start_replay ())
	{
		LOG_printf ("PLAY_player_thread_routine: Error: couldn't start player.\n");
		PLAY_player_thread_state = -1;
		return;
	}

	PLAY_player_thread_state = 1;

	tick_start_time = OS_get_time ();
	message = Thread_MESSAGE_NONE;
	do
	{
		/* Gere la partition et fixe une nouvelle longueur de frame */
		PLAY_main_interrupt ();

		/* Fin du chronometrage du tick. */
		tick_duration = OS_get_time () - tick_start_time;

		/* Signale au driver la nouvelle frame et attend de pouvoir
			transferer les donnees. */
		OS_sound_driver.set_new_frame (MIX_frame_length);

		/* Calcul du temps machine utilise (debut du chrono) */
		current_time = OS_get_time ();
		if (current_time - tick_start_time > 0)
		{
			OS_cpu_time = (int)(((double)tick_duration * 100) / (current_time - tick_start_time));
		}
		tick_start_time = current_time;

		/* Recupere les donnees en entree */
		OS_sound_driver.get_data (MIX_driver_in_buffer_ptr);

		/* Effectue le mixage */
		MIX_do_mix ();

		/* Expedie au driver les donnees resultat */
		OS_sound_driver.set_data (MIX_driver_out_buffer_ptr, clip_level_array);

		/* Recupere les informations de saturation */
		clip_level = PLAY_get_max_clipping_level ();
		for (track = 0; track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; track ++)
		{
			PLAY_set_clipping_level (track, clip_level_array [track]);
			clip_level = MAX (clip_level, clip_level_array [track]);
		}
		PLAY_set_max_clipping_level (clip_level);

		/* Regarde si on nous demande d'arreter le replay */
		message = PLAY_player_thread.get_message ();
	}
	while (message != Thread_MESSAGE_QUIT);

	OS_sound_driver.stop_replay ();

	PLAY_player_thread_state = 0;
}



/*==========================================================================*/
/*      Nom: PLAY_start_player                                              */
/*      Description: Demarre le player, une fois que toutes les donnees     */
/*                   necessaires ont ete initialisees.                      */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	PLAY_start_player (void)
{
	if (! PLAY_player_started_flag)
	{
		/* Passe en mode Stop pour demarrer */
		PLAY_new_play_mode = PLAY_MODE_STOP_ALL;
		PLAY_new_play_mode_flag = true;

		PLAY_player_thread_state = 0;
		if (PLAY_player_thread.spawn ())
		{
			LOG_printf ("PLAY_start_player: Error: couldn't spawn player thread.\n");
			return (-1);
		}

		/* Attend que le thread ait fini son initialisation */
		while (PLAY_player_thread_state == 0)
		{
			Thread::sleep (10);
		}

		if (PLAY_player_thread_state < 0)
		{
			LOG_printf ("PLAY_start_player: Error: Problem in player thread.\n");
			while (PLAY_player_thread.is_running ())
			{
				Thread::sleep (10);
			}
			PLAY_player_thread_state = 0;
			return (-1);
		}

		PLAY_player_started_flag = true;

		/* On se fixe un niveau de priorite en-dessous de la normale. */
		if (OS_main_thread.set_priority (Thread_PRIORITY_BELOW_NORMAL))
		{
			LOG_printf ("PLAY_start_player: Warning: couldn't set Idle Priority for main thread.\n");
		}
	}

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_stop_player                                               */
/*      Description: Arrete le player, qui peut etre remis en marche plus   */
/*                   tard.                                                  */
/*==========================================================================*/

void	PLAY_stop_player (void)
{
	if (PLAY_player_started_flag)
	{
		PLAY_set_play_mode (PLAY_MODE_STOP_ALL, -1, -1, false);

		/* Arrete le thread du player */
		PLAY_player_thread.post_message (Thread_MESSAGE_QUIT);
		while (PLAY_player_thread.is_running ())
		{
			Thread::sleep (10);
		}

		PLAY_player_started_flag = false;

		/* On revient a un niveau de priorite normal */
		if (OS_main_thread.set_priority (Thread_PRIORITY_NORMAL))
		{
			LOG_printf ("PLAY_stop_player: Warning: couldn't set Normal Priority for main thread.\n");
		}
	}
}



/*==========================================================================*/
/*      Nom: PLAY_get_player_started_flag                                   */
/*      Description: Indique si le player est en marche ou pas.             */
/*      Retour: true si le player joue, false sinon.                        */
/*==========================================================================*/

bool	PLAY_get_player_started_flag (void)
{
	return (PLAY_player_started_flag);
}



/*==========================================================================*/
/*      Nom: PLAY_add_new_track                                             */
/*      Description: Ajoute de nouvelles pistes d'un type donne dans le     */
/*                   tableau des numeros de pistes absolus. Aucune modif    */
/*                   n'est faite au niveau de la partition, mais les        */
/*                   donnees des pistes sont initialisees. Cette routine    */
/*                   doit etre appelee avant de modifier GTK_nbr_tracks []. */
/*      Parametres en entree:                                               */
/*        - track_type: type des pistes a rajouter. Ce nombre peut etre     */
/*                      negatif si des pistes sont enlevees.                */
/*        - nbr_tracks: nombre de pistes a rajouter.                        */
/*==========================================================================*/

void	PLAY_add_new_track (int track_type, signed int nbr_tracks)
{
	int		global_track_nbr;
	int		track_type_2;
	int		track;
	int		nbr_tracks_2;
	signed int	track_count;
	bool		free_flag;
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

/*______________________________________________
 *
 * On retire des pistes
 *______________________________________________
 */

	if (nbr_tracks <= 0)
	{
		/* Pour les pistes de samples, il faut virer les filtres de
		   reconstruction et les buffers de Direct-To-Disk. */
		if (track_type == Pattern_TYPE_SPL)
		{
			/* Filtres de reconstruction */
			if (MIX_get_sound_quality () > MIX_SOUND_QUALITY_NORMAL)
			{
				for (track_count = nbr_tracks; track_count < 0; track_count ++)
				{
					global_track_nbr = PLAY_track_info_list [Pattern_TYPE_SPL]
					                                        [GTK_nbr_tracks [Pattern_TYPE_SPL] + track_count];
					track_info_ptr = PLAY_track_info + global_track_nbr;

					if (track_info_ptr->mix.spl.recons_filter_ptr != NULL)
					{
						delete track_info_ptr->mix.spl.recons_filter_ptr;
						track_info_ptr->mix.spl.recons_filter_ptr = NULL;
					}
				}
			}

			/* Buffers de D2D */
			for (track_count = nbr_tracks; track_count < 0; track_count ++)
			{
				D2D_free_d2d_cache_cell (GTK_nbr_tracks [Pattern_TYPE_SPL] + track_count);
			}
		}

		/* Pour les pistes d'effets, il faut virer proprement l'effet */
		else if  (track_type == Pattern_TYPE_SPL)
		{
			for (track_count = nbr_tracks; track_count < 0; track_count ++)
			{
				global_track_nbr = PLAY_track_info_list [Pattern_TYPE_SPL]
				                                        [GTK_nbr_tracks [Pattern_TYPE_SPL] + track_count];
				track_info_ptr = PLAY_track_info + global_track_nbr;

				if (   track_info_ptr->mix.fx.effect != FxPreset_TYPE_NONE
				    && ! track_info_ptr->mix.fx.bad_init_flag)
				{
					PLAYFX_fx_restore (track_info_ptr);
				}
			}
		}

		/* Il faut egalement virer ces pistes des presets de mixage,
			voire carrement virer les presets. */
		for (track_count = nbr_tracks; track_count < 0; track_count ++)
		{
			track = GTK_nbr_tracks [track_type] + track_count;
			MIXP_remove_track_from_all_presets (track_type, track);
		}

		GTK_mutex.signal ();
		return;
	}

/*______________________________________________
 *
 * On ajoute des pistes
 *______________________________________________
 */

	global_track_nbr = 0;

	for (track_count = 0; track_count < nbr_tracks; track_count ++)
	{
		/* Recherche un numero global de piste inutilise */
		for (;;)
		{
			free_flag = true;
			for (track_type_2 = 0; track_type_2 < GTK_NBR_MIX_TRACK_TYPE; track_type_2 ++)
			{
				nbr_tracks_2 = GTK_nbr_tracks [track_type_2];

				for (track = 0; track < nbr_tracks_2; track ++)
				{
					if (PLAY_track_info_list [track_type_2] [track] == global_track_nbr)
					{
						free_flag = false;
						break;
					}
				}

				if (! free_flag)
				{
					break;
				}
			}

			if (free_flag)
			{
				break;
			}

			global_track_nbr ++;
		}

		/* Affecte le numero a la piste */
		track = GTK_nbr_tracks [track_type] + track_count;
		PLAY_track_info_list [track_type] [track] = global_track_nbr;

		/* Initialise les info */
		track_info_ptr = PLAY_track_info + global_track_nbr;
		track_info_ptr->track = track;
		track_info_ptr->track_type = track_type;
		track_info_ptr->outvol = 0x1000;
		track_info_ptr->outpan = 0x8000U;
		track_info_ptr->dry_outvol = 0x1000;
		track_info_ptr->dry_outpan = 0x8000U;
		track_info_ptr->wet_mute_flag = false;
		track_info_ptr->dry_mute_flag = false;
		PLAY_set_track_name (track_type, track, "");
		PLAY_set_lin_volume (track_info_ptr, 0x1000);
		PLAY_set_dry_lin_volume (track_info_ptr, 0x1000);

		track_info_ptr->stereo = 1;
		track_info_ptr->mix.spl.sample_ptr = SAMP_get_sample_data_adr (0);
		track_info_ptr->mix.spl.loopbuf_ptr = SAMP_get_sample_loopbuf_adr (0);
		track_info_ptr->mix.spl.loopbuf_len = SAMP_get_sample_loopbuf (0);
		track_info_ptr->mix.spl.resol = SAMP_get_sample_resolution (0) >> 3;
		track_info_ptr->mix.spl.tracks = SAMP_get_sample_stereo (0);
		track_info_ptr->mix.spl.reppos = 0;
		track_info_ptr->mix.spl.replen = 0;
		track_info_ptr->mix.spl.curpos_int = 0;
		track_info_ptr->mix.spl.curpos_frac = 0;
		track_info_ptr->mix.spl.freq = SAMP_get_sample_freq (0);
		track_info_ptr->mix.spl.loopmode = 0;
		track_info_ptr->mix.spl.direction = 0;
		track_info_ptr->mix.spl.d2d_flag = false;

		track_info_ptr->mix.spl.final_freq = 10000.0;
		track_info_ptr->mix.spl.final_q = 0.5;
		for (int spl = 0; spl < 4*2; spl ++)
		{
			track_info_ptr->mix.spl.buffer_y [spl] = 0;
			track_info_ptr->mix.spl.buffer_x [spl] = 0;
		}
		track_info_ptr->mix.spl.buffer_pos = 0;

		switch (track_type)
		{
		case	Pattern_TYPE_SPL:
			if (MIX_get_sound_quality () > MIX_SOUND_QUALITY_NORMAL)
			{
				track_info_ptr->mix.spl.recons_filter_ptr = new ReconstructionFilter (MIX_recons_filter_width);
				if (track_info_ptr->mix.spl.recons_filter_ptr == NULL)
				{
					LOG_printf ("PLAY_add_new_track: Warning: couldn't create a reconstruction filter for Sample track # %d.\n",
					            track);
				}
				else if (track_info_ptr->mix.spl.recons_filter_ptr->check_ok ())
				{
					LOG_printf ("PLAY_add_new_track: Warning: couldn't initialize reconstruction filter for Sample track # %d.\n",
					            track);
					delete track_info_ptr->mix.spl.recons_filter_ptr;
					track_info_ptr->mix.spl.recons_filter_ptr = NULL;
				}
			}
			else
			{
				track_info_ptr->mix.spl.recons_filter_ptr = NULL;
			}
			break;

		case	Pattern_TYPE_AIN:
			/*** A voir ***/
			track_info_ptr->stereo = 2;
			break;

		case	Pattern_TYPE_FX:
			track_info_ptr->stereo = 1;
			track_info_ptr->mix.fx.old_effect = FxPreset_TYPE_NONE;
			track_info_ptr->mix.fx.effect = FxPreset_TYPE_NONE;
			track_info_ptr->mix.fx.bypass_flag = false;
			track_info_ptr->mix.fx.update_flag = false;
			track_info_ptr->mix.fx.bad_init_flag = false;

			track_info_ptr->mix.src.nbr_s_tracks = 1;
			track_info_ptr->mix.src.track_conf [0].track_type = Pattern_TYPE_SPL;
			track_info_ptr->mix.src.track_conf [0].wet_flag = true;
			track_info_ptr->mix.src.track_conf [0].cr.old_vol [0] = 0;
			track_info_ptr->mix.src.track_conf [0].cr.old_vol [1] = 0;
			if (track < GTK_nbr_tracks [Pattern_TYPE_SPL])
			{
				track_info_ptr->mix.src.track_conf [0].track_nbr = track;
			}
			else
			{
				track_info_ptr->mix.src.track_conf [0].track_nbr = 0;
			}
			track_info_ptr->mix.src.track_conf [0].inpvol = 0x1000;
			track_info_ptr->mix.src.track_conf [0].inppan = 0x8000U;
			break;

		case	Pattern_TYPE_MID:

			/*** A faire ***/

			break;
		}
		PLAY_reset_track_partition_info (track_info_ptr);
		track_info_ptr->score.track.panning = 0x8000U;
		track_info_ptr->score.track.drypan = 0x8000U;

		global_track_nbr ++;
	}

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_set_replay_freq                                           */
/*      Description: Fixe une nouvelle frequence de replay. Ne doit etre    */
/*                   appele que quand la deuxieme initialisation est        */
/*                   terminee.                                              */
/*      Parametres en entree:                                               */
/*        - freq: la nouvelle frequence, en Hz.                             */
/*==========================================================================*/

void	PLAY_set_replay_freq (long freq)
{
	int		track;
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	MIX_replay_freq = freq;

	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_FX]; track ++)
	{
		track_info_ptr =   PLAY_track_info
		                 + PLAY_track_info_list [Pattern_TYPE_FX] [track];
		PLAYFX_fx_silence (track_info_ptr);
		PLAYFX_fx_update (track_info_ptr);
	}

	SimpleCalculator::set_spec_var (SimpleCalculator::SAMPLE_FREQ,
	                                static_cast <double> (freq));

	PLAY_elapsed_time = 0;

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_main_interrupt                                            */
/*      Description: Routine appelee par l'interruption du driver de la     */
/*                   carte son. Elle doit etre placee avant la routine de   */
/*                   mixage car elle definit la nouvelle longueur de la     */
/*                   frame.                                                 */
/*==========================================================================*/

void	PLAY_main_interrupt (void)
{
	int		frame;
	int		nbr_frames;
	LWORD		frame_len;
	double	tick_len_dbl;

	/* On s'isole */
	GTK_mutex.wait ();

/*______________________________________________
 *
 * Changement de frame
 *______________________________________________
 */

	PLAY_score_info.frame_pos_int += PLAY_score_info.frame_len;
	nbr_frames = PLAY_score_info.nbr_frames;
	frame = PLAY_score_info.frame_counter;
	frame ++;

	/* Nouveau tick */
	if (frame >= nbr_frames)
	{
		frame = 0;

		/* Reinitialise la position de la premiere frame */
		PLAY_score_info.frame_pos_int =
			BASE_SUBX (ULWORD, PLAY_score_info.frame_pos_int,
									 PLAY_score_info.frame_pos_frac,
									 PLAY_score_info.tick_len_int,
									 PLAY_score_info.tick_len_frac);
		PLAY_score_info.frame_pos_frac -= PLAY_score_info.tick_len_frac;

		/* Gestion de la partition */
		PLAY_handle_score ();

		/* Calcul de la nouvelle duree d'un tick
			tick_len = replay_freq * 60 / tempo / 4 / 6
						= (replay_freq * 5) / (tempo * 2) */
		tick_len_dbl =   MIX_replay_freq * 2.5
		               / (  PLAY_score_info.tempo_int
		                  + PLAY_score_info.tempo_frac * (1.0 / 65536));
		PLAY_score_info.tick_len_frac = (ULWORD) (  modf (tick_len_dbl, &tick_len_dbl)
		                                          * 4294967296.0);
		PLAY_score_info.tick_len_int = (LWORD) tick_len_dbl;

		/* Calcul du nouveau nombre de frames/tick. On ajoute 1 car il
			faut arrondir par exces et tenir compte du fait qu'avec la
			partie frac, tick_len_int <= tick_len < tick_len_int + 1 */
		nbr_frames = (int)(PLAY_score_info.tick_len_int / MIX_MAX_FRAME_LENGTH) + 1;
		PLAY_score_info.nbr_frames = nbr_frames;
	}

	PLAY_score_info.frame_counter = frame;

	/* Calcule la duree de cette nouvelle frame */
	if (frame == nbr_frames - 1)
	{
		frame_len = PLAY_score_info.tick_len_int - PLAY_score_info.frame_pos_int;
		/* Reequilibre la derniere frame de facon a ce que la totale
			dure tres legerement plus longtemps (0 <= d < +1 sample) que
			veritable duree du tick, ceci afin que le prochain tick
			redemarre avec une position >= 0. */
		if (PLAY_score_info.frame_pos_frac < PLAY_score_info.tick_len_frac)
		{
			frame_len ++;
		}
	}
	else
	{
		frame_len =   (  PLAY_score_info.tick_len_int - PLAY_score_info.frame_pos_int
							+ nbr_frames - frame - 1)
						/ (nbr_frames - frame);
	}
	PLAY_score_info.frame_len = frame_len;

	// Clock
	if (PLAY_score_info.current_play_mode != PLAY_MODE_STOP)
	{
		PLAY_elapsed_time += frame_len;
	}

/*______________________________________________
 *
 * Prise en compte des nouvelles donnees
 * introduites par le tracker
 *______________________________________________
 */

	if (PLAY_new_line_flag [Pattern_TYPE_SPL] != 0)
	{
		PLAY_play_pattern_line_spl ();
	}

	if (PLAY_new_line_flag [Pattern_TYPE_AIN] != 0)
	{
		PLAY_play_pattern_line_ain ();
	}

	if (PLAY_new_line_flag [Pattern_TYPE_FX] != 0)
	{
		PLAY_play_pattern_line_fx ();
	}

	if (PLAY_new_line_flag [Pattern_TYPE_MID] != 0)
	{
		PLAY_play_pattern_line_mid ();
	}

	/* Nouvelle longueur officielle */
	MIX_frame_length = frame_len;

	/* On peut autoriser quelqu'un d'autre a utiliser les donees */
	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_handle_score                                              */
/*      Description: Gestion generale de la partition.                      */
/*==========================================================================*/

void	PLAY_handle_score (void)
{
	PLAY_score_info.tick_counter ++;

	/* Changement de mode de replay */
	if (PLAY_new_play_mode_flag)
	{
		if (   ! PLAY_new_play_mode_cont_flag
			 || PLAY_score_info.current_play_mode == PLAY_MODE_STOP)
		{
			PLAY_score_info.tick_counter = PLAY_score_info.speed;
			PLAY_score_info.pattern_delay = 0;
		}

		/* Recupere ligne et position courante */
		if (PLAY_score_info.current_play_mode != PLAY_MODE_STOP)
		{
			PLAY_current_linepos = PLAY_score_info.current_line;
		}
		if (PLAY_score_info.current_play_mode == PLAY_MODE_SONG)
		{
			PLAY_current_songpos = PLAY_score_info.current_pos;
		}

		/* Met a jour ces positions avec celles demandees */
		if (PLAY_new_songpos >= 0)
		{
			PLAY_current_songpos = PLAY_new_songpos;
		}
		if (   PLAY_new_linepos >= 0
			 && PLAY_new_play_mode_cont_flag)
		{
			PLAY_current_linepos = PLAY_new_linepos;
		}

		switch (PLAY_new_play_mode)
		{
		/* Stop/Direct */
		case	PLAY_MODE_STOP_ALL:
		case	PLAY_MODE_STOP:
			if (PLAY_new_play_mode == PLAY_MODE_STOP_ALL)
			{
				PLAY_score_info.frame_pos_int = 0;
				PLAY_score_info.frame_pos_frac = 0;
				PLAY_reset_all_tracks ();
			}
			PLAY_score_info.tick_counter = PLAY_score_info.speed;
			PLAY_score_info.pattern_delay = 0;
			PLAY_score_info.next_pos = 0;
			PLAY_score_info.next_line = 0;
			PLAY_score_info.next_flag = true;
			PLAY_score_info.current_play_mode = PLAY_MODE_STOP;
			break;

		/* Play Song */
		case	PLAY_MODE_SONG:
			if (PLAY_score_info.current_play_mode == PLAY_MODE_STOP)
			{
				PLAY_score_info.next_pos = PLAY_current_songpos;
				PLAY_score_info.next_line = PLAY_current_linepos;
				PLAY_score_info.next_flag = true;
			}
			else
			{
				PLAY_score_info.current_pos = PLAY_current_songpos;
				PLAY_score_info.current_pattern = SONG_get_pattern_number (PLAY_current_songpos);
				PLAY_score_info.current_line = PLAY_current_linepos;
			}

			if (! PLAY_new_play_mode_cont_flag)
			{
				PLAY_score_info.next_pos = PLAY_current_songpos;
				PLAY_score_info.next_line = 0;
				PLAY_score_info.next_flag = true;
			}

			PLAY_score_info.current_play_mode = PLAY_MODE_SONG;
			PLAY_elapsed_time = 0;
			break;

		/* Play_pattern */
		case	PLAY_MODE_PATTERN:
			PLAY_score_info.current_pattern = SONG_get_pattern_number (PLAY_current_songpos);

			if (PLAY_score_info.current_play_mode == PLAY_MODE_STOP)
			{
				PLAY_score_info.next_pos = PLAY_current_songpos;
				PLAY_score_info.next_line = PLAY_current_linepos;
				PLAY_score_info.next_flag = true;
			}
			else
			{
				PLAY_score_info.current_pos = PLAY_current_songpos;
				PLAY_score_info.current_line = PLAY_current_linepos;
			}

			if (! PLAY_new_play_mode_cont_flag)
			{
				PLAY_score_info.next_pos = PLAY_current_songpos;
				PLAY_score_info.next_line = 0;
				PLAY_score_info.next_flag = true;
			}
			PLAY_score_info.current_play_mode = PLAY_MODE_PATTERN;
			PLAY_elapsed_time = 0;
			break;
		}

		PLAY_new_play_mode_flag = false;
	}

	/* Passage a la ligne suivante */
	if (   PLAY_score_info.tick_counter
		 >= PLAY_score_info.speed * (PLAY_score_info.pattern_delay + 1))
	{
		PLAY_score_info.tick_counter = 0;
		PLAY_score_info.pattern_delay = 0;

		/* Deroulement devie de sa trajectoire habituelle */
		if (PLAY_score_info.next_flag)
		{
			PLAY_score_info.current_line = PLAY_score_info.next_line;
			PLAY_score_info.next_flag = false;
			PLAY_score_info.next_line = 0;
			PLAY_score_info.current_pos = PLAY_score_info.next_pos;
			switch (PLAY_score_info.current_play_mode)
			{
			case	PLAY_MODE_SONG:
			case	PLAY_MODE_PATTERN:
				PLAY_score_info.current_pattern = SONG_get_pattern_number (PLAY_score_info.current_pos);
				break;
			case	PLAY_MODE_STOP:
				if (PLAY_score_info.current_pos <= 0)
				{
					PLAY_score_info.current_pattern = GTK_PATTERN_BIDON_A;
				}
				else
				{
					PLAY_score_info.current_pattern = GTK_PATTERN_BIDON_B;
				}
				break;
			}
		}

		/* Ligne suivante dans la partition */
		else
		{
			PLAY_score_info.current_line ++;
		}

		/* Teste si on est toujours dans le pattern et la song */
		if (PLAY_score_info.current_line >= PAT_get_pattern_height (PLAY_score_info.current_pattern))
		{
			PLAY_score_info.current_line = 0;
			PLAY_score_info.current_pos ++;

			switch (PLAY_score_info.current_play_mode)
			{
			case	PLAY_MODE_SONG:
				if (PLAY_score_info.current_pos >= SONG_get_song_length ())
				{
					PLAY_score_info.current_pos = SONG_get_song_repeat ();
				}
				PLAY_score_info.current_pattern = SONG_get_pattern_number (PLAY_score_info.current_pos);
				break;
			case	PLAY_MODE_STOP:
				if (PLAY_score_info.current_pos <= 0)
				{
					PLAY_score_info.current_pattern = GTK_PATTERN_BIDON_A;
				}
				else
				{
					PLAY_score_info.current_pattern = GTK_PATTERN_BIDON_B;
				}
				break;
			}
		}

		/* Recopie toute la ligne pointee par current_line dans les info des pistes */
		PLAY_copy_line_to_track_info ();
	}

	if (PLAY_score_info.tick_counter == 0)
	{
		PLAY_handle_effects_1st_tick ();
	}
	PLAY_handle_effects_all_ticks ();
	PLAYENV_manage_all_envelopes ();
	PLAY_compute_final_parameters ();
}



/*==========================================================================*/
/*      Nom: PLAY_copy_line_to_track_info                                   */
/*      Description: Copie la ligne courante du pattern courant dans les    */
/*                   blocs d'informations sur les pistes.                   */
/*      Parametres en entree:                                               */
/*        - score_info_ptr: pointeur sur les info de la partition.          */
/*==========================================================================*/

void	PLAY_copy_line_to_track_info (void)
{
	int					track;
	int					track_max;
	PLAY_TRACK_INFO	*track_info_ptr;
	UBYTE					*note_ptr;
	int					*track_info_list_ptr;

	/* Samples */
	note_ptr = (UBYTE *) PAT_get_spl_note_adr_pat (PLAY_score_info.current_pattern,
	                                               PLAY_score_info.current_line, 0);
	track_max = GTK_nbr_tracks [Pattern_TYPE_SPL];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_SPL];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		track_info_ptr->score.score.note = *note_ptr;
		track_info_ptr->score.score.instr = *(note_ptr + 1);
		track_info_ptr->score.score.effect = BASE_dpeek_moto (note_ptr + 2);
		track_info_ptr->score.score.volume = *(note_ptr + 4);
		note_ptr += MODS_GT2_SPL_NOTE_LEN;
	}

	/* Audio in */
	note_ptr = (UBYTE *) PAT_get_ain_note_adr_pat (PLAY_score_info.current_pattern,
	                                               PLAY_score_info.current_line, 0);
	track_max = GTK_nbr_tracks [Pattern_TYPE_AIN];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_AIN];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		track_info_ptr->score.score.effect = BASE_dpeek_moto (note_ptr);
		track_info_ptr->score.score.volume = *(note_ptr + 2);
		note_ptr += MODS_GT2_AIN_NOTE_LEN;
	}

	/* Effets */
	note_ptr = (UBYTE *) PAT_get_fx_note_adr_pat (PLAY_score_info.current_pattern,
	                                              PLAY_score_info.current_line, 0);
	track_max = GTK_nbr_tracks [Pattern_TYPE_FX];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_FX];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		track_info_ptr->score.score.fx_data = BASE_lpeek_moto (note_ptr);
		track_info_ptr->score.score.effect = BASE_dpeek_moto (note_ptr + 4);
		track_info_ptr->score.score.volume = *(note_ptr + 6);
		note_ptr += MODS_GT2_FX_NOTE_LEN;
	}

	/* MIDI */
	note_ptr = (UBYTE *) PAT_get_mid_note_adr_pat (PLAY_score_info.current_pattern,
	                                               PLAY_score_info.current_line, 0);
	track_max = GTK_nbr_tracks [Pattern_TYPE_MID];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_MID];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		track_info_ptr->score.score.mid_data = BASE_lpeek_moto (note_ptr);
		note_ptr += MODS_GT2_MID_NOTE_LEN;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_handle_effects_1st_tick												 */
/*      Description: Gere les effets, notes et parametres qui n'agissent    */
/*                   qu'au premier tick d'une ligne.                        */
/*==========================================================================*/

void	PLAY_handle_effects_1st_tick (void)
{
	int							track;
	int							track_max;

	/* Samples */
	track_max = GTK_nbr_tracks [Pattern_TYPE_SPL];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_1st_tick_spl (track);
	}

	/* Audio In */
	track_max = GTK_nbr_tracks [Pattern_TYPE_AIN];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_1st_tick_ain (track);
	}

	/* Effets */
	track_max = GTK_nbr_tracks [Pattern_TYPE_FX];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_1st_tick_fx (track);
	}

	/* MIDI */
	track_max = GTK_nbr_tracks [Pattern_TYPE_MID];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_1st_tick_mid (track);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_handle_effects_all_ticks                                  */
/*      Description: Gere les effets qui agissent pendant toute la duree de */
/*                   le ligne.                                              */
/*==========================================================================*/

void	PLAY_handle_effects_all_ticks (void)
{
	int							track;
	int							track_max;

	/* Samples */
	track_max = GTK_nbr_tracks [Pattern_TYPE_SPL];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_all_ticks_spl (track);
	}

	/* Audio In */
	track_max = GTK_nbr_tracks [Pattern_TYPE_AIN];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_all_ticks_ain (track);
	}

	/* Effets */
	track_max = GTK_nbr_tracks [Pattern_TYPE_FX];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_all_ticks_fx (track);
	}

	/* MIDI */
	track_max = GTK_nbr_tracks [Pattern_TYPE_MID];
	for (track = 0; track < track_max; track ++)
	{
		PLAYCMD_manage_effects_all_ticks_mid (track);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_compute_final_parameters                                  */
/*      Description: Calcule les parametres de mixage finaux a partir des   */
/*                   donnees fournies par chaque piste.                     */
/*==========================================================================*/

void	PLAY_compute_final_parameters (void)
{
	int							track;
	int							track_max;

	/* Samples */
	track_max = GTK_nbr_tracks [Pattern_TYPE_SPL];
	for (track = 0; track < track_max; track ++)
	{
		PLAY_compute_final_parameters_track (Pattern_TYPE_SPL, track);
	}

	/* Audio In */
	track_max = GTK_nbr_tracks [Pattern_TYPE_AIN];
	for (track = 0; track < track_max; track ++)
	{
		PLAY_compute_final_parameters_track (Pattern_TYPE_AIN, track);
	}

	/* Effets */
	track_max = GTK_nbr_tracks [Pattern_TYPE_FX];
	for (track = 0; track < track_max; track ++)
	{
		PLAY_compute_final_parameters_track (Pattern_TYPE_FX, track);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_compute_final_parameters_track                            */
/*      Description: Calcule les parametres de mixage finaux d'une piste a  */
/*                   partir des donnees de la partition.                    */
/*      Parametres en entree:                                               */
/*        - track_type: type de la piste.                                   */
/*        - track: numero de la piste a calculer.                           */
/*==========================================================================*/

void	PLAY_compute_final_parameters_track (int track_type, int track)
{
	ULWORD	volume;
	LWORD		panning;
	float		old_period;
	float		period;
	SLWORD	pan_env;
	SLWORD	temp;
	double	note_freq;
	double	base_freq;
	double	min_vol_freq;
	double	base_q;
	double	min_vol_q;
	double	k;
	double	c;
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr = PLAY_track_info + PLAY_track_info_list [track_type] [track];

/*______________________________________________
 *
 * Volume
 *______________________________________________
 */

	if (   track_type == Pattern_TYPE_SPL
	    || track_type == Pattern_TYPE_AIN
		 || track_type == Pattern_TYPE_FX)
	{
		if (track_info_ptr->score.changed_volume_flag)
		{
			volume = track_info_ptr->score.sample.volume;	/* 0...100...FFF */
			volume *= track_info_ptr->score.velocity_loc;	/* 0...80000...7FF800 */
			if (track_info_ptr->score.com_env [Envelope_TYPE_VOL].nbr != 0)
			{
				volume >>= 8;	/* 0...800...7FF8 */
				volume *= track_info_ptr->score.com_env [Envelope_TYPE_VOL].proc.final_val;	/* 0...800000...3FFB8008 */
				volume >>= 15;	/* 0...100...7FF7 */
				volume = MIN (volume, 0xFFF);	/* 0...100...FFF */
			}
			else
			{
				volume >>= 11;	/* 0...100...FFF */
			}

			volume *= track_info_ptr->score.track.volume_lin >> 4;
			track_info_ptr->outvol = (UWORD)(volume >> 4);
		}

/*______________________________________________
 *
 * Balance
 *______________________________________________
 */

		panning = track_info_ptr->score.track.panning;	/* 0...8000...FFFF */
		if (track_info_ptr->score.com_env [Envelope_TYPE_PAN].nbr != 0)
		{
			temp = panning;
			if (temp > 0x8000)
			{
				temp = 0x10000 - temp;	/* 0...8000 */
			}
			pan_env = track_info_ptr->score.com_env [Envelope_TYPE_PAN].proc.final_val;	/* -80...0...7F */
			pan_env *= temp;		/* -400000...0...3FFF80 */
			pan_env /= 0x80;	/* -8000...0...8000 */
			panning += pan_env;	/* 0...8000...10000 */
			panning = MIN (panning, 0xFFFF);
		}
		track_info_ptr->outpan = (UWORD)panning;
	}

/*______________________________________________
 *
 * Periode
 *______________________________________________
 */

	if (   track_type == Pattern_TYPE_SPL
		 || track_type == Pattern_TYPE_FX)
	{
		old_period = track_info_ptr->mix.spl.outperiod;
		period = track_info_ptr->score.period_loc;
		if (track_info_ptr->score.com_env [Envelope_TYPE_TON].nbr != 0)
		{
			if (PLAY_get_linear_period_flag ())
			{
				period -= (float) (track_info_ptr->score.com_env [Envelope_TYPE_TON].proc.final_val * PLAY_LIN_PER_NOTE * 0.01);
			}
			else
			{
				period = (float) (period * exp (track_info_ptr->score.com_env [Envelope_TYPE_TON].proc.final_val * (-LN2 / 1200)));
			}
		}
		period = MIN (period, PLAY_MAX_PERIOD);
		period = MAX (period, PLAY_MIN_PERIOD);
		track_info_ptr->mix.spl.outperiod = period;
	}

/*______________________________________________
 *
 * Filtre passe-bas
 *______________________________________________
 */

	if (   track_type == Pattern_TYPE_SPL
	    && track_info_ptr->score.lpf.filter_flag)
	{
		if (   track_info_ptr->score.changed_filter_flag
		    || (   track_info_ptr->score.changed_volume_flag
		        && (   track_info_ptr->score.lpf.freq_vol_flag
		            || track_info_ptr->score.lpf.q_vol_flag))
		    || (   period != old_period
		        && (   track_info_ptr->score.lpf.freq [0] <= 20.0
		            || track_info_ptr->score.lpf.freq [1] <= 20.0)))
		{
			note_freq = PLAY_convert_period_to_freq (period);

			/* Calcule la frequence de base */
			base_freq = track_info_ptr->score.lpf.freq [0];
			if (base_freq <= 20.0)
			{
				base_freq *= note_freq;
			}

			/* Ajuste si necessaire la frequence sur la velocite */
			if (track_info_ptr->score.lpf.freq_vol_flag)
			{
				min_vol_freq = track_info_ptr->score.lpf.freq [1];
				if (min_vol_freq <= 20.0)
				{
					min_vol_freq *= note_freq;
				}

				base_freq =   min_vol_freq
				            +   (base_freq - min_vol_freq)
				              * track_info_ptr->score.velocity_loc
				              / 0x800;
			}

			/* Resonnance de base */
			base_q = track_info_ptr->score.lpf.q [0];

			/* Ajuste si necessaire la resonnance sur la velocite */
			if (track_info_ptr->score.lpf.q_vol_flag)
			{
				min_vol_q = track_info_ptr->score.lpf.q [1];
				base_q =   min_vol_q
				         +   (base_q - min_vol_q)
				           * track_info_ptr->score.velocity_loc
				           / 0x800;
			}

			/* Tient en compte les enveloppes */
			if (track_info_ptr->score.com_env [Envelope_TYPE_CUTOFF].nbr > 0)
			{
				base_freq *= exp (track_info_ptr->score.com_env [Envelope_TYPE_CUTOFF].proc.final_val * LN2 / 1200);
			}
			if (track_info_ptr->score.com_env [Envelope_TYPE_RESO].nbr > 0)
			{
				base_q *= exp (track_info_ptr->score.com_env [Envelope_TYPE_RESO].proc.final_val * LN10 / 2000);
			}

			/* Limitations */
			base_freq = MAX (base_freq, 20.0);
			base_q = MIN (base_q, 100.0);
			base_q = MAX (base_q, 0.5);

			/* Si on a change les valeurs du filtre depuis la derniere fois, on recalcule les coefficients */
			if (   base_freq != track_info_ptr->mix.spl.final_freq
			    || base_q != track_info_ptr->mix.spl.final_q)
			{
				if (base_freq * 2.2 > MIX_replay_freq)
				{
					track_info_ptr->mix.spl.coef_x [0] = 1;
					track_info_ptr->mix.spl.coef_x [1] = 0;
					track_info_ptr->mix.spl.coef_x [2] = 0;
					track_info_ptr->mix.spl.coef_y [1] = 0;
					track_info_ptr->mix.spl.coef_y [2] = 0;
				}

				else
				{
					k = 1.0 / tan (PI * base_freq / MIX_replay_freq);
					c = base_q / (k * (base_q * k + 1) + base_q);
					track_info_ptr->mix.spl.coef_x [0] = c;
					track_info_ptr->mix.spl.coef_x [1] = c * 2;
					track_info_ptr->mix.spl.coef_x [2] = c;
					track_info_ptr->mix.spl.coef_y [1] = c * (1.0 - k*k) * 2;
					track_info_ptr->mix.spl.coef_y [2] = c * (k*k - k / base_q + 1);
				}
			}

			track_info_ptr->mix.spl.final_freq = base_freq;
			track_info_ptr->mix.spl.final_q = base_q;
		}
	}

/*______________________________________________
 *
 * Effets (pour les pistes d'effet)
 *______________________________________________
 */

	if (track_type == Pattern_TYPE_FX)
	{
		PLAYFX_manage_all_ticks (track_info_ptr);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_reset_all_tracks                                          */
/*      Description: Remet toutes les voies a 0 et coupe toute source       */
/*                   sonore. La configuration des pistes d'effet est        */
/*                   toutefois gardee.                                      */
/*==========================================================================*/

void	PLAY_reset_all_tracks (void)
{
	int					track;
	int					track_max;
	PLAY_TRACK_INFO	*track_info_ptr;
	int					*track_info_list_ptr;

/*______________________________________________
 *
 * Samples
 *______________________________________________
 */

	track_max = GTK_nbr_tracks [Pattern_TYPE_SPL];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_SPL];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;

		/* Parametres de mixage */
		track_info_ptr->mix.spl.sample_ptr = SAMP_get_sample_data_adr (0);
		track_info_ptr->mix.spl.loopbuf_ptr = SAMP_get_sample_loopbuf_adr (0);
		track_info_ptr->mix.spl.loopbuf_len = SAMP_get_sample_loopbuf (0);
		track_info_ptr->mix.spl.resol = SAMP_get_sample_resolution (0) >> 3;
		track_info_ptr->mix.spl.tracks = SAMP_get_sample_stereo (0);
		track_info_ptr->mix.spl.reppos = 0;
		track_info_ptr->mix.spl.replen = 0;
		track_info_ptr->mix.spl.curpos_int = 0;
		track_info_ptr->mix.spl.curpos_frac = 0;
		track_info_ptr->mix.spl.freq = SAMP_get_sample_freq (0);
		track_info_ptr->mix.spl.loopmode = 0;
		track_info_ptr->mix.spl.direction = 0;
		track_info_ptr->mix.spl.d2d_flag = false;

		PLAY_reset_click_removal (track_info_ptr);

		/* Filtres de reconstruction */
		if (track_info_ptr->mix.spl.recons_filter_ptr != NULL)
		{
			track_info_ptr->mix.spl.recons_filter_ptr->clear_buffer ();
		}

		/* Filtres resonnant */
		for (int spl = 0; spl < 4*2; spl ++)
		{
			track_info_ptr->mix.spl.buffer_x [spl] = 0;
			track_info_ptr->mix.spl.buffer_y [spl] = 0;
		}
		track_info_ptr->mix.spl.buffer_pos = 0;

		/* Parametres de partition */
		PLAY_reset_track_partition_info (track_info_ptr);
	}

/*______________________________________________
 *
 * Audio In
 *______________________________________________
 */

	track_max = GTK_nbr_tracks [Pattern_TYPE_AIN];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_AIN];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		PLAY_reset_track_partition_info (track_info_ptr);
	}

/*______________________________________________
 *
 * Effets
 *______________________________________________
 */

	track_max = GTK_nbr_tracks [Pattern_TYPE_FX];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_FX];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;

		/* Parametres de mixage de la partie sample */
		track_info_ptr->mix.spl.sample_ptr = SAMP_get_sample_data_adr (0);
		track_info_ptr->mix.spl.loopbuf_ptr = SAMP_get_sample_loopbuf_adr (0);
		track_info_ptr->mix.spl.loopbuf_len = SAMP_get_sample_loopbuf (0);
		track_info_ptr->mix.spl.resol = SAMP_get_sample_resolution (0) >> 3;
		track_info_ptr->mix.spl.tracks = SAMP_get_sample_stereo (0);
		track_info_ptr->mix.spl.reppos = 0;
		track_info_ptr->mix.spl.replen = 0;
		track_info_ptr->mix.spl.curpos_int = 0;
		track_info_ptr->mix.spl.curpos_frac = 0;
		track_info_ptr->mix.spl.freq = SAMP_get_sample_freq (0);
		track_info_ptr->mix.spl.loopmode = 0;
		track_info_ptr->mix.spl.direction = 0;
		track_info_ptr->mix.spl.d2d_flag = false;

		PLAY_reset_click_removal (track_info_ptr);

		/* Parametres de partition */
		PLAY_reset_track_partition_info (track_info_ptr);

		if (   track_info_ptr->mix.fx.effect != FxPreset_TYPE_NONE
		    && ! track_info_ptr->mix.fx.bad_init_flag)
		{
			/* Vide les buffers pour que l'effet ne produise que du silence
				avec une entree nulle. */
			PLAYFX_fx_silence (track_info_ptr);
		}
	}

/*______________________________________________
 *
 * MIDI
 *______________________________________________
 */

	track_max = GTK_nbr_tracks [Pattern_TYPE_MID];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_MID];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;

		PLAY_reset_click_removal (track_info_ptr);

		PLAY_reset_track_partition_info (track_info_ptr);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_reset_track_partition_info                                */
/*      Description: Met a 0 la partie partition d'une piste.               */
/*==========================================================================*/

void	PLAY_reset_track_partition_info (PLAY_TRACK_INFO *track_info_ptr)
{
	int		env_type;
	int		cnt;

	track_info_ptr->score.score.note = 0;
	track_info_ptr->score.score.instr = 0;
	track_info_ptr->score.score.effect = 0;
	track_info_ptr->score.score.effect_par = 0;
	track_info_ptr->score.score.volume = 0;
	track_info_ptr->score.score.fx_data = 0;
	track_info_ptr->score.score.fx_par = 0;
	track_info_ptr->score.score.mid_data = 0;
	track_info_ptr->score.instr = 0;
	track_info_ptr->score.note = Sample_REF_C2;
	PLAY_set_lin_velocity (track_info_ptr, 0x800);
	track_info_ptr->score.velocity_loc = track_info_ptr->score.velocity_lin;
	track_info_ptr->score.period = (float) PLAY_convert_note_to_internal_period (Sample_REF_C2, 0);
	track_info_ptr->score.changed_volume_flag = true;
	track_info_ptr->score.changed_filter_flag = true;
	track_info_ptr->score.track.lin_vol_speed = 0;
	track_info_ptr->score.track.lin_vol_fine_speed = 0;
	track_info_ptr->score.track.log_vol_speed = 0;
	track_info_ptr->score.track.log_vol_fine_speed = 0;
	track_info_ptr->score.sample.number = 0;
	track_info_ptr->score.sample.volume = 0x100;
	track_info_ptr->score.sample.transp = 0;
	track_info_ptr->score.sample.finetune = 0;
	track_info_ptr->score.offset.pos_int = 0;
	track_info_ptr->score.offset.pos_frac = 0;
	track_info_ptr->score.arpeggio_cnt = 0;
	track_info_ptr->score.porta.note = Sample_REF_C2;
	track_info_ptr->score.porta.period = (float) PLAY_convert_note_to_internal_period (Sample_REF_C2, 0);
	track_info_ptr->score.porta.speed = 0;
	track_info_ptr->score.porta.fine_speed = 0;
	track_info_ptr->score.vibrato.speed = 0;
	track_info_ptr->score.vibrato.cnt = 0;
	track_info_ptr->score.vibrato.amp = 0;
	track_info_ptr->score.vibrato.waveform = 0;
	track_info_ptr->score.tremolo.speed = 0;
	track_info_ptr->score.tremolo.cnt = 0;
	track_info_ptr->score.tremolo.amp = 0;
	track_info_ptr->score.tremolo.waveform = 0;
	track_info_ptr->score.tremor.cnt = 0;
	track_info_ptr->score.tremor.on = 3;
	track_info_ptr->score.tremor.period = 6;
	track_info_ptr->score.roll.speed = 3;
	track_info_ptr->score.roll.cnt = 0;
	track_info_ptr->score.roll.nbr = 0;
	track_info_ptr->score.delay.play = 0;
	track_info_ptr->score.delay.cut = 0;
	track_info_ptr->score.ploop.songpos = -1;
	track_info_ptr->score.ploop.linepos = 0;
	track_info_ptr->score.ploop.cnt = 0;
	track_info_ptr->score.vol_slide.speed = 0;
	track_info_ptr->score.vol_slide.fine_speed = 0;
	track_info_ptr->score.pan_slide.speed = 0;
	track_info_ptr->score.lpf.filter_flag = false;
	track_info_ptr->score.lpf.freq_vol_flag = false;
	track_info_ptr->score.lpf.q_vol_flag = false;
	track_info_ptr->score.lpf.freq [0] = 10000.0;
	track_info_ptr->score.lpf.freq [1] = 1000.0;
	track_info_ptr->score.lpf.q [0] = 0.5;
	track_info_ptr->score.lpf.q [1] = 0.5;
	for (env_type = 0; env_type < Envelope_NBR_TYPES; env_type ++)
	{
		track_info_ptr->score.com_env [env_type].nbr = 0;
	}

	for (cnt = 0; cnt < PLAY_FX_NBR_TIME_SPEED; cnt ++)
	{
		track_info_ptr->score.fx.time [cnt].speed = 0.0;
		track_info_ptr->score.fx.time [cnt].time = 1.0;
	}

	for (cnt = 0; cnt < PLAY_FX_NBR_FREQ_SPEED; cnt ++)
	{
		track_info_ptr->score.fx.freq [cnt].speed = 0.0;
		track_info_ptr->score.fx.freq [cnt].freq = 440.0;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_reset_click_removal                                       */
/*      Description: Reninitialise le decliqueur.                           */
/*==========================================================================*/

void	PLAY_reset_click_removal (PLAY_TRACK_INFO *track_info_ptr)
{
	int		chn_cnt;
	int		type_cnt;

	for (type_cnt = 0; type_cnt < 2; type_cnt ++)
	{
		for (chn_cnt = 0; chn_cnt < 2; chn_cnt ++)
		{
			track_info_ptr->mix.spl.cr.old_last_value [type_cnt] [chn_cnt] = 0;
			track_info_ptr->mix.spl.cr.last_value [type_cnt] [chn_cnt] = 0;
			track_info_ptr->mix.spl.cr.first_value [type_cnt] [chn_cnt] = 0;
		}
	}

	track_info_ptr->mix.spl.cr.broken_curve_state = SPLH_BrokenCurveState_NONE;
}



/*==========================================================================*/
/*      Nom: PLAY_set_new_note                                              */
/*      Description: Declanche une note de la partition.                    */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur la piste de samples sur laquelle   */
/*                          la nouvelle note doit etre prise en compte.     */
/*==========================================================================*/

void	PLAY_set_new_note (PLAY_TRACK_INFO *track_info_ptr)
{
	unsigned int	effect;
	unsigned int	effect_nbr;
	int		instr;
	int		note;
	int		current_note;
	int		volume;
	signed int	played_note;
	int		sample;
	signed int	autobal;

	note = track_info_ptr->score.score.note;
	instr = track_info_ptr->score.score.instr;
	effect = track_info_ptr->score.score.effect;
	effect_nbr = effect >> 8;

/*______________________________________________
 *
 * Changement d'instrument
 *______________________________________________
 */

	if (instr != 0)
	{
		track_info_ptr->score.instr = instr;
		PLAY_set_lin_velocity (track_info_ptr, INST_get_instr_volume (instr) * 8);
		current_note = track_info_ptr->score.note;
		sample = INST_get_instr_sample (instr, current_note);
		track_info_ptr->score.sample.number = sample;
		track_info_ptr->score.sample.transp = INST_get_instr_transp (instr, current_note);
		track_info_ptr->score.sample.volume = SAMP_get_sample_volume (sample);
		track_info_ptr->score.sample.finetune = SAMP_get_sample_finetune (sample);
		if (track_info_ptr->track_type == Pattern_TYPE_SPL)
		{
			track_info_ptr->stereo = SAMP_get_sample_stereo (sample);
		}
		autobal = SAMP_get_sample_balance (sample);
		if (autobal != -1)
		{
			track_info_ptr->score.track.panning = (UWORD)autobal << 4;
		}
		track_info_ptr->score.lpf.filter_flag = INST_get_filter_flag (instr);
		if (track_info_ptr->score.lpf.filter_flag)
		{
			track_info_ptr->score.changed_filter_flag = true;
			track_info_ptr->score.lpf.freq_vol_flag = INST_get_filter_freq_vol_flag (instr);
			track_info_ptr->score.lpf.q_vol_flag = INST_get_filter_q_vol_flag (instr);
			track_info_ptr->score.lpf.freq [0] = INST_get_filter_freq (instr, 0);
			track_info_ptr->score.lpf.freq [1] = INST_get_filter_freq (instr, 1);
			track_info_ptr->score.lpf.q [0] = INST_get_filter_q (instr, 0);
			track_info_ptr->score.lpf.q [1] = INST_get_filter_q (instr, 1);
		}
	}

/*______________________________________________
 *
 * Note
 *______________________________________________
 */

	if (note != 0)
	{
		/* Avec Tone Portamento */
		if (   effect_nbr == 0x03
			 || effect_nbr == 0x05
			 || effect_nbr == 0x06
			 || effect_nbr == 0xAB
			 || (effect_nbr >= 0x18 && effect_nbr <= 0x1B))
		{
			track_info_ptr->score.note = note;
			track_info_ptr->score.porta.note = note;
			played_note = note + (signed int)track_info_ptr->score.sample.transp;
			played_note = MIN (played_note, GTK_NBRNOTES_MAXI-1);
			played_note = MAX (played_note, 1);
			track_info_ptr->score.porta.period =
				(float) PLAY_convert_note_to_internal_period
				(played_note, track_info_ptr->score.sample.finetune);
		}

		/* Sans Tone Portamento */
		else
		{
			track_info_ptr->score.note = note;
			track_info_ptr->score.porta.note = note;
			instr = track_info_ptr->score.instr;

			if (instr != 0)
			{
				sample = INST_get_instr_sample (instr, note);
				track_info_ptr->score.sample.number = sample;
				track_info_ptr->score.sample.transp = INST_get_instr_transp (instr, note);
				track_info_ptr->score.sample.volume = SAMP_get_sample_volume (sample);
				track_info_ptr->score.sample.finetune = SAMP_get_sample_finetune (sample);
				autobal = SAMP_get_sample_balance (sample);
				if (autobal != -1)
				{
					track_info_ptr->score.track.panning = (UWORD)autobal << 4;
				}

				track_info_ptr->mix.spl.sample_ptr = SAMP_get_sample_data_adr (sample);
				track_info_ptr->mix.spl.loopbuf_ptr = SAMP_get_sample_loopbuf_adr (sample);
				track_info_ptr->mix.spl.loopbuf_len = SAMP_get_sample_loopbuf (sample);
				track_info_ptr->mix.spl.resol = SAMP_get_sample_resolution (sample) >> 3;
				track_info_ptr->mix.spl.tracks = SAMP_get_sample_stereo (sample);
				track_info_ptr->mix.spl.freq = SAMP_get_sample_freq (sample);
				track_info_ptr->mix.spl.loopmode = (WORD) SAMP_get_sample_loop (sample);

				/* Pas de bouclage */
				if (SAMP_get_sample_loop (sample) == WaveForm_LOOP_TYPE_NONE)
				{
					track_info_ptr->mix.spl.reppos = SAMP_get_sample_length (sample);
					track_info_ptr->mix.spl.replen = 0;
				}

				/* Bouclage */
				else
				{
					track_info_ptr->mix.spl.reppos = SAMP_get_sample_repeat (sample);
					track_info_ptr->mix.spl.replen = SAMP_get_sample_replen (sample);
				}

				/* Direct-2-Disk */
				if (SAMP_get_sample_type (sample) == 1)
				{
					track_info_ptr->mix.spl.d2d_flag = true;
					track_info_ptr->mix.spl.file_ptr = SAMP_get_sample_file_ptr (sample);
					track_info_ptr->mix.spl.file_data_offset = SAMP_get_sample_file_data_offset (sample);;
					track_info_ptr->mix.spl.startbuf_ptr = SAMP_get_sample_startbuf_ptr (sample);
					track_info_ptr->mix.spl.startbuf_len = SAMP_get_sample_buffer_len (sample, 0);
					track_info_ptr->mix.spl.midbuf1_len = SAMP_get_sample_buffer_len (sample, 1);
					track_info_ptr->mix.spl.midbuf2_len = SAMP_get_sample_buffer_len (sample, 2);
				}

				else
				{
					track_info_ptr->mix.spl.d2d_flag = false;
				}
			}	/* Fin de instr != 0 */

			track_info_ptr->mix.spl.direction = 0;
			PLAYENV_init_all_envelopes (track_info_ptr, instr);

			/* Demarre la note au debut (sauf si effet Sample Offset) */
			if (   (effect_nbr & 0xF0) != 0x90	/* Sample Offset */
				 && effect_nbr != 0xBA				/* Fine Sample Offset */
				 && effect_nbr != 0xBB)				/* Very Fine Sample Offset */
			{
				track_info_ptr->mix.spl.curpos_int = 0;
				track_info_ptr->mix.spl.curpos_frac = 0;
			}

			/* Fixe la note maintenant */
			played_note = note + (signed int)track_info_ptr->score.sample.transp;
			played_note = MIN (played_note, GTK_NBRNOTES_MAXI - 1);
			played_note = MAX (played_note, 1);
			track_info_ptr->score.period =
				(float) PLAY_convert_note_to_internal_period
				(played_note, track_info_ptr->score.sample.finetune);
			track_info_ptr->score.porta.period = track_info_ptr->score.period;
			track_info_ptr->score.porta.note = note;

			/* Indique au click removal une rupture de courbe */
			if (track_info_ptr->track_type == Pattern_TYPE_SPL)
			{
				track_info_ptr->mix.spl.cr.broken_curve_state = SPLH_BrokenCurveState_START;
			}

			/* Debut de note: vide le buffer du filtre */
			if (track_info_ptr->score.lpf.filter_flag)
			{
				for (int spl = 0; spl < 4*2; spl ++)
				{
					track_info_ptr->mix.spl.buffer_y [spl] = 0;
					track_info_ptr->mix.spl.buffer_x [spl] = 0;
				}
				track_info_ptr->mix.spl.buffer_pos = 0;
			}
		}	// Fin de Sans Tone portamento
	}	// Fin du if note != 0

/*______________________________________________
 *
 * Colonne de volume
 *______________________________________________
 */

	volume = track_info_ptr->score.score.volume;
	if (   volume >= 0x10
	    && volume <= 0x50)
	{
		PLAY_set_lin_velocity (track_info_ptr, (volume - 0x10) << 5);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_set_lin_velocity                                          */
/*      Description: Fixe la velocite lineaire d'une piste (et recalcule la */
/*                   velocite logarithmique).                               */
/*      Parametres en entree:                                               */
/*        - velo: La nouvelle velocite de la piste.                         */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_lin_velocity (PLAY_TRACK_INFO *track_info_ptr, int velo)
{
	velo = MIN (velo, 0x800);
	velo = MAX (velo, 0);
	track_info_ptr->score.velocity_lin = velo;
	track_info_ptr->score.velocity_loc = velo;
	track_info_ptr->score.velocity_log = PLAY_instr_vol_table_log [velo];
	track_info_ptr->score.changed_volume_flag = true;
}



/*==========================================================================*/
/*      Nom: PLAY_set_log_velocity                                          */
/*      Description: Fixe la velocite logarithmique d'une piste (et         */
/*                   recalcule la velocite lineaire).                       */
/*      Parametres en entree:                                               */
/*        - velo: La nouvelle velocite de la piste.                         */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_log_velocity (PLAY_TRACK_INFO *track_info_ptr, int velo)
{
	velo = MIN (velo, 0x800);
	velo = MAX (velo, 0);
	track_info_ptr->score.velocity_log = velo;
	track_info_ptr->score.velocity_lin = PLAY_instr_vol_table_exp [velo];
	track_info_ptr->score.velocity_loc = track_info_ptr->score.velocity_lin;
	track_info_ptr->score.changed_volume_flag = true;
}



/*==========================================================================*/
/*      Nom: PLAY_set_lin_volume                                            */
/*      Description: Fixe le volume lineaire d'une piste (et                */
/*                   recalcule le volume logarithmique).                    */
/*      Parametres en entree:                                               */
/*        - lin_vol: Le nouveau volume de la piste.                         */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_lin_volume (PLAY_TRACK_INFO *track_info_ptr, long lin_vol)
{
	int		log_vol;

	lin_vol = MIN (lin_vol, 0xFFFFL);
	lin_vol = MAX (lin_vol, 0);
	track_info_ptr->score.track.volume_lin = (UWORD)lin_vol;
	log_vol = PLAY_lin_to_log_volume (lin_vol);
	track_info_ptr->score.track.volume_log = (UWORD)log_vol;
	if (   track_info_ptr->track_type == Pattern_TYPE_SPL
	    || track_info_ptr->track_type == Pattern_TYPE_AIN
		 || track_info_ptr->track_type == Pattern_TYPE_FX)
	{
		track_info_ptr->score.changed_volume_flag = true;
	}
	else
	{
		track_info_ptr->outvol = (UWORD)lin_vol;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_set_log_volume                                            */
/*      Description: Fixe le volume logarithmique d'une piste (et           */
/*                   recalcule le volume lineaire).                         */
/*      Parametres en entree:                                               */
/*        - log_vol: Le nouveau volume de la piste.                         */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_log_volume (PLAY_TRACK_INFO *track_info_ptr, int log_vol)
{
	unsigned int	lin_vol;

	log_vol = MIN (log_vol, 0xFFF);
	log_vol = MAX (log_vol, 0);
	track_info_ptr->score.track.volume_log = (UWORD)log_vol;
	lin_vol = PLAY_log_to_lin_volume (log_vol);
	track_info_ptr->score.track.volume_lin = (UWORD)lin_vol;
	if (   track_info_ptr->track_type == Pattern_TYPE_SPL
	    || track_info_ptr->track_type == Pattern_TYPE_AIN
		 || track_info_ptr->track_type == Pattern_TYPE_FX)
	{
		track_info_ptr->score.changed_volume_flag = true;
	}
	else
	{
		track_info_ptr->outvol = (UWORD)lin_vol;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_set_dry_lin_volume                                        */
/*      Description: Fixe le volume lineaire dry d'une piste (et            */
/*                   recalcule le volume logarithmique).                    */
/*      Parametres en entree:                                               */
/*        - lin_vol: Le nouveau volume de la piste.                         */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_dry_lin_volume (PLAY_TRACK_INFO *track_info_ptr, long lin_vol)
{
	int		log_vol;

	lin_vol = MIN (lin_vol, 0xFFFFL);
	lin_vol = MAX (lin_vol, 0);
	track_info_ptr->score.track.dryvol_lin = (UWORD)lin_vol;
	log_vol = PLAY_lin_to_log_volume (lin_vol);
	track_info_ptr->score.track.dryvol_log = (UWORD)log_vol;
	if (   track_info_ptr->track_type == Pattern_TYPE_SPL
	    || track_info_ptr->track_type == Pattern_TYPE_AIN
		 || track_info_ptr->track_type == Pattern_TYPE_FX)
	{
		track_info_ptr->score.changed_volume_flag = true;
	}
	else
	{
		track_info_ptr->dry_outvol = (UWORD)lin_vol;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_set_dry_log_volume                                        */
/*      Description: Fixe le volume logarithmique dry d'une piste (et       */
/*                   recalcule le volume lineaire).                         */
/*      Parametres en entree:                                               */
/*        - log_vol: Le nouveau volume de la piste.                         */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_dry_log_volume (PLAY_TRACK_INFO *track_info_ptr, int log_vol)
{
	unsigned int	lin_vol;

	log_vol = MIN (log_vol, 0xFFF);
	log_vol = MAX (log_vol, 0);
	track_info_ptr->score.track.dryvol_log = (UWORD)log_vol;
	lin_vol = PLAY_log_to_lin_volume (log_vol);
	track_info_ptr->score.track.dryvol_lin = (UWORD)lin_vol;
	if (   track_info_ptr->track_type == Pattern_TYPE_SPL
	    || track_info_ptr->track_type == Pattern_TYPE_AIN
		 || track_info_ptr->track_type == Pattern_TYPE_FX)
	{
		track_info_ptr->score.changed_volume_flag = true;
	}
	else
	{
		track_info_ptr->dry_outvol = (UWORD)lin_vol;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_set_period                                                */
/*      Description: Fixe la periode de la note pour une piste.             */
/*      Parametres en entree:                                               */
/*        - period: La nouvelle periode de la piste.                        */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_period (PLAY_TRACK_INFO *track_info_ptr, float period)
{
	period = MIN (period, PLAY_MAX_PERIOD);
	period = MAX (period, PLAY_MIN_PERIOD);
	track_info_ptr->score.period = period;
	track_info_ptr->score.period_loc = period;
}



/*==========================================================================*/
/*      Nom: PLAY_set_panning                                               */
/*      Description: Fixe le panning pour une piste.                        */
/*      Parametres en entree:                                               */
/*        - dry_flag: indique qu'on veut manipuler la partie Dry.           */
/*        - pan: La nouvelle valeur du panning.                             */
/*      Parametres en entree/sortie:                                        */
/*        - track_info_ptr: pointeur sur les informations de la piste.      */
/*==========================================================================*/

void	PLAY_set_panning (PLAY_TRACK_INFO *track_info_ptr, bool dry_flag, long pan)
{
	pan = MIN (pan, 0xFFFFL);
	pan = MAX (pan, 0);
	if (dry_flag)
	{
		track_info_ptr->score.track.drypan = (UWORD) pan;
	}
	else
	{
		track_info_ptr->score.track.panning = (UWORD) pan;
	}
}




/****************************************************************************/
/*                                                                          */
/*      INTERFACAGE AVEC L'EXTERIEUR                                        */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom: PLAY_get_song_position                                         */
/*      Description: Renvoie le mode de replay dans lequel on est.          */
/*      Retour: Le numero du mode.                                          */
/*==========================================================================*/

int	PLAY_get_play_mode (void)
{
	return (PLAY_score_info.current_play_mode);
}



/*==========================================================================*/
/*      Nom: PLAY_set_play_mode                                             */
/*      Description: Change le mode de replay (Stop/Step, Song et Pattern). */
/*                   Le mode Stop/Step permet de stopper toute activite     */
/*                   sonore ou de jouer une ligne de pattern. Cette ligne   */
/*                   devra etre placee dans le(s) pattern(s) bidon(s).      */
/*                   La fonction rend la main une fois que le nouveau mode  */
/*                   a effectivement ete pris en compte. Le nouveau mode    */
/*                   est cense prendre effet au prochain tick.              */
/*      Parametres en entree:                                               */
/*        - play_mode: Mode de replay (0, 1 ou 2)                           */
/*        - position: position de debut de replay dans la song. N'est       */
/*                    utilise que par les modes 1 et 2.                     */
/*        - line: numero de ligne dans le pattern. N'est utilise que par    */
/*                les modes 1 et 2 avec cont_flag = true.                   */
/*        - cont_flag: Pour les modes song et pattern, true indique que le  */
/*                     replay commence (ou continue) exactement a la        */
/*                     position et ligne courante.                          */
/*==========================================================================*/

void	PLAY_set_play_mode (signed int play_mode, signed int position, signed int line, bool cont_flag)
{
	PLAY_new_play_mode = play_mode;
	PLAY_new_play_mode_cont_flag = cont_flag;
	PLAY_new_linepos = line;
	PLAY_new_songpos = position;
	PLAY_new_play_mode_flag = true;

	PLAY_wait_for_new_play_mode ();
}



/*==========================================================================*/
/*      Nom: PLAY_wait_for_new_play_mode                                    */
/*      Description: Attend que le nouveau mode de replay ait ete pris en   */
/*                   compte.                                                */
/*==========================================================================*/

void	PLAY_wait_for_new_play_mode (void)
{
	/* Si le player est en marche, attend que le nouveau
	   mode ait ete pris en compte. */
	if (PLAY_get_player_started_flag ())
	{
		while (PLAY_new_play_mode_flag)
		{
			/* Boucle d'attente vide */
		}
	}
}



/*==========================================================================*/
/*      Nom: PLAY_clear_all_effects                                         */
/*      Description: Retire tous les effets de toutes les pistes d'effet.   */
/*==========================================================================*/

void	PLAY_clear_all_effects (void)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	for (int track = 0; track < GTK_nbr_tracks [Pattern_TYPE_FX]; track ++)
	{
		track_info_ptr =   PLAY_track_info
		                 + PLAY_track_info_list [Pattern_TYPE_FX] [track];

		if (   track_info_ptr->mix.fx.effect != FxPreset_TYPE_NONE
		    && ! track_info_ptr->mix.fx.bad_init_flag)
		{
			PLAYFX_fx_restore (track_info_ptr);
		}
	}

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_silence_all_effect_buffers                                */
/*      Description: Reduit au silence tous les buffers des effets.         */
/*==========================================================================*/

void	PLAY_silence_all_effect_buffers (void)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	for (int track = 0; track < GTK_nbr_tracks [Pattern_TYPE_FX]; track ++)
	{
		track_info_ptr =   PLAY_track_info
		                 + PLAY_track_info_list [Pattern_TYPE_FX] [track];

		if (   track_info_ptr->mix.fx.effect != FxPreset_TYPE_NONE
		    && ! track_info_ptr->mix.fx.bad_init_flag)
		{
			PLAYFX_fx_silence (track_info_ptr);
		}
	}

	GTK_mutex.signal ();
}


/*==========================================================================*/
/*      Nom: PLAY_get_song_position                                         */
/*      Description: Renvoie la position courante dans la song.             */
/*      Retour: La position.                                                */
/*==========================================================================*/

int	PLAY_get_song_position (void)
{
	if (PLAY_score_info.current_play_mode != PLAY_MODE_SONG)
	{
		return (PLAY_current_songpos);
	}

	return (PLAY_score_info.current_pos);
}



/*==========================================================================*/
/*      Nom: PLAY_set_song_position                                         */
/*      Description: Fixe une nouvelle position dans la chanson.            */
/*      Parametres en entree:                                               */
/*        - songpos: La nouvelle position.                                  */
/*==========================================================================*/

void	PLAY_set_song_position (int songpos)
{
	if (PLAY_score_info.current_play_mode != PLAY_MODE_STOP)
	{
		PLAY_set_play_mode (PLAY_score_info.current_play_mode, songpos, -1, true);
		return;
	}

	PLAY_current_songpos = songpos;
}



/*==========================================================================*/
/*      Nom: PLAY_get_line_position                                         */
/*      Description: Renvoie la ligne courante dans la pattern courant.     */
/*      Retour: La ligne.                                                   */
/*==========================================================================*/

int	PLAY_get_line_position (void)
{
	if (PLAY_score_info.current_play_mode == PLAY_MODE_STOP)
	{
		return (PLAY_current_linepos);
	}

	return (PLAY_score_info.current_line);
}



/*==========================================================================*/
/*      Nom: PLAY_set_line_position                                         */
/*      Description: Fixe une nouvelle ligne courante dans le pattern.      */
/*      Parametres en entree:                                               */
/*        - songpos: La nouvelle ligne.                                     */
/*==========================================================================*/

void	PLAY_set_line_position (int linepos)
{
	if (PLAY_score_info.current_play_mode != PLAY_MODE_STOP)
	{
		PLAY_set_play_mode (PLAY_score_info.current_play_mode, -1, linepos, true);
		return;
	}

	PLAY_current_linepos = linepos;
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

int	PLAY_get_current_tick (void)
{
	return (PLAY_score_info.tick_counter);
}



void	PLAY_get_elapsed_time (int &minutes, int &seconds, int &miliseconds)
{
	minutes = 0;
	seconds = 0;
	miliseconds = 0;

	const double		sample_freq = static_cast <double> (MIX_replay_freq);
	const double		elapsed_spl = static_cast <double> (PLAY_elapsed_time);

	if (elapsed_spl >= 0)
	{
		const double	elapsed_s_flt = elapsed_spl / sample_freq;
		const long		elapsed_ms =
			static_cast <long> (floor (elapsed_s_flt * 1000));

		const long		elapsed_s = elapsed_ms / 1000;
		miliseconds = elapsed_ms - elapsed_s * 1000;

		const long		elapsed_min = elapsed_s / 60;
		seconds = elapsed_s - elapsed_min * 60;

		minutes = elapsed_min;
	}
}



void	PLAY_reset_elapsed_time ()
{
	PLAY_elapsed_time = 0;
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	PLAY_reset_mix_parameters (void)
{
	int		track;
	int		track_type;
	int		track_cnt;
	int		global_track_nbr;
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	/* Initialisation des niveaux de clipping */
	PLAY_set_max_clipping_level (0);
	for (track = 0; track < GTK_NBROUT_MAXI; track ++)
	{
		PLAY_set_clipping_level (track, 0);
	}

	/* Initialisation des pistes de samples */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_SPL]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_SPL] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		PLAY_set_lin_volume (track_info_ptr, 0x1000);
		track_info_ptr->score.track.panning = 0x8000U;
		track_info_ptr->stereo = 1;
		if (MIX_get_sound_quality () > MIX_SOUND_QUALITY_NORMAL)
		{
			track_info_ptr->mix.spl.recons_filter_ptr = new ReconstructionFilter (MIX_recons_filter_width);
			if (track_info_ptr->mix.spl.recons_filter_ptr == NULL)
			{
				LOG_printf ("PLAY_reset_mix_parameters: Warning: couldn't create a reconstruction filter for Sample track # %d.\n",
				            track);
			}
			else if (track_info_ptr->mix.spl.recons_filter_ptr->check_ok ())
			{
				LOG_printf ("PLAY_reset_mix_parameters: Warning: couldn't initialize reconstruction filter for Sample track # %d.\n",
				            track);
				delete track_info_ptr->mix.spl.recons_filter_ptr;
				track_info_ptr->mix.spl.recons_filter_ptr = NULL;
			}
		}
		else
		{
			track_info_ptr->mix.spl.recons_filter_ptr = NULL;
		}
	}

	/* Initialisation des pistes d'entree */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_AIN]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_AIN] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		if (track >= 2)
		{
			track_info_ptr->stereo = 1;
		}
		else if (track == 1)
		{
			track_info_ptr->stereo = 0;
		}
		else
		{
			track_info_ptr->stereo = 2;
		}
		PLAY_set_lin_volume (track_info_ptr, 0x1000);
		track_info_ptr->score.track.panning = 0x8000U;
		track_info_ptr->wet_mute_flag = false;
		track_info_ptr->dry_mute_flag = false;
	}

	/* Initialisation de la premiere piste de sortie: elle contient
		toutes les pistes de samples et d'effet en mode Wet. */
	global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [0];
	track_info_ptr = PLAY_track_info + global_track_nbr;
	track_info_ptr->stereo = 2;
	PLAY_set_lin_volume (
		track_info_ptr,
	   (long) (0x2000 / (log (static_cast <double> (GTK_nbr_tracks [Pattern_TYPE_SPL])) / LN2 + 1))
	);
	track_info_ptr->mix.src.nbr_s_tracks = GTK_nbr_tracks [Pattern_TYPE_SPL];
	track_cnt = 0;
	for (track_type = 0; track_type < Pattern_NBR_TYPES; track_type ++)
	{
		if (   track_type != Pattern_TYPE_SPL
		    && track_type != Pattern_TYPE_FX)
		{
			continue;
		}

		for (track = 0; track < GTK_nbr_tracks [track_type]; track ++)
		{
			track_info_ptr->mix.src.track_conf [track_cnt].track_type = track_type;
			track_info_ptr->mix.src.track_conf [track_cnt].wet_flag = true;
			track_info_ptr->mix.src.track_conf [track_cnt].track_nbr = track;
			/* Ces deux lignes n'ont pas d'importance car les pistes Audio Out n'ont
				pas de reglage de volume ou de balance pour les pistes d'entree. */
			track_info_ptr->mix.src.track_conf [track_cnt].inpvol = 0x1000;
			track_info_ptr->mix.src.track_conf [track_cnt].inppan = 0x8000U;
			track_info_ptr->mix.src.track_conf [track_cnt].cr.old_vol [0] = 0;
			track_info_ptr->mix.src.track_conf [track_cnt].cr.old_vol [1] = 0;

			track_cnt ++;
		}
	}
	track_info_ptr->mix.src.nbr_s_tracks = track_cnt;

	/* Initialisation des autres pistes de sortie */
	for (track = 1; track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		track_info_ptr->stereo = 1;
		PLAY_set_lin_volume (track_info_ptr, 0x1000);
		track_info_ptr->mix.src.nbr_s_tracks = 0;
	}

	/* Initialisations des pistes d'effets: 1 piste de sample en entree. */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_FX]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_FX] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;
		track_info_ptr->stereo = 1;
		PLAY_set_dry_lin_volume (track_info_ptr, 0x1000);
		track_info_ptr->score.track.drypan = 0x8000U;
		PLAY_set_lin_volume (track_info_ptr, 0x1000);
		track_info_ptr->score.track.panning = 0x8000U;
		track_info_ptr->wet_mute_flag = false;
		track_info_ptr->dry_mute_flag = false;
		track_info_ptr->mix.src.nbr_s_tracks = 1;

		track_info_ptr->mix.src.track_conf [0].track_type = Pattern_TYPE_SPL;
		track_info_ptr->mix.src.track_conf [0].wet_flag = true;
		if (track < GTK_nbr_tracks [Pattern_TYPE_SPL])
		{
			track_info_ptr->mix.src.track_conf [0].track_nbr = track;
		}
		else
		{
			track_info_ptr->mix.src.track_conf [0].track_nbr = 0;
		}
		track_info_ptr->mix.src.track_conf [0].inpvol = 0x1000;
		track_info_ptr->mix.src.track_conf [0].inppan = 0x8000U;
		track_info_ptr->mix.src.track_conf [0].cr.old_vol [0] = 0;
		track_info_ptr->mix.src.track_conf [0].cr.old_vol [1] = 0;
	}

	/* Initialisation des pistes MIDI */
	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_FX]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [Pattern_TYPE_FX] [track];
		track_info_ptr = PLAY_track_info + global_track_nbr;

		/*** A faire ***/

	}

	PLAY_set_linear_period_flag (PLAY_INIT_LINEAR_PERIOD_FLAG);
	PLAY_no_mute ();
	PLAY_default_interpolation ();

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_add_source_track                                          */
/*      Description: Ajoute une piste source a une piste d'effet ou d'Audio */
/*                   Out. Les tests necessaires sont effectues pour savoir  */
/*                   si la nouvelle piste ne va pas introduire de bouclage, */
/*                   ou si elle est deja presente.                          */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste sur laquelle on va ajouter une     */
/*                      source.                                             */
/*        - track_nbr: Numero de la piste.                                  */
/*        - new_track_conf_ptr: pointeur sur la structure de configuration  */
/*                              de la piste.                                */
/*        - flags: permet de savoir de quelle facon  ajouter la piste       */
/*                 source (type PLAY_MIX_PRESET_???). Tenter d'introduire   */
/*                 une piste deja existante en mode REPL n'a absolument     */
/*                 aucun effet.                                             */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si la piste etait deja dans les sources (rien n'est       */
/*                change),                                                  */
/*              2 si l'introduction de la piste aurait induit un bouclage   */
/*                (operation annulee).                                      */
/*              -1 si erreur.                                               */
/*==========================================================================*/

signed int	PLAY_add_source_track (int track_type, int track_nbr, const PLAY_SOURCE_TRACK_CONF_BLOCK *new_track_conf_ptr, int flags)
{
	int		nbr_src;
	int		insert_pos;
	signed int	ret_val;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (track_type, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_add_source_track: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	/* S'il s'agit d'une piste d'Audio In, regarde si elle est valide */
	if (! PLAY_check_track_validity (new_track_conf_ptr->track_type, new_track_conf_ptr->track_nbr))
	{
		GTK_mutex.signal ();
		return (3);
	}

	/* Cherche si la piste est deja dans les sources; cherche aussi ou inserer
	   la nouvelle piste. */
	ret_val = PLAY_check_track_is_in_sources (track_type, track_nbr, &insert_pos, new_track_conf_ptr);

	/* La piste existe deja. On va donc seulement modifier ses parametres */
	if (ret_val == 1)
	{
		if ((flags & PLAY_MIX_PRESET_MASK) != PLAY_MIX_PRESET_REPL)
		{
			if ((flags & PLAY_MIX_PRESET_NO_VOL) == 0)
			{
				track_conf_ptr [insert_pos].inpvol = new_track_conf_ptr->inpvol;
			}
			if ((flags & PLAY_MIX_PRESET_NO_PAN) == 0)
			{
				track_conf_ptr [insert_pos].inppan = new_track_conf_ptr->inppan;
			}
		}

		GTK_mutex.signal ();
		return (1);
	}

	else if (ret_val < 0)
	{
		LOG_printf ("PLAY_add_source_track: Error: couldn't get source track information.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	/* Regarde si en connectant cette piste on provoque un bouclage */
	if (   new_track_conf_ptr->track_type == Pattern_TYPE_FX
	    && track_type == Pattern_TYPE_FX)
	{
		if (PLAY_find_track_loop (new_track_conf_ptr->track_nbr, track_nbr))
		{
			GTK_mutex.signal ();
			return (2);
		}
	}

	/* Insere la piste dans la liste */
	if (insert_pos < nbr_src)
	{
		memmove (track_conf_ptr + insert_pos + 1,
		         track_conf_ptr + insert_pos,
		         sizeof (*track_conf_ptr) * (nbr_src - insert_pos));
	}
	track_conf_ptr [insert_pos] = *new_track_conf_ptr;
	track_conf_ptr [insert_pos].cr.old_vol [0] = 0;
	track_conf_ptr [insert_pos].cr.old_vol [1] = 0;
	nbr_src ++;
	if (PLAY_set_nbr_source_tracks (track_type, track_nbr, nbr_src))
	{
		LOG_printf ("PLAY_add_source_track: Error: bad destination track type.");
		GTK_mutex.signal ();
		return (-1);
	}

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_change_source_track                                       */
/*      Description: Change une piste source a une piste d'effet ou d'Audio */
/*                   Out. Aucun test d'integrite n'est effectue pour savoir */
/*                   si la nouvelle piste ne va pas introduire de bouclage, */
/*                   ou si elle est deja presente.                          */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste sur laquelle on va modifier une    */
/*                      source.                                             */
/*        - track_nbr: Numero de la piste.                                  */
/*        - position: numero de la source                                   */
/*        - new_track_conf_ptr: pointeur sur la structure de configuration  */
/*                              de la piste.                                */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si la piste etait deja dans les sources (rien n'est       */
/*                change),                                                  */
/*              2 si l'introduction de la piste aurait induit un bouclage   */
/*                (operation annulee).                                      */
/*              -1 si erreur.                                               */
/*==========================================================================*/

signed int	PLAY_change_source_track (int track_type, int track_nbr, int position, const PLAY_SOURCE_TRACK_CONF_BLOCK *new_track_conf_ptr)
{
	int		nbr_src;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (track_type, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_add_source_track: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	track_conf_ptr [position] = *new_track_conf_ptr;
	track_conf_ptr [position].cr.old_vol [0] = 0;
	track_conf_ptr [position].cr.old_vol [1] = 0;

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_remove_source_track                                       */
/*      Description: Retire une piste des sources d'une piste d'effet ou    */
/*                   d'Audio Out, si cela est possible.                     */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste sur laquelle on va ajouter une     */
/*                      source.                                             */
/*        - track_nbr: Numero de la piste.                                  */
/*        - new_track_conf_ptr: pointeur sur la structure de configuration  */
/*                              de la piste a retirer. Seuls les champs     */
/*                              track_type, track_nbr et wet_flag ont       */
/*                              besoin d'etre rempli.                       */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si la piste n'etait pas dans les sources.                 */
/*              2 si la piste etait la derniere et qu'on ne pouvait pas     */
/*                l'enlever.                                                */
/*              -1 si erreur.                                               */
/*==========================================================================*/

signed int	PLAY_remove_source_track (int track_type, int track_nbr, const PLAY_SOURCE_TRACK_CONF_BLOCK *new_track_conf_ptr)
{
	int		src_cnt;
	int		nbr_src;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (track_type, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_remove_source_track: Error: bad destination track type.");
		GTK_mutex.signal ();
		return (-1);
	}

	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		if (   track_conf_ptr [src_cnt].track_type == new_track_conf_ptr->track_type
		    && track_conf_ptr [src_cnt].track_nbr == new_track_conf_ptr->track_nbr)
		{
			if (   new_track_conf_ptr->track_type != Pattern_TYPE_FX
			    || new_track_conf_ptr->wet_flag == track_conf_ptr [src_cnt].wet_flag)
			{
				/* On ne peut pas retirer la derniere piste source */
				if (nbr_src <= 1)
				{
					GTK_mutex.signal ();
					return (2);
				}

				if (src_cnt < nbr_src - 1)
				{
					memmove (track_conf_ptr + src_cnt,
					         track_conf_ptr + src_cnt + 1,
					         sizeof (*track_conf_ptr) * (nbr_src - src_cnt - 1));
				}
				nbr_src --;
				if (PLAY_set_nbr_source_tracks (track_type, track_nbr, nbr_src))
				{
					LOG_printf ("PLAY_remove_source_track: Error: bad destination track type.");
					GTK_mutex.signal ();
					return (-1);
				}

				GTK_mutex.signal ();
				return (0);
			}
		}
	}

	GTK_mutex.signal ();

	return (1);
}



/*==========================================================================*/
/*      Nom: PLAY_check_track_validity                                      */
/*      Description: Teste si une piste donnee existe bien reellement.      */
/*                   Permet particulierement de tester les pistes AIN/AOU   */
/*                   qui feraient en fait partie d'une piste stereo situee  */
/*                   juste avant (elle n'existe donc pas a proprement       */
/*                   parler).                                               */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste concernee.                         */
/*        - track_nbr: Numero de la piste.                                  */
/*      Retour: true si la piste existe, false sinon.                       */
/*==========================================================================*/

bool	PLAY_check_track_validity (int track_type, int track_nbr)
{
	int		nbr_tracks;
	int		track_count;
	int		stereo;

	if (   track_type != GTK_TRACK_TYPE_AOU
	    && track_type != Pattern_TYPE_AIN)
	{
		if (track_nbr < GTK_nbr_tracks [track_type])
		{
			return (true);
		}
		else
		{
			return (false);
		}
	}
	
	nbr_tracks = MIN (GTK_nbr_tracks [track_type], track_nbr + 1);
	for (track_count = 0; track_count < nbr_tracks; track_count ++)
	{
		if (track_nbr == track_count)
		{
			return (true);
		}

		stereo = PLAY_get_track_stereo (track_type, track_count, false) - 1;
		while (stereo > 0)
		{
			track_count ++;
			stereo --;
		}
	}

	return (false);
}



/*==========================================================================*/
/*      Nom: PLAY_check_track_is_in_sources                                 */
/*      Description: Regarde si une piste donnee fait partie des sources    */
/*                   d'une certaine piste.                                  */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste concernee.                         */
/*        - track_nbr: Numero de la piste.                                  */
/*        - new_track_conf_ptr: pointeur sur la structure de configuration  */
/*                              de la piste a examiner. Seuls les champs    */
/*                              track_type, track_nbr et wet_flag ont       */
/*                              besoin d'etre rempli.                       */
/*      Parametres en sortie:                                               */
/*        - position_ptr: pointeur sur la position de la piste dans les     */
/*                        sources si on l'a trouvee, ou indique l'endroit   */
/*                        ou cette piste devrait etre inseree.              */
/*                        Si le pointeur est NULL, ne fait rien.            */
/*      Retour: 0 si la piste n'est pas trouvee,                            */
/*              1 si la piste est trouvee,                                  */
/*              -1 si erreur.                                               */
/*==========================================================================*/

signed int	PLAY_check_track_is_in_sources (int track_type, int track_nbr, int *position_ptr, const PLAY_SOURCE_TRACK_CONF_BLOCK *new_track_conf_ptr)
{
	int		src_cnt;
	int		nbr_src;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (track_type, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_check_track_is_in_sources: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		if (   track_conf_ptr [src_cnt].track_type == new_track_conf_ptr->track_type
		    && track_conf_ptr [src_cnt].track_nbr == new_track_conf_ptr->track_nbr)
		{
			if (   new_track_conf_ptr->track_type != Pattern_TYPE_FX
			    || track_conf_ptr [src_cnt].wet_flag == new_track_conf_ptr->wet_flag)
			{
				if (position_ptr != NULL)
				{
					*position_ptr = src_cnt;
				}
				GTK_mutex.signal ();
				return (1);
			}
		}
		else if (   track_conf_ptr [src_cnt].track_type > new_track_conf_ptr->track_type
		         || (   track_conf_ptr [src_cnt].track_type == new_track_conf_ptr->track_type
		             && track_conf_ptr [src_cnt].track_nbr >= new_track_conf_ptr->track_nbr))
		{
			break;
		}
	}

	if (position_ptr != NULL)
	{
		*position_ptr = src_cnt;
	}

	GTK_mutex.signal ();
	
	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_find_track_to_add                                         */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste concernee.                         */
/*        - track_nbr: Numero de la piste.                                  */
/*      Parametres en entree/sortie:                                        */
/*        - new_track_conf_ptr: pointeur sur la structure de configuration  */
/*                              de la piste dont on veut tester la          */
/*                              possibilite d'insertion. Seuls les champs   */
/*                              track_type, track_nbr et wet_flag ont       */
/*                              besoin d'etre rempli. Ces champs sont       */
/*                              modifies si besoin est.                     */
/*      Retour: 0 si la piste est inserable telle-quelle,                   */
/*              1 si on a du changer le numero de la piste,                 */
/*                les donnees de la piste d'entree ont alors ete modifiees. */
/*              2 si il n'est pas possible d'inserer une nouvelle piste,    */
/*                les donnees de la piste d'entree ont alors ete modifiees. */
/*              -1 si erreur.                                               */
/*==========================================================================*/

signed int	PLAY_find_track_to_add (int track_type, int track_nbr, PLAY_SOURCE_TRACK_CONF_BLOCK *new_track_conf_ptr)
{
	signed int	ret_val;
	signed int	track_cnt;

	new_track_conf_ptr->wet_flag =    new_track_conf_ptr->wet_flag
	                               || new_track_conf_ptr->track_type != Pattern_TYPE_FX;

	GTK_mutex.wait ();

	/* Cherche si c'est possible d'inserer la piste */
	for (track_cnt = -1; track_cnt < GTK_nbr_tracks [new_track_conf_ptr->track_type]; track_cnt ++)
	{
		if (track_cnt >= 0)
		{
			new_track_conf_ptr->track_nbr = track_cnt;
		}

		if (PLAY_check_track_validity (new_track_conf_ptr->track_type, new_track_conf_ptr->track_nbr))
		{
			ret_val = PLAY_check_track_is_in_sources (track_type, track_nbr, NULL, new_track_conf_ptr);
			if (ret_val == 0)
			{
				if (   new_track_conf_ptr->track_type == Pattern_TYPE_FX
				    && track_type == Pattern_TYPE_FX)
				{
					if (! PLAY_find_track_loop (new_track_conf_ptr->track_nbr, track_nbr))
					{
						GTK_mutex.signal ();
						return ((track_cnt < 0) ? 0 : 1);
					}
				}
				else
				{
					GTK_mutex.signal ();
					return ((track_cnt < 0) ? 0 : 1);
				}
			}

			else if (ret_val < 0)
			{
				LOG_printf ("PLAY_find_track_to_add: Error: bad destination track type.\n");
				GTK_mutex.signal ();
				return (-1);
			}
		}
	}

	GTK_mutex.signal ();

	return (2);
}



/*==========================================================================*/
/*      Nom: PLAY_find_track_loop                                           */
/*      Description: Recherche si une piste d'effet est dans l'arborescence */
/*                   d'une autre piste d'effet.                             */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste d'effet a scanner                     */
/*        - track_to_find: piste d'effet a rechercher.                      */
/*      Retour: true si la piste est trouvee dans l'arborescence, et donc   */
/*              qu'il va peut-etre y avoir une boucle si insertion de la    */
/*              premiere dans les sources de la deuxieme, false sinon.      */
/*==========================================================================*/

bool	PLAY_find_track_loop (int track, int track_to_find)
{
	int		nbr_src;
	int		src_cnt;
	int		track_to_scan;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	if (track == track_to_find)
	{
		return (true);
	}

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (Pattern_TYPE_FX, track, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_find_track_loop: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (true);
	}

	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		if (track_conf_ptr [src_cnt].track_type == Pattern_TYPE_FX)
		{
			track_to_scan = track_conf_ptr [src_cnt].track_nbr;

			/* Le premier test n'est pas indispensable
			   mais accelere un peu les choses */
			if (track_to_scan == track_to_find)
			{
				GTK_mutex.signal ();
				return (true);
			}

			else if (PLAY_find_track_loop (track_to_scan, track_to_find))
			{
				GTK_mutex.signal ();
				return (true);
			}
		}
	}

	GTK_mutex.signal ();

	return (false);
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_nbr_s_tracks                                    */
/*      Description: Renvoie des informations sur les sources d'une piste:  */
/*                   le nombre de sources et un pointeur sur le tableau des */
/*                   configurations de ces sources.                         */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*      Parametres en sortie:                                               */
/*        - nbr_s_tracks_ptr: pointeur sur le nombre de pistes sources.     */
/*        - track_conf_ptr_ptr: pointeur sur le pointeur des configurations */
/*                              des pistes sources.                         */
/*      Retour: 0 si tout s'est bien passe, ou -1 si le type de la piste ne */
/*              correspondait pas a un type utilisant des pistes sources.   */
/*==========================================================================*/

signed int	PLAY_get_track_nbr_s_tracks (int track_type, int track_nbr, int *nbr_s_tracks_ptr, 	PLAY_SOURCE_TRACK_CONF_BLOCK	**track_conf_ptr_ptr)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type]
	                                        [track_nbr];
	if (   track_type != GTK_TRACK_TYPE_AOU
	    && track_type != Pattern_TYPE_FX)
	{
		return (-1);
	}

	*nbr_s_tracks_ptr = track_info_ptr->mix.src.nbr_s_tracks;
	*track_conf_ptr_ptr = track_info_ptr->mix.src.track_conf;

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_load_mix_preset                                           */
/*      Description: Applique un preset de mixage sur une piste.            */
/*      Parametres en entree:                                               */
/*        - track_type: type de la piste sur laquelle on applique le preset */
/*                      (FX ou Audio Out).                                  */
/*        - track_nbr: numero de la piste.                                  */
/*        - preset_nbr: numero du preset a appliquer.                       */
/*        - flags: permet de savoir de quelle facon  ajouter la piste       */
/*                 source (type PLAY_MIX_PRESET_???).                       */
/*      Retour: 0 si tout s'est bien passe.                                 */
/*==========================================================================*/

signed int	PLAY_load_mix_preset (int track_type, int track_nbr, int preset_nbr, int flags)
{
	int		nbr_src;
	int		src_cnt;
	PLAY_SOURCE_TRACK_CONF_BLOCK	track_conf;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	/* Si on n'a pas precise la destination, on recherche la piste par defaut
	   sur laquelle on peut appliquer le preset. */
	if (   track_type < 0
	    || track_nbr < 0)
	{
		track_type = MIXP_get_track_type (preset_nbr);
		if (   track_type != Pattern_TYPE_FX
		    && track_type != GTK_TRACK_TYPE_AOU)
		{
			LOG_printf ("PLAY_load_mix_preset: Error: bad destination track type.\n");
			GTK_mutex.signal ();
			return (-1);
		}
		track_nbr = MIXP_get_track_number (preset_nbr);
		if (   track_nbr < 0
		    || track_nbr >= GTK_nbr_tracks [track_type])
		{
			LOG_printf ("PLAY_load_mix_preset: Error: bad destination track number.\n");
			GTK_mutex.signal ();
			return (-1);
		}
	}

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type]
	                                        [track_nbr];

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (track_type, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_load_mix_preset: Error: bad destination track type (b).\n");
		GTK_mutex.signal ();
		return (-1);
	}

	nbr_src = MIXP_get_nbr_source_tracks (preset_nbr);

	if ((flags & PLAY_MIX_PRESET_MASK) == PLAY_MIX_PRESET_REPL)
	{
		/* Fixe le nombre de sources a 1 (le mininmum) */
		if (PLAY_set_nbr_source_tracks (track_type, track_nbr, 1))
		{
			LOG_printf ("PLAY_load_mix_preset: Error: bad destination track type.\n");
			GTK_mutex.signal ();
			return (-1);
		}
	}

	/* Ajoute chaque configuration de source */
	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		MIXP_get_source_track (preset_nbr, src_cnt, &track_conf);
		if (   src_cnt > 0
		    || (flags & PLAY_MIX_PRESET_MASK) == PLAY_MIX_PRESET_ADD)
		{
			PLAY_add_source_track (track_type, track_nbr, &track_conf, flags);
		}
		else
		{
			track_conf_ptr [0] = track_conf;
		}
	}
	if (track_type == GTK_TRACK_TYPE_AOU)
	{
		PLAY_update_aou_source_tracks_with_aou_source_conf (track_nbr);
	}
	
	/* Initialise les parametres propres a la piste */
	if ((flags & PLAY_MIX_PRESET_NO_DEST) == 0)
	{
		if ((flags & PLAY_MIX_PRESET_NO_VOL) == 0)
		{
			PLAY_set_lin_volume (track_info_ptr, MIXP_get_volume (preset_nbr, false));
			PLAY_set_dry_lin_volume (track_info_ptr, MIXP_get_volume (preset_nbr, true));
		}
		if ((flags & PLAY_MIX_PRESET_NO_PAN) == 0)
		{
			track_info_ptr->score.track.panning = (WORD) (MIXP_get_panning (preset_nbr, false) << 4);
			track_info_ptr->score.track.drypan = (WORD) (MIXP_get_panning (preset_nbr, true) << 4);
		}
		PLAY_set_track_stereo (track_type, track_nbr, MIXP_get_stereo (preset_nbr));
	}

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

signed int	PLAY_save_mix_preset (int track_type, int track_nbr, int preset_nbr)
{
	int		nbr_src;
	int		src_cnt;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type]
	                                        [track_nbr];

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (track_type, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_save_mix_preset: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	/* Fixe le nombre de sources */
	if (MIXP_set_nbr_source_tracks (preset_nbr, nbr_src))
	{
		LOG_printf ("PLAY_save_mix_preset: Error: couldn't change preset size.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	/* Recopie chaque configuration de source */
	if (track_type == GTK_TRACK_TYPE_AOU)
	{
		PLAY_update_aou_source_conf_with_aou_source_tracks (track_nbr);
	}
	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		MIXP_set_source_track (preset_nbr, src_cnt, track_conf_ptr + src_cnt);
	}
	
	MIXP_set_track_type (preset_nbr, track_type);
	MIXP_set_track_number (preset_nbr, track_nbr);
	MIXP_set_volume (preset_nbr, false, track_info_ptr->score.track.volume_lin);
	MIXP_set_volume (preset_nbr, true, track_info_ptr->score.track.dryvol_lin);
	MIXP_set_panning (preset_nbr, false, track_info_ptr->score.track.panning >> 4);
	MIXP_set_panning (preset_nbr, true, track_info_ptr->score.track.drypan >> 4);
	MIXP_set_stereo (preset_nbr, track_info_ptr->stereo);

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

signed int	PLAY_update_aou_source_tracks_with_aou_source_conf (int track_nbr)
{
	int		track;
	int		type;
	bool		dry_flag;
	int		src_cnt;
	int		nbr_src;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (GTK_TRACK_TYPE_AOU, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_update_aou_source_tracks_with_aou_source_conf: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		type = track_conf_ptr [src_cnt].track_type;
		track = track_conf_ptr [src_cnt].track_nbr;
		dry_flag = ! track_conf_ptr [src_cnt].wet_flag;

		PLAY_set_track_lin_volume (type, track, dry_flag,
		                           track_conf_ptr [src_cnt].inpvol);
		PLAY_set_track_panning (type, track, dry_flag,
		                        (UWORD)track_conf_ptr [src_cnt].inppan);
	}

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

signed int	PLAY_update_aou_source_conf_with_aou_source_tracks (int track_nbr)
{
	int		track;
	int		type;
	bool		dry_flag;
	int		src_cnt;
	int		nbr_src;
	PLAY_SOURCE_TRACK_CONF_BLOCK	*track_conf_ptr;

	GTK_mutex.wait ();

	/* Recueille les informations sur les sources */
	if (PLAY_get_track_nbr_s_tracks (GTK_TRACK_TYPE_AOU, track_nbr, &nbr_src, &track_conf_ptr))
	{
		LOG_printf ("PLAY_update_aou_source_conf_with_aou_source_tracks: Error: bad destination track type.\n");
		GTK_mutex.signal ();
		return (-1);
	}

	for (src_cnt = 0; src_cnt < nbr_src; src_cnt ++)
	{
		type = track_conf_ptr [src_cnt].track_type;
		track = track_conf_ptr [src_cnt].track_nbr;
		dry_flag = ! track_conf_ptr [src_cnt].wet_flag;

		track_conf_ptr [src_cnt].inpvol = PLAY_get_track_lin_volume (type, track, dry_flag);
		track_conf_ptr [src_cnt].inppan = PLAY_get_track_panning (type, track, dry_flag);
	}

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_name                                            */
/*      Description: Renvoie le nom d'une piste.                            */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*      Parametres en sortie:                                               */
/*        - name_0: pointeur sur la chaine contenant le nom de la piste,    */
/*                  termine par 0. L'espace doit etre deja reserve, et      */
/*                  suffisant pour accueillir la chaine.                    */
/*==========================================================================*/

void	PLAY_get_track_name (int track_type, int track_nbr, char *name_0)
{
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type]
	                                        [track_nbr];
	BASE_copy_string (track_info_ptr->name, name_0,
							TRK_NAME_LEN, TRK_NAME_LEN);
	name_0 [TRK_NAME_LEN] = 0;
}



/*==========================================================================*/
/*      Nom: PLAY_set_track_name                                            */
/*      Description: Nomme une piste.                                       */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - name_0: pointeur sur la chaine contenant le nom de la piste,    */
/*                  termine par 0.                                          */
/*==========================================================================*/

void	PLAY_set_track_name (int track_type, int track_nbr, const char *name_0)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type]
	                                        [track_nbr];
	BASE_copy_string (name_0, track_info_ptr->name,
	                  strlen (name_0), TRK_NAME_LEN);
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_onoff                                           */
/*      Description: Renvoie l'etat d'activite d'une piste.                 */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste demandee.                          */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie dry.           */
/*      Retour: true si la piste est active.                                */
/*==========================================================================*/

bool	PLAY_get_track_onoff (int track_type, int track_nbr, bool dry_flag)
{
	if (dry_flag)
	{
		return (! PLAY_track_info [PLAY_track_info_list [track_type] [track_nbr]].dry_mute_flag);
	}
	
	return (! PLAY_track_info [PLAY_track_info_list [track_type] [track_nbr]].wet_mute_flag);
}



/*==========================================================================*/
/*      Nom: PLAY_set_track_onoff                                           */
/*      Description: Fixe l'etat d'activite d'une piste.                    */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie dry.           */
/*        - state: true si la piste doit etre active, false si elle doit    */
/*                 etre coupee.                                             */
/*==========================================================================*/

void	PLAY_set_track_onoff (int track_type, int track_nbr, bool dry_flag, bool state)
{
	if (dry_flag)
	{
		PLAY_track_info [PLAY_track_info_list [track_type] [track_nbr]].dry_mute_flag = ! state;
	}
	else
	{
		PLAY_track_info [PLAY_track_info_list [track_type] [track_nbr]].wet_mute_flag = ! state;
	}
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_panning                                         */
/*      Description: Renvoie la balance d'une piste.                        */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie dry.           */
/*      Retour: la balance.                                                 */
/*==========================================================================*/

LWORD	PLAY_get_track_panning (int track_type, int track_nbr, bool dry_flag)
{
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];

	if (dry_flag)
	{
		return (track_info_ptr->score.track.drypan);
	}

	return (track_info_ptr->score.track.panning);
}



/*==========================================================================*/
/*      Nom: PLAY_set_track_panning                                         */
/*      Description: Fixe la balance d'une piste.                           */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie wet.           */
/*        - pan: Balance (0000...FFFF).                                     */
/*==========================================================================*/

void	PLAY_set_track_panning (int track_type, int track_nbr, bool dry_flag, LWORD pan)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];
	PLAY_set_panning (track_info_ptr, dry_flag, pan);
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_lin_volume                                      */
/*      Description: Renvoie le volume lineaire d'une piste specifiee.      */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie wet.           */
/*      Retour: Le volume (0...1000...FFFF)                                 */
/*==========================================================================*/

LWORD	PLAY_get_track_lin_volume (int track_type, int track_nbr, bool dry_flag)
{
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];

	if (dry_flag)
	{
		return (track_info_ptr->score.track.dryvol_lin);
	}

	return (track_info_ptr->score.track.volume_lin);
}



/*==========================================================================*/
/*      Nom: PLAY_set_track_lin_volume                                      */
/*      Description: Fixe le volume lineaire d'une piste specifiee.         */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie dry.           */
/*        - volume: le volume (0...1000...FFFF)                             */
/*==========================================================================*/

void	PLAY_set_track_lin_volume (int track_type, int track_nbr, bool dry_flag, LWORD volume)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];
	if (dry_flag)
	{
		PLAY_set_dry_lin_volume (track_info_ptr, volume);
	}
	else
	{
		PLAY_set_lin_volume (track_info_ptr, volume);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_log_volume                                      */
/*      Description: Renvoie le volume logarithmique d'une piste specifiee  */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie wet.           */
/*      Retour: Le volume (0...C00...FFF)                                   */
/*==========================================================================*/

LWORD	PLAY_get_track_log_volume (int track_type, int track_nbr, bool dry_flag)
{
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];

	if (dry_flag)
	{
		return (track_info_ptr->score.track.dryvol_log);
	}

	return (track_info_ptr->score.track.volume_log);
}



/*==========================================================================*/
/*      Nom: PLAY_set_track_log_volume                                      */
/*      Description: Fixe le volume logarithmique d'une piste specifiee     */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie dry.           */
/*        - volume: le volume (0...C00...FFF)                               */
/*==========================================================================*/

void	PLAY_set_track_log_volume (int track_type, int track_nbr, bool dry_flag, LWORD volume)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];
	if (dry_flag)
	{
		PLAY_set_dry_log_volume (track_info_ptr, volume);
	}
	else
	{
		PLAY_set_log_volume (track_info_ptr, volume);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_lin_to_log_volume                                         */
/*      Description: Convertit un volume de piste lineaire en volume log.   */
/*      Parametres en entree:                                               */
/*        - lin_vol: volume lineaire (0...1000...FFFF)                      */
/*      Retour: volume logarithmique correspondant (0...C00...FFF).         */
/*==========================================================================*/

LWORD	PLAY_lin_to_log_volume (LWORD lin_vol)
{
	LWORD		log_vol;

	if (lin_vol == 0)
	{
		log_vol = 0;
	}

	else
	{
		/* Recherche de la partie entiere de la puissance de 2 */
		log_vol = 0x1000;
		do
		{
			log_vol -= 0x100;
			lin_vol <<= 1;
		}
		while ((lin_vol & 0x10000) == 0);
		lin_vol = (lin_vol >> 8) & 0xFF;

		/* Recherche de la partie fractionnaire */
		log_vol += (int) PLAY_track_vol_table_log [lin_vol];
	}

	return (log_vol);
}



/*==========================================================================*/
/*      Nom: PLAY_log_to_lin_volume                                         */
/*      Description: Convertit un volume de piste log en volume lineaire.   */
/*      Parametres en entree:                                               */
/*        - lin_vol: volume logarithmique (0...C00...FFF)                   */
/*      Retour: volume lineaire correspondant (0...1000...FFFF).            */
/*==========================================================================*/

LWORD	PLAY_log_to_lin_volume (LWORD log_vol)
{
	LWORD		lin_vol;

	if (log_vol == 0)
	{
		lin_vol = 0;
	}

	else
	{
		lin_vol =    PLAY_track_vol_table_exp [log_vol & 0xFF]
					 >> (0xF - (log_vol >> 8));
	}

	return (lin_vol);
}



/*==========================================================================*/
/*      Nom: PLAY_get_track_stereo                                          */
/*      Description: Renvoie la stereo d'une piste donnee.                  */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - dry_flag: indique qu'on veut manipuler la partie dry.           */
/*      Retour: Nombre de voies de la piste.                                */
/*==========================================================================*/

int	PLAY_get_track_stereo (int track_type, int track_nbr, bool dry_flag)
{
	const PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];

	if (dry_flag && track_type == Pattern_TYPE_FX)
	{
		return (2);	/* Les pistes FX ont toujours leur dry en stereo. */
	}

	return (track_info_ptr->stereo);
}



/*==========================================================================*/
/*      Nom: PLAY_set_track_stereo                                          */
/*      Description: Fixe la stereo d'une piste si c'est possible. Pour les */
/*                   pistes de FX, c'est toujours le wet.                   */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  */
/*        - stereo: nombre de voies demandees.                              */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si on n'a pas pu changer la stereo,                       */
/*              -1 si erreur.                                               */
/*==========================================================================*/

signed int	PLAY_set_track_stereo (int track_type, int track_nbr, int stereo)
{
	PLAY_TRACK_INFO	*track_info_ptr;
	signed int	ret_val;

	if (stereo < 1)
	{
		return (1);
	}

	GTK_mutex.wait ();

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type] [track_nbr];

	ret_val = 0;
	switch (track_type)
	{
	case	Pattern_TYPE_SPL:		// Ne peut etre change
	case	Pattern_TYPE_AIN:		// A voir
	case	Pattern_TYPE_MID:		// Ne peut etre change
	case	GTK_TRACK_TYPE_AOU:	// A voir
		ret_val = 1;
		break;

	case	Pattern_TYPE_FX:
		ret_val = PLAYFX_fx_set_stereo (track_info_ptr, stereo);
		if (ret_val < 0)
		{
			LOG_printf ("PLAY_set_track_stereo: Error: failed to change stereo of FX track # %d.\n", track_nbr);
			ret_val = -1;
		}
		else
		{
			ret_val = (ret_val != stereo) ? 1 : 0;
		}
		break;
	}

	GTK_mutex.signal ();

	return (ret_val);
}



/*==========================================================================*/
/*      Nom: PLAY_set_nbr_source_tracks                                     */
/*      Description: Fixe le nouveau nombre de pistes sources d'une piste.  */
/*                   Les configurations doivent etre mises a jour           */
/*                   separement.                                            */
/*      Parametres en entree:                                               */
/*        - track_type: Type de la piste.                                   */
/*        - track_nbr: Numero de la piste.                                  S*/
/*        - nbr_src: nouveau nombre de pistes sources > 0.                  */
/*      Retour: 0 si tout s'est bien passe, ou -1 si le type de la piste ne */
/*              correspondait pas a un type utilisant des pistes sources.   */
/*==========================================================================*/

signed int	PLAY_set_nbr_source_tracks (int track_type, int track_nbr, int nbr_src)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [track_type]
	                                        [track_nbr];

	if (track_type == GTK_TRACK_TYPE_AOU)
	{
		track_info_ptr->mix.src.nbr_s_tracks = nbr_src;
	}
	else if (track_type == Pattern_TYPE_FX)
	{
		track_info_ptr->mix.src.nbr_s_tracks = nbr_src;
	}
	else
	{
		/* Mauvais type destination */
		assert (false);
		return (-1);
	}

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_get_tempo                                                 */
/*      Description: Renvoie le tempo du module.                            */
/*      Retour: Le tempo, en BPM.                                           */
/*==========================================================================*/

double	PLAY_get_tempo (void)
{
	return (PLAY_score_info.tempo_int + (double)PLAY_score_info.tempo_frac / 65536);
}



/*==========================================================================*/
/*      Nom: PLAY_set_tempo                                                 */
/*      Description: Fixe le tempo du module.                               */
/*      Parametres en entree:                                               */
/*        - tempo: tempo en BPM (32 a 999).                                 */
/*==========================================================================*/

void	PLAY_set_tempo (double tempo)
{
	int		tempo_int;

	GTK_mutex.wait ();

	tempo = MIN (tempo, PLAY_MAX_TEMPO);
	tempo = MAX (tempo, PLAY_MIN_TEMPO);
	tempo_int = (int) floor (tempo + 0.5 / 65536);
	PLAY_score_info.tempo_int = tempo_int;
	PLAY_score_info.tempo_frac = (UWORD) (floor ((tempo - tempo_int) * 65536 + 0.5));
	SimpleCalculator::set_spec_var (SimpleCalculator::TEMPO, tempo);

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_get_speed                                                 */
/*      Description: Renvoie la vitesse courante du module.                 */
/*      Retour: Vitesse                                                     */
/*==========================================================================*/

int	PLAY_get_speed (void)
{
	return (PLAY_score_info.speed);
}



/*==========================================================================*/
/*      Nom: PLAY_set_speed                                                 */
/*      Description: Fixe la vitesse courante du module                     */
/*      Parametres en entree:                                               */
/*        - speed: vitesse (1 - 255).                                       */
/*==========================================================================*/

void	PLAY_set_speed (int speed)
{
	GTK_mutex.wait ();

	speed = MIN (speed, PLAY_MAX_SPEED);
	speed = MAX (speed, PLAY_MIN_SPEED);
	PLAY_score_info.speed = speed;

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_get_lin_master_volume                                     */
/*      Description: Renvoie le master volume lineaire.                     */
/*      Retour: volume lineaire (0...FFF).                                  */
/*==========================================================================*/

LWORD	PLAY_get_lin_master_volume (void)
{
	LWORD		vol;
	const PLAY_TRACK_INFO	*aou_track_info_ptr;

	aou_track_info_ptr =   PLAY_track_info
	                     + PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [0];
	vol = (LWORD)aou_track_info_ptr->score.track.volume_lin;
	vol = MIN (vol, 0x0FFF);
	vol = MAX (vol, 0);

	return (vol);
}



/*==========================================================================*/
/*      Nom: PLAY_set_lin_master_volume                                     */
/*      Description: Fixe le master volume lineaire. En fait il s'agit      */
/*                   de fixer les volumes de toutes les voies Audio Out au  */
/*                   meme niveau.                                           */
/*      Parametres en entree:                                               */
/*        - vol: Le nouveau master volume.                                  */
/*==========================================================================*/

void	PLAY_set_lin_master_volume (LWORD vol)
{
	int		track;
	int		global_track_nbr;
	PLAY_TRACK_INFO	*aou_track_info_ptr;

	vol = MIN (vol, 0x0FFF);
	vol = MAX (vol, 0);

	for (track = 0; track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [track];
		aou_track_info_ptr = PLAY_track_info + global_track_nbr;
		PLAY_set_lin_volume (aou_track_info_ptr, vol);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_get_log_master_volume                                     */
/*      Description: Renvoie le master volume logarithmique.                */
/*      Retour: volume logarithmique (0...800).                             */
/*==========================================================================*/

LWORD	PLAY_get_log_master_volume (void)
{
	LWORD		vol;
	const PLAY_TRACK_INFO	*aou_track_info_ptr;

	aou_track_info_ptr =   PLAY_track_info
	                     + PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [0];
	vol = (LWORD)aou_track_info_ptr->score.track.volume_log + 0x0400;
	vol = MIN (vol, 0x13FF);
	vol = MAX (vol, 0);

	return (vol);
}



/*==========================================================================*/
/*      Nom: PLAY_set_log_master_volume                                     */
/*      Description: Fixe le master volume logarithmique. En fait il s'agit */
/*                   de fixer les volumes de toutes les voies Audio Out au  */
/*                   meme niveau.                                           */
/*      Parametres en entree:                                               */
/*        - vol: Le nouveau master volume.                                  */
/*==========================================================================*/

void	PLAY_set_log_master_volume (LWORD vol)
{
	int		track;
	int		global_track_nbr;
	PLAY_TRACK_INFO	*aou_track_info_ptr;

	vol -= 0x0400;
	vol = MIN (vol, 0x1000);
	vol = MAX (vol, 0);

	for (track = 0; track < GTK_nbr_tracks [GTK_TRACK_TYPE_AOU]; track ++)
	{
		global_track_nbr = PLAY_track_info_list [GTK_TRACK_TYPE_AOU] [track];
		aou_track_info_ptr = PLAY_track_info + global_track_nbr;
		PLAY_set_log_volume (aou_track_info_ptr, vol);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_get_interpolation_flag                                    */
/*      Description: Renvoie l'etat d'interpolation d'une piste de samples. */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste.                                      */
/*      Retour: true s'il y a interpolation.                                */
/*==========================================================================*/

bool	PLAY_get_interpolation_flag (int track)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr = PLAY_track_info + PLAY_track_info_list [Pattern_TYPE_SPL] [track];
	return (track_info_ptr->mix.spl.interpol_flag);
}



/*==========================================================================*/
/*      Nom: PLAY_set_interpolation_flag                                    */
/*      Description: Fixe l'etat d'interpolation d'un piste de samples.     */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste.                                      */
/*        - interpol_flag: true si on doit interpoler les samples.          */
/*==========================================================================*/

void	PLAY_set_interpolation_flag (int track, bool interpol_flag)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	track_info_ptr = PLAY_track_info + PLAY_track_info_list [Pattern_TYPE_SPL] [track];
	track_info_ptr->mix.spl.interpol_flag = interpol_flag;
}



/*==========================================================================*/
/*      Nom: PLAY_no_mute                                                   */
/*      Description: Active toutes les voies.                               */
/*==========================================================================*/

void	PLAY_no_mute (void)
{
	int		track_type;
	int		track;

	for (track_type = 0; track_type < Pattern_NBR_TYPES; track_type ++)
	{
		for (track = 0; track < GTK_nbr_tracks [track_type]; track ++)
		{
			PLAY_set_track_onoff (track_type, track, false, true);
			PLAY_set_track_onoff (track_type, track, true, true);
		}
	}
}



/*==========================================================================*/
/*      Nom: PLAY_invert_all_mute                                           */
/*      Description: Inverse l'etat muet/actif de toutes les voies.         */
/*==========================================================================*/

void	PLAY_invert_all_mute (void)
{
	int		track_type;
	int		track;
	bool		active_flag;

	for (track_type = 0; track_type < Pattern_NBR_TYPES; track_type ++)
	{
		for (track = 0; track < GTK_nbr_tracks [track_type]; track ++)
		{
			active_flag = ! PLAY_get_track_onoff (track_type, track, false);
			PLAY_set_track_onoff (track_type, track, false, active_flag);
			PLAY_set_track_onoff (track_type, track, true, active_flag);
		}
	}
}



/*==========================================================================*/
/*      Nom: PLAY_default_interpolation                                     */
/*      Description: Remet par defaut l'interpolation de toutes les voies.  */
/*==========================================================================*/

void	PLAY_default_interpolation (void)
{
	int		track;

	for (track = 0; track < GTK_nbr_tracks [Pattern_TYPE_SPL]; track ++)
	{
		PLAY_set_interpolation_flag (track, PLAY_default_interpolation_flag);
	}
}



/*==========================================================================*/
/*      Nom: PLAY_get_clipping_level                                        */
/*      Description: Renvoie le niveau de clipping d'une piste AudioOut.    */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste de sortie.                            */
/*      Retour: Niveau de clipping. Si la piste a clippe, la valeur sort du */
/*              domaine representable par 24 bits signes.                   */
/*==========================================================================*/

SLWORD	PLAY_get_clipping_level (int track)
{
	return (PLAY_clipping_level [track]);
}



/*==========================================================================*/
/*      Nom: PLAY_set_clipping_level                                        */
/*      Description: Fixe le niveau de clipping (il faut d'abord l'avoir    */
/*                   mesure).                                               */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste de sortie.                            */
/*        - level: niveau de clipping de la piste (positif). Si la valeur   */
/*                 n'est pas representable sur 24 bits signes, la piste a   */
/*                 clippe.                                                  */
/*==========================================================================*/

void	PLAY_set_clipping_level (int track, SLWORD level)
{
	PLAY_clipping_level [track] = level;
}




/*==========================================================================*/
/*      Nom: PLAY_get_max_clipping_level                                    */
/*      Description: Renvoie le niveau de clipping maximum atteint depuis   */
/*                   la derniere reinitialisation du niveau.                */
/*      Retour: Niveau de clipping. Si la piste a clippe, la valeur sort du */
/*              domaine representable par 24 bits signes.                   */
/*==========================================================================*/

SLWORD	PLAY_get_max_clipping_level (void)
{
	return (PLAY_max_clipping_level);
}



/*==========================================================================*/
/*      Nom: PLAY_set_max_clipping_level                                    */
/*      Description: Fixe le niveau de clipping maximum. Il faut d'abord    */
/*                   avoir mesure le clipping courant et compare avec le    */
/*                   clipping maximum precedent.                            */
/*      Parametres en entree:                                               */
/*        - level: niveau de clipping de la piste (positif). Si la valeur   */
/*                 n'est pas representable sur 24 bits signes, la piste a   */
/*                 clippe.                                                  */
/*                 Une valeur de 0 sigale un reset du niveau                */
/*==========================================================================*/

void	PLAY_set_max_clipping_level (SLWORD level)
{
	PLAY_max_clipping_level = level;
}



/*==========================================================================*/
/*      Nom: PLAY_get_bar_time                                              */
/*      Description: Renvoie la mesure de la partition                      */
/*      Parametres en sortie:                                               */
/*        - high: nombre de temps dans une mesure.                          */
/*        - low: subdivision de la ronde donnant le temps (puissance de 2)  */
/*==========================================================================*/

void	PLAY_get_bar_time (int &high, int &low)
{
	GTK_mutex.wait ();

	high = PLAY_score_info.time_high;
	low = PLAY_score_info.time_low;

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_set_bar_time                                              */
/*      Description: Fixe le temps de la mesure. Les parametres sont        */
/*                   changes que s'ils sont positifs.                       */
/*      Parametres en entree:                                               */
/*        - high: nombre de temps dans une mesure.                          */
/*        - low: subdivision de la ronde donnant le temps (doit etre une    */
/*               puissance de 2).                                           */
/*==========================================================================*/

void	PLAY_set_bar_time (int high, int low)
{
	GTK_mutex.wait ();

	if (high > 0)
	{
		PLAY_score_info.time_high = high;
	}
	if (low > 0)
	{
		PLAY_score_info.time_low = low;
	}

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_get_bar_length                                            */
/*      Description: Donne le nombre de lignes par mesure.                  */
/*      Retour: nombre de lignes.                                           */
/*==========================================================================*/

int	PLAY_get_bar_length (void)
{
	int		time_hi;
	int		time_lo;

	PLAY_get_bar_time (time_hi, time_lo);

	return ((time_hi << 4) / time_lo);
}



/*==========================================================================*/
/*      Nom: PLAY_get_fx_preset                                             */
/*      Description: Memorise un effet d'une piste dans un preset.          */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste d'effet de laquelle on veut extraire  */
/*                 l'effet.                                                 */
/*        - preset: numero du preset d'effet dans lequel l'effet sera       */
/*                  memorise.                                               */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si le player n'etait pas en marche,                       */
/*              2 si la piste n'existait pas,                               */
/*              3 si le preset n'existait pas.                              */
/*==========================================================================*/

signed int	PLAY_get_fx_preset (int track, int preset)
{
	int		type;
	FxPreset_EFFECT_CONF	temp_conf;

	if (! PLAY_player_started_flag)
	{
		return (1);
	}
	else if (track >= GTK_nbr_tracks [Pattern_TYPE_FX])
	{
		return (2);
	}
	else if (preset > GTK_NBRFXP_TOTAL)
	{
		return (3);
	}

	GTK_mutex.wait ();

	PLAYFX_fx_get_preset (  PLAY_track_info
	                      + PLAY_track_info_list [Pattern_TYPE_FX] [track],
	                      temp_conf,
	                      type);
	FXP_set_effect_type (preset, type);
	FXP_set_parameters (preset, temp_conf);
	
	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom: PLAY_set_fx_preset                                             */
/*      Description: Applique un preset d'effet a une piste.                */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste d'effet sur laquelle on veut          */
/*                 appliquer l'effet.                                       */
/*        - preset: numero du preset d'effet a appliquer.                   */
/*        - auto_val: champ de bit precisant quelles variables temps reel   */
/*                    doivent etre explicitement modifiees lors d'une mise  */
/*                    a jour d'un effet. Les autres valeurs modifiees       */
/*                    automatiquement par l'effet en temps reel sont        */
/*                    recopiee a partir du preset precedant.                */
/*      Retour: 0 si tout s'est bien passe,                                 */
/*              1 si le player n'etait pas en marche,                       */
/*              2 si la piste n'existait pas,                               */
/*==========================================================================*/

signed int	PLAY_set_fx_preset (int track, int preset, long auto_val)
{
	if (! PLAY_player_started_flag)
	{
		return (1);
	}
	else if (track >= GTK_nbr_tracks [Pattern_TYPE_FX])
	{
		return (2);
	}

	GTK_mutex.wait ();

	PLAYFX_fx_set_preset (  PLAY_track_info
	                      + PLAY_track_info_list [Pattern_TYPE_FX] [track],
	                      FXP_get_parameters (preset),
	                      FXP_get_effect_type (preset), auto_val);

	GTK_mutex.signal ();

	return (0);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

bool	PLAY_get_bypass_flag (int track)
{
	bool		bypass_flag;
	PLAY_TRACK_INFO	*track_info_ptr;

	if (! PLAY_player_started_flag)
	{
		return (false);
	}
	else if (track >= GTK_nbr_tracks [Pattern_TYPE_FX])
	{
		return (false);
	}

	GTK_mutex.wait ();

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [Pattern_TYPE_FX] [track];
	bypass_flag = track_info_ptr->mix.fx.bypass_flag;
	
	GTK_mutex.signal ();

	return (bypass_flag);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	PLAY_set_bypass_flag (int track, bool bypass_flag)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	if (! PLAY_player_started_flag)
	{
		return;
	}
	else if (track >= GTK_nbr_tracks [Pattern_TYPE_FX])
	{
		return;
	}

	GTK_mutex.wait ();

	track_info_ptr =   PLAY_track_info
	                 + PLAY_track_info_list [Pattern_TYPE_FX] [track];
	track_info_ptr->mix.fx.bypass_flag = bypass_flag;
	
	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_play_sound                                                */
/*      Description: Joue un son sur une piste de sample donnee.            */
/*      Parametres en entree:                                               */
/*        - track: numero de la piste sur laquelle on veut jouer le son.    */
/*        - sample: numero du sample a jouer (peut etre le sample           */
/*                  temporaire).                                            */
/*        - start_pos: position de depart de replay (en samples).           */
/*==========================================================================*/

void	PLAY_play_sound (int track, int sample, long start_pos)
{
	PLAY_TRACK_INFO	*track_info_ptr;

	GTK_mutex.wait ();

	track_info_ptr = PLAY_track_info + PLAY_track_info_list [Pattern_TYPE_SPL] [track];

	PLAY_set_lin_velocity (track_info_ptr, 0x800);
	track_info_ptr->score.sample.number = sample;
	track_info_ptr->score.sample.transp = 0;
	track_info_ptr->score.sample.volume = 0x100;
	track_info_ptr->score.sample.finetune = 0;
	track_info_ptr->stereo = SAMP_get_sample_stereo (sample);

	track_info_ptr->score.note = Sample_REF_C2;
	track_info_ptr->score.porta.note = Sample_REF_C2;
	track_info_ptr->score.period = (float) PLAY_convert_note_to_internal_period (Sample_REF_C2, 0);
	track_info_ptr->score.porta.period = track_info_ptr->score.period;

	track_info_ptr->mix.spl.sample_ptr = SAMP_get_sample_data_adr (sample);
	track_info_ptr->mix.spl.loopbuf_ptr = SAMP_get_sample_loopbuf_adr (sample);
	track_info_ptr->mix.spl.loopbuf_len = SAMP_get_sample_loopbuf (sample);
	track_info_ptr->mix.spl.resol = SAMP_get_sample_resolution (sample) >> 3;
	track_info_ptr->mix.spl.tracks = SAMP_get_sample_stereo (sample);
	track_info_ptr->mix.spl.freq = SAMP_get_sample_freq (sample);
	track_info_ptr->mix.spl.loopmode = (WORD) SAMP_get_sample_loop (sample);

	/* Pas de bouclage */
	if (SAMP_get_sample_loop (sample) == WaveForm_LOOP_TYPE_NONE)
	{
		track_info_ptr->mix.spl.reppos = SAMP_get_sample_length (sample);
		track_info_ptr->mix.spl.replen = 0;
	}

	/* Bouclage */
	else
	{
		track_info_ptr->mix.spl.reppos = SAMP_get_sample_repeat (sample);
		track_info_ptr->mix.spl.replen = SAMP_get_sample_replen (sample);
	}

	/* Direct-2-Disk */
	if (SAMP_get_sample_type (sample) == 1)
	{
		track_info_ptr->mix.spl.d2d_flag = true;
		track_info_ptr->mix.spl.file_ptr = SAMP_get_sample_file_ptr (sample);
		track_info_ptr->mix.spl.file_data_offset = SAMP_get_sample_file_data_offset (sample);;
		track_info_ptr->mix.spl.startbuf_ptr = SAMP_get_sample_startbuf_ptr (sample);
		track_info_ptr->mix.spl.startbuf_len = SAMP_get_sample_buffer_len (sample, 0);
		track_info_ptr->mix.spl.midbuf1_len = SAMP_get_sample_buffer_len (sample, 1);
		track_info_ptr->mix.spl.midbuf2_len = SAMP_get_sample_buffer_len (sample, 2);
	}

	else
	{
		track_info_ptr->mix.spl.d2d_flag = false;
	}

	track_info_ptr->mix.spl.direction = 0;
	PLAYENV_init_all_envelopes (track_info_ptr, -1);

	track_info_ptr->mix.spl.curpos_int = start_pos;
	track_info_ptr->mix.spl.curpos_frac = 0;

	GTK_mutex.signal ();
}



/*==========================================================================*/
/*      Nom: PLAY_play_pattern_line_spl                                     */
/*      Description: Joue des notes sur certaines voies, a n'importe quel   */
/*                   moment. Les notes sont placees dans la variable        */
/*                   PLAY_new_spl_notes adequate et le flag                 */
/*                   PLAY_new_line_flag [???] sera remis a 0 au moment ou   */
/*                   la nouvelle ligne sera prise en compte, ainsi que les  */
/*                   PLAY_new_note flag [???] [] correspondants.            */
/*==========================================================================*/

void	PLAY_play_pattern_line_spl (void)
{
	int					track;
	int					track_max;
	PLAY_TRACK_INFO	*track_info_ptr;
	UBYTE					*note_ptr;
	int					*track_info_list_ptr;

	note_ptr = (UBYTE *)PLAY_new_spl_notes;
	track_max = GTK_nbr_tracks [Pattern_TYPE_SPL];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_SPL];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		if (PLAY_new_note_flag [Pattern_TYPE_SPL] [track] != 0)
		{
			track_info_ptr->score.score.note = *note_ptr;
			track_info_ptr->score.score.instr = *(note_ptr + 1);
			track_info_ptr->score.score.effect = BASE_dpeek_moto (note_ptr + 2);
			track_info_ptr->score.score.volume = *(note_ptr + 4);
			PLAYCMD_manage_effects_1st_tick_spl (track);
			PLAYENV_manage_envelope (Pattern_TYPE_SPL, track, Envelope_TYPE_VOL);
			PLAYENV_manage_envelope (Pattern_TYPE_SPL, track, Envelope_TYPE_TON);
			PLAYENV_manage_envelope (Pattern_TYPE_SPL, track, Envelope_TYPE_PAN);
			PLAY_compute_final_parameters_track (Pattern_TYPE_SPL, track);
			PLAY_new_note_flag [Pattern_TYPE_SPL] [track] = 0;
		}
		note_ptr += MODS_GT2_SPL_NOTE_LEN;
	}

	PLAY_new_line_flag [Pattern_TYPE_SPL] = 0;
}



/*==========================================================================*/
/*      Nom: PLAY_play_pattern_line_ain                                     */
/*      Description: Joue des notes sur certaines voies, a n'importe quel   */
/*                   moment. Les notes sont placees dans la variable        */
/*                   PLAY_new_ain_notes adequate et le flag                 */
/*                   PLAY_new_line_flag [???] sera remis a 0 au moment ou   */
/*                   la nouvelle ligne sera prise en compte, ainsi que les  */
/*                   PLAY_new_note flag [???] [] correspondants.            */
/*==========================================================================*/

void	PLAY_play_pattern_line_ain (void)
{
	int					track;
	int					track_max;
	PLAY_TRACK_INFO	*track_info_ptr;
	UBYTE					*note_ptr;
	int					*track_info_list_ptr;

	note_ptr = (UBYTE *)PLAY_new_ain_notes;
	track_max = GTK_nbr_tracks [Pattern_TYPE_AIN];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_AIN];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		if (PLAY_new_note_flag [Pattern_TYPE_AIN] [track] != 0)
		{
			track_info_ptr->score.score.effect = BASE_dpeek_moto (note_ptr);
			track_info_ptr->score.score.volume = *(note_ptr + 2);
			PLAYCMD_manage_effects_1st_tick_ain (track);
			PLAYENV_manage_envelope (Pattern_TYPE_AIN, track, Envelope_TYPE_VOL);
			PLAYENV_manage_envelope (Pattern_TYPE_AIN, track, Envelope_TYPE_PAN);
			PLAY_compute_final_parameters_track (Pattern_TYPE_AIN, track);
			PLAY_new_note_flag [Pattern_TYPE_AIN] [track] = 0;
		}
		note_ptr += MODS_GT2_AIN_NOTE_LEN;
	}

	PLAY_new_line_flag [Pattern_TYPE_AIN] = 0;
}



/*==========================================================================*/
/*      Nom: PLAY_play_pattern_line_fx                                      */
/*      Description: Joue des notes sur certaines voies, a n'importe quel   */
/*                   moment. Les notes sont placees dans la variable        */
/*                   PLAY_new_fx_notes adequate et le flag                  */
/*                   PLAY_new_line_flag [???] sera remis a 0 au moment ou   */
/*                   la nouvelle ligne sera prise en compte, ainsi que les  */
/*                   PLAY_new_note flag [???] [] correspondants.            */
/*==========================================================================*/

void	PLAY_play_pattern_line_fx (void)
{
	int					track;
	int					track_max;
	PLAY_TRACK_INFO	*track_info_ptr;
	UBYTE					*note_ptr;
	int					*track_info_list_ptr;

	note_ptr = (UBYTE *)PLAY_new_fx_notes;
	track_max = GTK_nbr_tracks [Pattern_TYPE_FX];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_FX];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		if (PLAY_new_note_flag [Pattern_TYPE_FX] [track] != 0)
		{
			track_info_ptr->score.score.fx_data = BASE_lpeek_moto (note_ptr);
			track_info_ptr->score.score.effect = BASE_dpeek_moto (note_ptr + 4);
			track_info_ptr->score.score.volume = *(note_ptr + 6);
			PLAYCMD_manage_effects_1st_tick_fx (track);
			PLAYENV_manage_envelope (Pattern_TYPE_FX, track, Envelope_TYPE_VOL);
			PLAYENV_manage_envelope (Pattern_TYPE_SPL, track, Envelope_TYPE_TON);
			PLAYENV_manage_envelope (Pattern_TYPE_FX, track, Envelope_TYPE_PAN);
			PLAY_compute_final_parameters_track (Pattern_TYPE_FX, track);
			PLAY_new_note_flag [Pattern_TYPE_FX] [track] = 0;
		}
		note_ptr += MODS_GT2_FX_NOTE_LEN;
	}

	PLAY_new_line_flag [Pattern_TYPE_FX] = 0;
}



/*==========================================================================*/
/*      Nom: PLAY_play_pattern_line_mid                                     */
/*      Description: Joue des notes sur certaines voies, a n'importe quel   */
/*                   moment. Les notes sont placees dans la variable        */
/*                   PLAY_new_mid_notes adequate et le flag                 */
/*                   PLAY_new_line_flag [???] sera remis a 0 au moment ou   */
/*                   la nouvelle ligne sera prise en compte, ainsi que les  */
/*                   PLAY_new_note flag [???] [] correspondants.            */
/*==========================================================================*/

void	PLAY_play_pattern_line_mid (void)
{
	int					track;
	int					track_max;
	PLAY_TRACK_INFO	*track_info_ptr;
	UBYTE					*note_ptr;
	int					*track_info_list_ptr;

	note_ptr = (UBYTE *)PLAY_new_ain_notes;
	track_max = GTK_nbr_tracks [Pattern_TYPE_MID];
	track_info_list_ptr = PLAY_track_info_list [Pattern_TYPE_MID];
	for (track = 0; track < track_max; track ++)
	{
		track_info_ptr = PLAY_track_info + *track_info_list_ptr++;
		if (PLAY_new_note_flag [Pattern_TYPE_MID] [track] != 0)
		{
			track_info_ptr->score.score.mid_data = BASE_lpeek_moto (note_ptr);
			PLAYCMD_manage_effects_1st_tick_mid (track);
			PLAY_new_note_flag [Pattern_TYPE_MID] [track] = 0;
		}
		note_ptr += MODS_GT2_MID_NOTE_LEN;
	}

	PLAY_new_line_flag [Pattern_TYPE_MID] = 0;
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

bool	PLAY_get_linear_period_flag (void)
{
	return (PLAY_score_info.linear_period_flag);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	PLAY_set_linear_period_flag (bool flag)
{
	GTK_mutex.wait ();

	PLAY_score_info.linear_period_flag = flag;

	GTK_mutex.signal ();
}



/****************************************************************************/
/*                                                                          */
/*      DIVERS                                                              */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom: PLAY_convert_freq_to_note                                      */
/*      Description: Convertit une frequence en une note midi + finetune.   */
/*      Parametres en entree:                                               */
/*        - freq: frequence (en Hz) a convertir en note.                    */
/*      Parametres en sortie:                                               */
/*        - finetune: erreur sur la conversion de la note, en 1/2 tons.     */
/*                    Toujours dans [-0.5; 0.5[.                            */
/*      Retour: La note au format MIDI. Le C2 de ref donne 48 et le La 440  */
/*              donne 69. La note peut sortir des intervales MIDI si la     */
/*              frequence est tres haute ou tres basse.                     */
/*==========================================================================*/

signed int	PLAY_convert_freq_to_note (double freq, double &finetune)
{
	signed int	note;
	double	real_note;

	real_note = log (freq / 440.0) * (12.0 / LN2) + Sample_REF_A440;
	note = (int) floor (real_note + 0.5);
	finetune = real_note - note;

	return (note);
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

double	PLAY_convert_note_to_freq (signed int note, double finetune)
{
	return (440.0 * pow (2.0, (note + finetune - Sample_REF_A440) * (1.0 / 12.0)));
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

/* per interne -> freq */
double	PLAY_convert_period_to_freq (float period)
{
	if (PLAY_get_linear_period_flag ())
	{
		return (440.0 * pow (2.0,   (  (double)PLAY_REF_LIN_PERIOD - period
		                             + ((Sample_REF_C2 - Sample_REF_A440) * PLAY_LIN_PER_NOTE))
		                          * (1.0 / (12*PLAY_LIN_PER_NOTE))));
	}
	else
	{
		return (  (PLAY_REF_PERIOD * PLAY_convert_note_to_freq (Sample_REF_C2, 0))	/*** A mettre en constante ***/
		        / (double)period);
	}
}


/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

/* Rel freq -> per interne */
double	PLAY_convert_rel_freq_to_period (double freq)
{
	if (PLAY_get_linear_period_flag ())
	{
		return (PLAY_REF_LIN_PERIOD - log (freq) * ((12*PLAY_LIN_PER_NOTE) / LN2));
	}
	else
	{
		return (PLAY_REF_PERIOD / freq);
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

/* per interne -> rel freq */
double	PLAY_convert_period_to_rel_freq (float period)
{
	if (PLAY_get_linear_period_flag ())
	{
		return (pow (2.0, ((double)PLAY_REF_LIN_PERIOD - period) * (1.0 / (12*PLAY_LIN_PER_NOTE))));
	}
	else
	{
		return ((double)PLAY_REF_PERIOD / period);
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

/* per interne -> Rel per */
double	PLAY_convert_period_to_rel_per (float period)
{
	if (PLAY_get_linear_period_flag ())
	{
		return (pow (2.0, (period - (double)PLAY_REF_LIN_PERIOD) * (1.0 / (12*PLAY_LIN_PER_NOTE))));
	}
	else
	{
		return (period / (double)PLAY_REF_PERIOD);
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

/* per interne -> per */
double	PLAY_convert_period_to_time (float period)
{
	if (PLAY_get_linear_period_flag ())
	{
		return (  (1.0 / 440.0)
		        * pow (2.0,   (  period - (double)PLAY_REF_LIN_PERIOD
		                       + ((Sample_REF_A440 - Sample_REF_C2) * PLAY_LIN_PER_NOTE))
		                    * (1.0 / (12*PLAY_LIN_PER_NOTE))));
	}
	else
	{
		return (  (double)period  
		        / (PLAY_REF_PERIOD * PLAY_convert_note_to_freq (Sample_REF_C2, 0)));
	}
}


/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

double	PLAY_convert_note_to_internal_period (int note, signed int finetune)
{
	if (PLAY_get_linear_period_flag ())
	{
		return (PLAY_REF_LIN_PERIOD - ((note - Sample_REF_C2) * PLAY_LIN_PER_NOTE) - (finetune * (PLAY_LIN_PER_NOTE >> 3)));
	}
	else
	{
		return (PLAY_note_period_table_ref [(note << 3) + finetune]);
	}
}




/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

