/*****************************************************************************

        MidiFile.h
        Author: Laurent de Soras, 2022

Simplified view of a SMF (standard MIDI file, .mid)

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



#pragma once
#if ! defined (MidiFile_HEADER_INCLUDED)
#define	MidiFile_HEADER_INCLUDED



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "MidiFileTrack.h"

#include <memory>
#include <optional>
#include <utility>
#include <vector>

#include <cstdint>



class MidiFile
{

/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

public:

	enum class Format
	{
		_unknown = -1,
		_single = 0,
		_multi,
		_patterns
	};

	class TimeSmpte
	{
	public:
		// SMPTE base framerate, in frames per second
		int            _fps_num = 0; // Framerate numerator
		int            _fps_den = 1; // Framerate denominator, > 0
	};

	class TimeInfo
	{
	public:
		// Number of ticks per time unit (quarter note or SMPTE frame)
		int            _nbr_ticks = 0;

		// Filled if the time unit is an SMPTE frame
		std::optional <TimeSmpte>
		               _smpte;
	};

	explicit       MidiFile (const uint8_t *mthd_ptr, const uint8_t *end_ptr, MidiFileTrack::NoteRepeat rep_strategy);

	Format         get_format () const noexcept;
	TimeInfo       get_time_info () const noexcept;
	int            get_nbr_tracks () const noexcept;
	const MidiFileTrack &
	               use_track (int trk_idx) const noexcept;



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

protected:



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	typedef std::unique_ptr <MidiFileTrack> TrackUPtr;
	typedef std::vector <TrackUPtr> TrackArray;

	uint8_t        read_byte (const uint8_t * &cur_ptr, const uint8_t *end_ptr);

	Format         _format = Format::_unknown;
	TimeInfo       _time_info;
	TrackArray     _track_arr;



/*\\\ FORBIDDEN MEMBER FUNCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	               MidiFile ()                               = delete;
	               MidiFile (const MidiFile &other)          = delete;
	               MidiFile (MidiFile &&other)               = delete;
	MidiFile &     operator = (const MidiFile &other)        = delete;
	MidiFile &     operator = (MidiFile &&other)             = delete;
	bool           operator == (const MidiFile &other) const = delete;
	bool           operator != (const MidiFile &other) const = delete;

};	// class MidiFile



//#include "MidiFile.hpp"



#endif	// MidiFile_HEADER_INCLUDED



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
