/*****************************************************************************

        MidiFileTrack.h
        Author: Laurent de Soras, 2022

Simplified view of a SMF (standard MIDI file, .mid) track

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



#pragma once
#if ! defined (MidiFileTrack_HEADER_INCLUDED)
#define	MidiFileTrack_HEADER_INCLUDED



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "MidiState.h"
#include "midi.h"

#include <array>
#include <map>
#include <vector>

#include <cstddef>
#include <cstdint>



class MidiFileTrack
{

/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

public:

	class Note
	{
	public:
		int32_t        _timestamp = -1;
		int32_t        _duration  = -1; // Negative means: note off not received yet
		uint8_t        _note      = 0;
		uint8_t        _velo      = 0;
	};

	enum class NoteRepeat
	{
		_cut = 0,
		_accumulate
	};

	class TimeSig
	{
	public:
		bool           is_defined () const noexcept { return (_num > 0); }
		int            _num = 0; // 0 if unspecified
		int            _den = 1;
	};

	typedef std::vector <Note> NoteArray;

	// <timestamp, content>
	typedef std::map <int32_t, TimeSig> TimeSigMap;
	typedef std::map <int32_t, double> TempoMap; // BPM (here, beat = quarter)

	explicit       MidiFileTrack (const uint8_t * &mtrk_ptr, const uint8_t *end_ptr, NoteRepeat rep_strategy);

	NoteArray      get_notes (int chn_idx) const;
	NoteArray      get_notes_all_chn () const;
	const TempoMap &
	               use_tempo_map () const noexcept;
	const TimeSigMap &
	               use_time_sig_map () const noexcept;

	static void    sort_notes (NoteArray &notes) noexcept;



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

protected:



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	enum class Meta : uint8_t
	{
		_seq_num   = 0x00,
		_text      = 0x01,
		_copyright = 0x02,
		_trk_name  = 0x03,
		_instr     = 0x04,
		_lyric     = 0x05,
		_marker    = 0x06,
		_cue_point = 0x07,
		_prog_name = 0x08,
		_dev_name  = 0x09,
		_trk_end   = 0x2F,
		_tempo     = 0x51,
		_smpte_ofs = 0x54,
		_time_sig  = 0x58,
		_key_sig   = 0x59,
		_prop_evt  = 0x7F
	};

	class Channel
	{
	public:
		bool           is_note_playing (uint8_t note) const noexcept;
		bool           is_any_note_playing () const noexcept;

		NoteArray      _note_arr;

		// The vector contains indexes in _note_arr of playing instances for
		// each note
		std::array <std::vector <int>, midi::_nbr_notes>
		               _note_alloc;
	};

	typedef std::array <Channel, midi::_nbr_chn> ChannelArray;

	typedef std::vector <uint8_t> ByteArray;

	void           process_event (const uint8_t * &cur_ptr, const uint8_t *end_ptr);
	void           process_meta (const uint8_t * &cur_ptr, const uint8_t *end_ptr);
	void           process_meta_content (const uint8_t *cur_ptr, ptrdiff_t len, uint8_t type);
	void           process_std_midi (const ByteArray &msg);
	void           start_note (Channel &chn, uint8_t note, uint8_t velo);
	void           stop_note (Channel &chn, uint8_t note, uint8_t velo);
	void           stop_all_notes (Channel &chn);
	void           process_sysex (const uint8_t * &cur_ptr, const uint8_t *end_ptr, bool continue_flag);
	int32_t        read_varlen (const uint8_t * &cur_ptr, const uint8_t *end_ptr);
	uint8_t        read_byte (const uint8_t * &cur_ptr, const uint8_t *end_ptr);

	NoteRepeat     _rep_strategy   = NoteRepeat::_cut;
	MidiState      _ms;
	int32_t        _timestamp      = 0;
	bool           _track_end_flag = false;

	ChannelArray   _chn_arr;
	ByteArray      _msg;
	TempoMap       _tempo_map;
	TimeSigMap     _time_sig_map;



/*\\\ FORBIDDEN MEMBER FUNCTIONS \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

private:

	               MidiFileTrack ()                               = delete;
	               MidiFileTrack (const MidiFileTrack &other)     = delete;
	               MidiFileTrack (MidiFileTrack &&other)          = delete;
	MidiFileTrack& operator = (const MidiFileTrack &other)        = delete;
	MidiFileTrack& operator = (MidiFileTrack &&other)             = delete;
	bool           operator == (const MidiFileTrack &other) const = delete;
	bool           operator != (const MidiFileTrack &other) const = delete;

};	// class MidiFileTrack



//#include "MidiFileTrack.hpp"



#endif	// MidiFileTrack_HEADER_INCLUDED



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
