/*****************************************************************************

        MidiState.cpp
        Author: Laurent de Soras, 2022

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "midi.h"
#include "MidiState.h"

#include <cassert>



/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



void	MidiState::reset ()
{
	_status = 0;
	_msg_rt = 0;
	_length = 0;
	_data.clear ();
	_sysex.clear ();
	_sysex_ready_flag      = false;
	_illformed_stream_flag = false;
}



void	MidiState::push_byte (uint8_t val)
{
	// Clears the previous state
	_msg_rt = 0;
	if (int (_data.size ()) >= _length && _status != midi::_st_com_sysex)
	{
		// Previous message finished, gets ready for a new one
		_data.clear ();
		_status = update_running_status (_status);
	}
	if (_sysex_ready_flag)
	{
		_sysex.clear ();
	}
	_sysex_ready_flag      = false;
	_illformed_stream_flag = false;

	// New status byte
	if (is_status (val))
	{
		if (is_realtime (val))
		{
			_msg_rt = val;
		}
		
		else
		{
			auto            status_new = val;

			// Previous status was a SysEx, flags it as ready
			if (_status == midi::_st_com_sysex)
			{
				_sysex.swap (_data);
				_sysex_ready_flag = true;
				_data.clear ();
				_length = 0;
			}
			// A SysEx end in a non-SysEx context is just ignored
			else if (val == midi::_st_com_sysex_end)
			{
				_illformed_stream_flag = true;
				_data.clear ();
				status_new = 0;
			}

			// Checks if previous message data was interrupted
			if (! _data.empty () && int (_data.size ()) < _length)
			{
				_illformed_stream_flag = true;
			}
			_data.clear ();

			if (val == midi::_st_com_sysex)
			{
				_length = 0;
			}
			else
			{
				if (val == midi::_st_com_sysex_end)
				{
					status_new = 0;
				}
				else
				{
					_length = get_param_len (val);
				}
			}

			_status = status_new;
		}
	}

	// Data byte
	else
	{
		// No current running status, byte is ignored
		if (_status == 0)
		{
			_illformed_stream_flag = true;
		}
		else
		{
			_data.push_back (val);
		}
	}
}



bool	MidiState::is_message_ready () const noexcept
{
	return (
		   _msg_rt != 0
		|| (   _status != 0
		    && _status != midi::_st_com_sysex
		    && int (_data.size ()) >= _length)
	);
}



void	MidiState::get_message (std::vector <uint8_t> &msg) const
{
	if (is_message_ready ())
	{
		if (_msg_rt != 0)
		{
			msg.assign (1, _msg_rt);
		}
		else
		{
			msg.assign (1, _status);
			msg.insert (msg.end (), _data.begin (), _data.end ());
			if (extract_cmd_categ (_status) == midi::_st_chn_note_on)
			{
				assert (_data.size () == 2);
				if (msg [2] == 0)
				{
					msg [0] =
						(midi::_st_chn_note_off << 4) | extract_channel (msg [0]);
					msg [2] = midi::_def_velo;
				}
			}
		}
	}
	else
	{
		msg.clear ();
	}
}



bool	MidiState::is_sysex_ready () const noexcept
{
	return _sysex_ready_flag;
}



// Returns the SysEx without surrounding status bytes (0xF0, 0xF7 or any other
// status)
void	MidiState::get_sysex (std::vector <uint8_t> &msg) const
{
	if (is_sysex_ready ())
	{
		msg = _sysex;
	}
	else
	{
		msg.clear ();
	}
}



// Returns 0 is there is no running status
uint8_t	MidiState::get_running_status () const noexcept
{
	return update_running_status (_status);
}



bool	MidiState::is_stream_illformed () const noexcept
{
	return _illformed_stream_flag;
}



bool	MidiState::is_status (uint8_t x) noexcept
{
	return ((x & midi::_st_mask) != 0);
}



// categ in [0 ; 0xF]
bool	MidiState::is_channel (uint8_t categ) noexcept
{
	assert (categ >= 0x8);
	assert (categ <= 0xF);

	return (categ < midi::_st_sys);
}



// s must be a status byte
bool	MidiState::is_realtime (uint8_t s) noexcept
{
	assert (is_status (s));

	return ((s & midi::_st_rt_mask) == midi::_st_rt_mask);
}



// s must be a status byte
// Should not be called for a SysEx because the length cannot be known in
// advance
int	MidiState::get_param_len (uint8_t s) noexcept
{
	assert (is_status (s));
	assert (s != midi::_st_com_sysex);

	// Channel commands
	const auto     categ = extract_cmd_categ (s);
	if (is_channel (categ))
	{
		const bool     single_byte_flag =
			(categ == midi::_st_chn_prog || categ == midi::_st_chn_press);
		return single_byte_flag ? 1 : 2;
	}

	// System common & realtime
	if (s == midi::_st_com_song_pos)
	{
		return 2;
	}
	else if (s == midi::_st_com_mtc_qfm || s == midi::_st_com_song_sel)
	{
		return 1;
	}

	return 0;
}



// s must be a status byte
uint8_t	MidiState::extract_channel (uint8_t s) noexcept
{
	assert (is_status (s));

	return s & midi::_chn_mask;
}



// s must be a status byte
uint8_t	MidiState::extract_cmd_categ (uint8_t s) noexcept
{
	assert (is_status (s));

	return s >> 4;
}



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



uint8_t	MidiState::update_running_status (uint8_t status) noexcept
{
	if (status != 0 && extract_cmd_categ (status) == midi::_st_sys)
	{
		// A system message cancels the running status
		status = 0;
	}

	return status;
}



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
