/*****************************************************************************

        GRAOUMF TRACKER 2
        Author: Laurent de Soras, 1996-2016

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*****************************************************************************/



/*\\\ FICHIERS INCLUDE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "fstb/fnc.h"
#include	"base.h"
#include	"base.hpp"
#include	"base_ct.h"
#include	"log.h"
#include	"memory.h"
#include	"ReconstructionFilter.h"
#include	"splhandl.h"

#include	<cassert>
#include	<cmath>
#include	<cstdio>
#include	<cstdlib>
#include	<cstring>



/*\\\ CONSTANTES PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ CONSTANTES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ TYPES & STRUCTURES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ DEFINITION DES VARIABLES DE CLASSE PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ DEFINITION DES VARIABLES DE CLASSE PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ METHODES PUBLIQUES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*==========================================================================*/
/*      Nom: (constructeur)                                                 */
/*      Description: Initialise                                             */
/*      Parametres en entree:                                               */
/*        - width: largeur du sinus cardinal utilise par le filtre (en      */
/*                 samples)                                                 */
/*==========================================================================*/

ReconstructionFilter::ReconstructionFilter (int width)
{
	if (width <= 0)
	{
		LOG_printf ("ReconstructionFilter::ReconstructionFilter: Error: width parameter (%d) must be > 0.\n",
		            width);
		return;
	}
	if ((width & 1) != 0)
	{
		LOG_printf ("ReconstructionFilter::ReconstructionFilter: Error: width parameter (%d) must be even.\n",
		            width);
		return;
	}

	_width = width;

	_buf_len = 1;
	while (_buf_len < _width + _max_blk_len)
	{
		_buf_len <<= 1;
	}

	for (auto &buf : _buffer_arr)
	{
		buf.resize (_buf_len, 0);
	}

	_filt_ptr = nullptr;
	build_filter ();

	clear_buffer ();
}



/*==========================================================================*/
/*      Nom: (destructeur)                                                  */
/*      Description: Detruit                                                */
/*==========================================================================*/

ReconstructionFilter::~ReconstructionFilter ()
{
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	ReconstructionFilter::resample_and_mix (const SWORD *s_buffer_ptr, float *d_buffer_ptr,
		                                        LWORD length, int s_stereo, int d_stereo,
		                                        SLWORD volume, ULWORD pos_frac, double rate)
{
	constexpr float   vol_mult = 1.f / 0x100;

	const int      buf_mask = _buf_len - 1;
	const double   stretch  = 1.0 / rate;

	float          chn_amp = vol_mult * float (volume);
	float          stp_snc = 1;
	if (rate < 1)
	{
		stp_snc *= float (rate);
	}
	else
	{
		chn_amp *= float (stretch);
	}
	double         pos_frac_f = double (pos_frac) * (1 / 4294967296.0);

	const int      width_h = _width >> 1;
	const float    stp_win = 1.f / float (width_h);

	// We resample as many source samples as necessary to complete the
	// destination samples.
	// So for each destination sample at time_dest, we resample all the source
	// samples falling in [time_dest ; time_dest + 1[

	for (int idx_dst = 0; idx_dst < length; ++idx_dst)
	{
		// Time range for the destination sample, relative to the source
		const double   pos_src_beg = pos_frac_f +  idx_dst      * rate;
		const double   pos_src_end = pos_frac_f + (idx_dst + 1) * rate;

		const int      idx_src_beg = int (ceil (pos_src_beg));
		const int      idx_src_end = int (ceil (pos_src_end));

		for (int idx_src = idx_src_beg; idx_src < idx_src_end; ++idx_src)
		{
			// Source position, relative to the destination
			const double   pos_dst = (idx_src - pos_frac_f) / rate;

			// Fractional part
			const double   pos_dst_frac = pos_dst - idx_dst;
			assert (pos_dst_frac >= 0);
			assert (pos_dst_frac < 1);

			// Gathers the source samples
			std::array <float, _max_nbr_chn> val;
			for (int chn_cnt = 0; chn_cnt < s_stereo; chn_cnt ++)
			{
				val [chn_cnt] = s_buffer_ptr [idx_src * s_stereo + chn_cnt] * chn_amp;
			}
			if (s_stereo == 1)
			{
				// Mono source: spreads it on the other channels
				val [1] = val [0];
			}
			if (d_stereo == 1)
			{
				// Mono destination: both channels are mixed together
				val [0] += val [1];
			}

			// Impulse
			// The phase calculation could be optimised
			const float    ofs = float (pos_dst_frac) + (width_h - 1);
			for (int k = 0; k < _width; ++k)
			{
				const float    pos_snc = (k - ofs) * stp_snc; // [-w/2; w/2]
				const float    pos_win = (k - ofs) * stp_win; // [-1  ; 1  ]
#if 1
				const float    p2_snc  = (pos_snc + width_h) * _nbr_phases;
				const float    p2_win  = (pos_win + 1) * (width_h * _nbr_phases);
				const int      snc_idx = fstb::floor_int (p2_snc);
				const int      win_idx = fstb::floor_int (p2_win);
				const float    snc_d   = p2_snc - snc_idx;
				const float    win_d   = p2_win - win_idx;
				const Filter::InterpData &   dat_snc = _filt_ptr->_snc [snc_idx];
				const Filter::InterpData &   dat_win = _filt_ptr->_win [win_idx];
				const float    val_snc = dat_snc [0] + snc_d * dat_snc [1];
				const float    val_win = dat_win [0] + win_d * dat_win [1];
				const float    s_val   = val_snc * val_win;
#else
				// Reference implementation
				const float    s_val   = sinc (pos_snc) * window (pos_win);
#endif

				const int      pos_w   = (_buffer_pos + k) & buf_mask;
				float          y = 0;
				for (int chn_cnt = 0; chn_cnt < d_stereo; ++chn_cnt)
				{
					y = val [chn_cnt] * s_val;
					_buffer_arr [chn_cnt] [pos_w] += y;
				}
				for (int chn_cnt = d_stereo; chn_cnt < _max_nbr_chn; ++chn_cnt)
				{
					_buffer_arr [chn_cnt] [pos_w] += y;
				}
			}
		}
		
		// Copies the sample frame from the buffer and clears it
		for (int chn_cnt = 0; chn_cnt < d_stereo; ++chn_cnt)
		{
			d_buffer_ptr [idx_dst * d_stereo + chn_cnt] = _buffer_arr [chn_cnt] [_buffer_pos];
			_buffer_arr [chn_cnt] [_buffer_pos] = 0;
		}
		for (int chn_cnt = d_stereo; chn_cnt < _max_nbr_chn; ++chn_cnt)
		{
			_buffer_arr [chn_cnt] [_buffer_pos] = 0;
		}

		// Next buffer position
		_buffer_pos = (_buffer_pos + 1) & buf_mask;
	}
}



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/

void	ReconstructionFilter::clear_buffer (void)
{
	for (auto &buf : _buffer_arr)
	{
		std::fill (buf.begin (), buf.end (), 0.f);
	}

	_buffer_pos = 0;
}



void	ReconstructionFilter::copy_state (const ReconstructionFilter &other)
{
	assert (&other != 0);

	if (&other != this)
	{
		assert (other._width   == _width);
		assert (other._buf_len == _buf_len);

		_buffer_arr = other._buffer_arr;
		_buffer_pos = other._buffer_pos;
	}
}



/*\\\ METHODES PRIVEES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



void	ReconstructionFilter::build_filter ()
{
	auto           it = _filter_map.find (_width);
	if (it != _filter_map.end ())
	{
		_filt_ptr = &(it->second);
	}
	else
	{
		Filter &          filter     = _filter_map [_width];

		constexpr float   step       = 1.f / _nbr_phases;
		const float       plen_h     = _width * 0.5f;
		const float       plen_h_inv = 1.f / plen_h;
		const int         imp_len    = _width * _nbr_phases + 1;

		filter._snc.resize (imp_len);
		filter._win.resize (imp_len);

		for (int k = 0; k < imp_len; ++k)
		{
			const float    pos_spl = k * step; // [0 ; _width]

			// Current phase position
			const float    pos_snc = -plen_h + pos_spl; // [-_width/2 ; _width/2]
			const float    pos_win = pos_snc * plen_h_inv; // [-1 ; 1]

			// Next phase position
			const float    nxt_snc = pos_snc + step;
			const float    nxt_win = nxt_snc * plen_h_inv;

			// Values for the current and next phases
			const float    val_snc_cur = sinc (pos_snc);
			const float    val_win_cur = window (pos_win);
			const float    val_snc_nxt = sinc (nxt_snc);
			const float    val_win_nxt = window (nxt_win);

			filter._snc [k] [0] = val_snc_cur;
			filter._snc [k] [1] = val_snc_nxt - val_snc_cur;
			filter._win [k] [0] = val_win_cur;
			filter._win [k] [1] = val_win_nxt - val_win_cur;
		}

		_filt_ptr = &filter;
	}
}



// sin (pi * x) / (pi * x)
float	ReconstructionFilter::sinc (float x)
{
	float          y = 1;

	if (x != 0)
	{
		const float    xp = x * float (PI);
		y = sinf (xp) / xp;
	}

	return y;
}



// Window is located in [-1 ; 1]
float	ReconstructionFilter::window (float x)
{
	float          y = 0;

	if (x >= -1 && x <= 1)
	{
		const float    xp = x * float (PI);
#if 1
		// Blackman window
		y =
			  0.42f
			+ 0.5f  * cosf (xp)
			+ 0.08f * cosf (xp * 2);
#else
		// Blackman-Harris window: wider main lobe, smaller side lobes
		// Not as efficient to remove the first mirrored image when upsampling.
		y =
			  0.35875f
			+ 0.48829f * cosf (xp)
			+ 0.14128f * cosf (xp * 2)
			+ 0.01168f * cosf (xp * 3);
#endif
	}

	return y;
}



std::map <int, ReconstructionFilter::Filter>	ReconstructionFilter::_filter_map;



/****************************************************************************/
/*                                                                          */
/*                                                                          */
/*                                                                          */
/****************************************************************************/



/*==========================================================================*/
/*      Nom:                                                                */
/*      Description:                                                        */
/*      Parametres en entree:                                               */
/*      Parametres en sortie:                                               */
/*      Parametres en entree/sortie:                                        */
/*      Retour:                                                             */
/*==========================================================================*/
