/*****************************************************************************

        RefPtr.hpp
        GRAOUMF TRACKER 2
        Author: Laurent de Soras, 1996-2016

--- Legal stuff ---

This program is free software. It comes without any warranty, to
the extent permitted by applicable law. You can redistribute it
and/or modify it under the terms of the Do What The Fuck You Want
To Public License, Version 2, as published by Sam Hocevar. See
http://sam.zoy.org/wtfpl/COPYING for more details.

*Tab=3***********************************************************************/



#if ! defined (gtp_RefPtr_CODEHEADER_INCLUDED)
#define gtp_RefPtr_CODEHEADER_INCLUDED



/*\\\ INCLUDE FILES \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/

#include "conc/AioAdd.h"
#include "conc/AtomicIntOp.h"

#include <cassert>



namespace gtp
{



/*\\\ PUBLIC \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



template <class T>
RefPtr <T>::RefPtr ()
:	_cell_ptr (0)
,	_pool_ptr (0)
,	_count_ptr (0)
{
	// Nothing
}



template <class T>
RefPtr <T>::RefPtr (const RefPtr <T> &other)
:	_cell_ptr (other._cell_ptr)
,	_pool_ptr (other._pool_ptr)
,	_count_ptr (other._count_ptr)
{
	assert (other.is_valid ());
	add_ref ();
	assert (is_consistent (other));
}



template <class T>
RefPtr <T>::RefPtr (CellType *cell_ptr, PoolType *pool_ptr)
:	_cell_ptr (cell_ptr)
,	_pool_ptr (pool_ptr)
,	_count_ptr ((cell_ptr != 0) ? new conc::AtomicInt <long> (1) : 0)
{
	assert (   (_cell_ptr == 0 && _pool_ptr == 0)
	        || (_cell_ptr != 0 && _pool_ptr != 0));
}



template <class T>
RefPtr <T>::~RefPtr ()
{
	assert (is_valid ());

	destroy_complete ();
}



template <class T>
RefPtr <T> &	RefPtr <T>::operator = (const RefPtr <T> &other)
{
	assert (is_valid ());
	assert (is_consistent (other));

	if (other._cell_ptr != _cell_ptr)
	{
		destroy_complete ();
		_cell_ptr  = other._cell_ptr;
		_pool_ptr  = other._pool_ptr;
		_count_ptr = other._count_ptr;
		add_ref ();
	}
	assert (is_consistent (other));

	return (*this);
}



template <class T>
void	RefPtr <T>::swap (RefPtr <T> &other)
{
	assert (&other != 0);

	CellType *                 tmp_cell_ptr  = _cell_ptr;
	PoolType *                 tmp_pool_ptr  = _pool_ptr;
	conc::AtomicInt <long> *   tmp_count_ptr = _count_ptr;

	_cell_ptr  = other._cell_ptr;
	_pool_ptr  = other._pooll_ptr;
	_count_ptr = other._count_ptr;

	other._cell_ptr  = tmp_cell_ptr;
	other._pool_ptr  = tmp_pool_ptr;
	other._count_ptr = tmp_count_ptr;
}


template <class T>
RefPtr <T>::operator RefPtr <const T> () const 
{
	return (reinterpret_cast <const RefPtr<const T> &> (*this));
}



template <class T>
conc::CellPool <T> *	RefPtr <T>::get_pool () const
{
	return (_pool_ptr);
}



template <class T>
T *	RefPtr <T>::get () const
{
	return (_cell_ptr != 0) ? &_cell_ptr->_val : 0;
}



template <class T>
T *	RefPtr <T>::operator -> () const
{
	assert (_cell_ptr != 0);
	
	return (&_cell_ptr->_val);
}



template <class T>
T &	RefPtr <T>::operator * () const
{
	assert (_cell_ptr != 0);
	
	return (_cell_ptr->_val);
}



template <class T>
void	RefPtr <T>::destroy ()
{
	assert (is_valid ());

	destroy_complete ();
	_cell_ptr  = 0;
	_pool_ptr  = 0;
	_count_ptr = 0;
}



template <class T>
long	RefPtr <T>::get_count () const
{
	assert (is_valid ());

	long				count = 0;
	if (_count_ptr != 0)
	{
		count = *_count_ptr;
		assert (count > 0);
	}

	return (count);
}



template <class T>
bool	RefPtr <T>::is_valid () const
{
	return (   (   (_cell_ptr == 0 && _pool_ptr == 0 && _count_ptr == 0)
	            || (_cell_ptr != 0 && _pool_ptr != 0 && _count_ptr != 0))
	        && (_count_ptr == 0 || (*_count_ptr) > 0));
}



/*\\\ INTERNAL \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



template <class T>
RefPtr <T>::RefPtr (CellType *other_ptr, conc::AtomicInt <long> *count_ptr, PoolType *pool_ptr)
:	_cell_ptr (other_ptr)
,	_pool_ptr (pool_ptr)
,	_count_ptr (count_ptr)
{
	assert (is_valid ());
	add_ref ();
}



template <class T>
conc::AtomicInt <long> *	RefPtr <T>::get_counter_ref () const
{
	return (_count_ptr);
}



/*\\\ PROTECTED \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



/*\\\ PRIVATE \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/



template <class T>
void	RefPtr <T>::add_ref ()
{
	if (_count_ptr != 0)
	{
		const conc::AioAdd <long>	ftor_inc (1);
		conc::AtomicIntOp::exec (*_count_ptr, ftor_inc);
	}
}



/*
==============================================================================
Name : is_consistent
==============================================================================
*/

template <class T>
bool	RefPtr <T>::is_consistent (const RefPtr <T> &other) const
{
	assert (&other != 0);

	return (   other.is_valid ()
	        && (   (_cell_ptr == other._cell_ptr && _pool_ptr == other._pool_ptr && _count_ptr == other._count_ptr)
	            || (_cell_ptr != other._cell_ptr && _pool_ptr != other._pool_ptr && _count_ptr != other._count_ptr)));
}



/*
==============================================================================
Name : destroy_complete
==============================================================================
*/

template <class T>
void	RefPtr <T>::destroy_complete ()
{
	assert (is_valid ());

	if (_cell_ptr != 0)
	{
		const conc::AioAdd <long>	ftor_inc (-1);
		long count = conc::AtomicIntOp::exec_new (*_count_ptr, ftor_inc);
		
		if (count == 0)
		{
			_pool_ptr->return_cell (*_cell_ptr);
			_cell_ptr = 0;
			_pool_ptr = 0;

			delete _count_ptr;
			_count_ptr = 0;
		}
	}
}



}  // namespace gtp



#endif   // gtp_RefPtr_CODEHEADER_INCLUDED



/*\\\ EOF \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\*/
