unit AE;

interface

uses Classes, MMSystem;

type
  //Buffer flags
  TBufferFlag = (bfRead, bfPlaying);
  TBufferFlags = Set of TBufferFlag;

  //Type for samples data
  TS16Stereo = packed record L, R: SmallInt end;
  TS16StereoArray = packed Array[1..MaxInt div SizeOf(TS16Stereo)] of TS16Stereo;
  PS16StereoArray = ^TS16StereoArray;

  //Record for buffer
  TBufferRec = packed record
    Hdr: TWaveHdr;
    Flags: TBufferFlags;
    Data: PS16StereoArray;
  end;

  TBufferArray = Array[1..MaxInt div SizeOf(TBufferRec)] of TBufferRec;
  PBufferArray = ^TBufferArray;

  TAudioEngine = class;
  //Read data function, it either reads sample or not
  TReadDataFunction = function(Render: Boolean; DataPtr: PS16StereoArray; Count: Integer): Boolean of object;
  TStartEngineFunction = procedure of object;
  TStopEngineFunction = procedure of object;
  TExceptionFunction = function(Message: String): Boolean of object;

  TAudioEngine = class
   private
    //Wave out device ID and handle
    FWaveDeviceID: Integer;
    FWaveHandle: HWAVEOUT;

    //Worker thread
    FThread: TThread;

    //Buffers count, size and data
    FBufCount: Integer;
    FBufSize: Integer;
    FBuffers: PBufferArray;

    //Execution control

    //Buffer that is currently being processed
    FCurBuf: Integer;

    //Number of successfully read buffers
    FBufRead: Integer;

    //Is thread needs to be terminated (no more data available)
    FNeedTerminate: Boolean;

    //Thread is going to suspend
    FSuspended: Boolean;

    //Events
    FReadData: TReadDataFunction;
    FStartEngine: TStartEngineFunction;
    FStopEngine: TStopEngineFunction;
    FException: TExceptionFunction;

    //Processing procedure (one step)
    procedure ProcessStep;

    //Create buffers
    procedure CreateBuffers;

    //Destroy buffers
    procedure DestroyBuffers;

    //Handle thread termination
    procedure DoStop;
   public
    constructor Create(AWaveDeviceID: Integer;
      ABufCount, ABufSize: Integer);
    destructor Destroy; override;

    //Start worker thread
    procedure Start;

    //Stop worker thread
    procedure Stop;

    //Suspend worker thread
    procedure Pause;

    //Resume worker thread
    procedure Resume;

    //Render to wave
    procedure Render(FileName: String);

    //Engine is stopped
    function Stopped: Boolean;

    //Engine is paused
    function Paused: Boolean;

    //To set handlers
    property OnReadData: TReadDataFunction read FReadData write FReadData;
    property OnStartEngine: TStartEngineFunction read FStartEngine write FStartEngine;
    property OnStopEngine: TStopEngineFunction read FStopEngine write FStopEngine;
    property OnException: TExceptionFunction read FException write FException;
  end;

implementation

uses SysUtils, Wave, Windows;

//Helper routine - raises Wave out error as exception
procedure Check(Error: MMResult);
var TxtBuf: Array[0..255] of Char;
begin
  if Error=0 then Exit;
  if waveOutGetErrorText(Error, @TxtBuf, 255)=0 then
    Raise Exception.Create(TxtBuf)
  else Raise Exception.Create('Wave out error, code '+IntToStr(Error));
end;

//Worker thread
type
  TWorkerThread = class(TThread)
   private
    FAudioEngine: TAudioEngine;
   protected
    procedure Execute; override;
   public
    constructor Create(AAudioEngine: TAudioEngine);
    destructor Destroy; override;
  end;

procedure TWorkerThread.Execute;
begin
  repeat
    try
      FAudioEngine.ProcessStep;
    except
      if not Assigned(FAudioEngine.FException) or
         not FAudioEngine.FException((ExceptObject as Exception).Message) then
        Raise;
    end;
    Sleep(1);
    if FAudioEngine.FSuspended then Suspend;
    FAudioEngine.FSuspended := false;
  until FAudioEngine.FNeedTerminate or Terminated;
  FAudioEngine.DoStop;
end;

constructor TWorkerThread.Create(AAudioEngine: TAudioEngine);
begin
  inherited Create(true);
  Priority := tpNormal;
  FreeOnTerminate := true;
  FAudioEngine := AAudioEngine;
  Resume;
end;

destructor TWorkerThread.Destroy;
begin
  inherited Destroy;
end;

//Audio engine
procedure TAudioEngine.ProcessStep;
procedure WaitBuffer;
begin
  if bfPlaying in FBuffers^[FCurBuf].Flags then
    begin
      while FBuffers^[FCurBuf].Hdr.dwFlags and WHDR_DONE=0 do Sleep(1);
      FBuffers^[FCurBuf].Flags := [];
      Dec(FBufRead);
    end;
end;
procedure ReadBuffer;
begin
  if not Assigned(FReadData) then
    begin
      FNeedTerminate := FBufRead=0;
      Exit;
    end;
  if not FReadData(false, FBuffers^[FCurBuf].Data, FBufSize) then
    FNeedTerminate := FBufRead=0
  else
    begin
      FBuffers^[FCurBuf].Flags := [bfRead];
      Inc(FBufRead);
      FNeedTerminate := false;
    end;
end;
procedure PlayBuffer;
begin
  if bfRead in FBuffers^[FCurBuf].Flags then
    begin
      WaveOutWrite(FWaveHandle, @FBuffers^[FCurBuf].Hdr, SizeOf(TWaveHdr));
      FBuffers^[FCurBuf].Flags := [bfPlaying];
    end;
end;
procedure NextBuffer;
begin
  if FCurBuf=FBufCount then FCurBuf := 1 else Inc(FCurBuf);
end;
begin
  WaitBuffer;
  ReadBuffer;
  PlayBuffer;
  NextBuffer;
end;

procedure TAudioEngine.CreateBuffers;
var i: Integer;
begin
  if FBuffers<>nil then Exit;
  GetMem(FBuffers, FBufCount*SizeOf(TBufferRec));
  for i := 1 to FBufCount do
    begin
      GetMem(FBuffers^[i].Data, FBufSize*4);
      FBuffers^[i].Hdr.lpData := Pointer(FBuffers^[i].Data);
      FBuffers^[i].Hdr.dwBufferLength := FBufSize*4;
      FBuffers^[i].Hdr.dwFlags := 0;
      FBuffers^[i].Hdr.dwLoops := 0;
      FBuffers^[i].Flags := [];
      Check(WaveOutPrepareHeader(FWaveHandle, @FBuffers^[i].Hdr, SizeOf(TWaveHdr)));
    end;
end;

procedure TAudioEngine.DestroyBuffers;
var i: Integer;
begin
  for i := 1 to FBufCount do
    begin
      Check(WaveOutUnPrepareHeader(FWaveHandle, @FBuffers^[i].Hdr, SizeOf(TWaveHdr)));
      FreeMem(FBuffers^[i].Data);
    end;
  FreeMem(FBuffers, FBufCount*SizeOf(TBufferRec));
  FBuffers := nil;
end;

procedure TAudioEngine.DoStop;
begin
  Check(WaveOutReset(FWaveHandle));
  DestroyBuffers;
  FThread := nil;
  if Assigned(FStopEngine) then FStopEngine;
end;

constructor TAudioEngine.Create(AWaveDeviceID: Integer;
  ABufCount, ABufSize: Integer);
var Fmt: TWaveFormatEx;
begin
  FThread := nil;
  FBuffers := nil;
  FBufCount := ABufCount;
  FBufSize := ABufSize;
  FSuspended := false;
  if (ABufCount<2) or (ABufCount>100) then
    Raise Exception.Create('Invalid number of buffers');
  if (ABufSize<1) or (ABufSize>44100) then
    Raise Exception.Create('Invalid buffer size');
  FWaveDeviceID := AWaveDeviceID;
  with Fmt do
    begin
      wFormatTag := WAVE_FORMAT_PCM;
      nChannels := 2;
      nSamplesPerSec := 44100;
      nAvgBytesPerSec := 44100*4;
      nBlockAlign := 4;
      wBitsPerSample := 16;
      cbSize := 0;
    end;
  Check(waveOutOpen(@FWaveHandle,
                    FWaveDeviceID,
                    @Fmt,
                    0,
                    0,
                    CALLBACK_NULL or WAVE_ALLOWSYNC));
end;

destructor TAudioEngine.Destroy;
begin
  Stop;
  if FWaveHandle<>0 then
    Check(waveOutClose(FWaveHandle));
end;

procedure TAudioEngine.Start;
begin
  if Paused and not Stopped then
    begin
      Resume;
      Exit;
    end;
  if not Assigned(FReadData) then
    Raise Exception.Create('Read data handler must be specified');
  if FWaveHandle=0 then
    Raise Exception.Create('Wave out not initialized');
  if Assigned(FThread) and not FNeedTerminate then Exit;
  if Assigned(FThread) then
    begin
      FSuspended := false;
      FNeedTerminate := true;
      FThread.Terminate;
    end;
  while Assigned(FThread) do Sleep(1);
  CreateBuffers;
  FCurBuf := 1;
  FBufRead := 0;
  FNeedTerminate := false;
  if Assigned(FStartEngine) then FStartEngine;
  FThread := TWorkerThread.Create(Self);
end;

procedure TAudioEngine.Stop;
begin
  if Assigned(FThread) and not FNeedTerminate then
    begin
      FSuspended := false;
      FNeedTerminate := true;
      FThread.Terminate;
    end;
  while Assigned(FThread) do Sleep(1);
end;

procedure TAudioEngine.Pause;
begin
  if Paused or Stopped then Exit;
  FSuspended := true;
  repeat until Paused;
end;

procedure TAudioEngine.Resume;
begin
  if not Paused or Stopped then Exit;
  FSuspended := false;
  if FThread.Suspended then FThread.Resume;
end;

procedure TAudioEngine.Render(FileName: String);
var W: TWave; A: TS16Stereo;
begin
  W := TWave.Create;
  try
    W.Length := 0;
    W.Bits := 16;
    W.Channels := 2;
    W.Rate := 44100;
    repeat
      if not Assigned(FReadData) then Break;
      if not FReadData(true, @A, 1) then Break;
      W.Length := W.Length+1;
      W.SampleL[W.Length] := A.L;
      W.SampleR[W.Length] := A.R;
    until false;
    if W.Length>0 then W.SaveToFile(FileName);
  finally
    W.Free;
  end;
end;

function TAudioEngine.Stopped: Boolean;
begin
  result := not Assigned(FThread);
end;

function TAudioEngine.Paused: Boolean;
begin
  result := not Stopped and FThread.Suspended;
end;

end.