unit u_011_012;

interface

uses Wave, Classes;

type
  TCommand = packed record
    Note: Byte;
    Instrument: Integer;
    Volume: Byte;
    Pan: ShortInt;
  end;

  TChannel = Array[1..MaxInt div SizeOf(TCommand)] of TCommand;
  PChannel = ^TChannel;

  TChannelArray = Array[1..MaxInt div SizeOf(PChannel)] of PChannel;
  PChannelArray = ^TChannelArray;

  TPattern = class
   private
    InDestroy: Boolean;
    FData: PChannelArray;
    FChannels: Integer;
    FLength: Integer;
    procedure SetChannels(AChannels: integer);
    procedure SetLength(ALength: Integer);
    procedure CheckPos(AChannel, APos: Integer);
    function GetCommand(AChannel, APos: Integer): TCommand;
    procedure SetCommand(AChannel, APos: Integer; ACommand: TCommand);
   public
    Name: String;
    constructor Create;
    destructor Destroy; override;
    property Channels: Integer read FChannels write SetChannels;
    property Length: Integer read FLength write SetLength;
    property Command[Channel, Pos: Integer]: TCommand read GetCommand write SetCommand; default;
  end;

  TLoopType = (lNone, lNormal, lBidi);

  TSample = class(TWave)
   private
    FDefVolume: Byte;
    FDefPan: ShortInt;
    FLoopStart: Integer;
    FLoopEnd: Integer;
    procedure SetDefVolume(ADefVolume: Byte);
    procedure SetDefPan(ADefPan: ShortInt);
    procedure SetLoopStart(ALoopStart: Integer);
    procedure SetLoopEnd(ALoopEnd: Integer);
   public
    Name: String;
    LoopType: TLoopType;
    Position: Integer; //used for currently playing only
    Track: Integer; //used for currently playing Note Off only
    Channel: Integer; //used for currently playing Note Off only
    Note: Byte; //used for currently playing Note Off only
    Freq: Integer; //used for currently playing Note Off only
    constructor Create;
    procedure CopyFrom(Source: TSample);
    procedure Resample(ARate: Integer);
    property DefVolume: Byte read FDefVolume write SetDefVolume;
    property DefPan: ShortInt read FDefPan write SetDefPan;
    property LoopStart: Integer read FLoopStart write SetLoopStart;
    property LoopEnd: Integer read FLoopEnd write SetLoopEnd;
  end;

  TSampleList = class(TList)
   private
    function GetSample(i: Integer): TSample;
   public
    destructor Destroy; override;
    procedure Delete(i: Integer);
    procedure Clear;
    property Sample[i: Integer]: TSample read GetSample; default;
  end;

  TPatternList = class(TList)
   private
    function GetPattern(i: Integer): TPattern;
   public
    destructor Destroy; override;
    procedure Delete(i: Integer);
    procedure Clear;
    property Pattern[i: Integer]: TPattern read GetPattern; default;
  end;

  TIntArray = Array[1..MaxInt div SizeOf(Integer)] of Integer;
  PIntArray = ^TIntArray;
  TTrack = record
   Count: Integer;
   Pos: Integer;
   PPos: Integer;
   Data: PIntArray;
  end;

  TTrackArray = Array[1..MaxInt div SizeOf(TTrack)] of TTrack;
  PTrackArray = ^TTrackArray;

  TSequence = class
   private
    InDestroy: Boolean;
    FData: PTrackArray;
    FTracks: Integer;
    procedure SetTracks(ATracks: Integer);
    procedure CheckTrack(Track: Integer);
    function GetTrackLength(Track: Integer): Integer;
    procedure SetTrackLength(Track, TrackLength: Integer);
    procedure CheckPos(Track, Pos: Integer);
    function GetPatternNum(Track, Pos: Integer): Integer;
    procedure SetPatternNum(Track, Pos, Pattern: Integer);
    //Positions in track and pattern for playing only
    function GetPos(Track: Integer): Integer;
    procedure SetPos(Track, Pos: Integer);
    function GetPPos(Track: Integer): Integer;
    procedure SetPPos(Track, PPos: Integer);
   public
    constructor Create;
    destructor Destroy; override;
    procedure InsertPattern(Track, Pos, PatNum: Integer);
    procedure DeletePattern(Track, Pos: Integer);
    procedure InsertTrack(Track: Integer);
    procedure DeleteTrack(Track: Integer);
    procedure CopyTrack(TFrom, TTo: Integer);
    property Tracks: Integer read FTracks write SetTracks;
    property TrackLength[Track: integer]: Integer
      read GetTrackLength write SetTrackLength;
    property PatternNum[Track, Pos: Integer]: Integer
      read GetPatternNum write SetPatternNum; default;
    property Pos[Track: Integer]: Integer read GetPos write SetPos;
    property PPos[Track: Integer]: Integer read GetPPos write SetPPos;
  end;

  TNNA = (nnaNone, nnaChannelCut, nnaChannelOff);
  TSong = class
   private
    FTempo: Word;
    FTPB: Byte;
    FTickTime: Integer;
    FSamplesPerTick: Integer;
    FSongVolume: Byte;
    procedure SetSongVolume(ASongVolume: Byte);
   public
    Title: String;
    Author: String;
    Info: String;
    NNA: TNNA;
    Samples: TSampleList;
    Patterns: TPatternList;
    Sequence: TSequence;
    procedure SetTempo(ATempo: Word);
    procedure SetTPB(ATPB: Byte);
    constructor Create;
    destructor Destroy; override;
    procedure Clear;
    function LoadFromFile(FileName: String): Boolean;
    procedure SaveToFile(FileName: String; Compress: Boolean);
    property Tempo: Word read FTempo write SetTempo;
    property TPB: Byte read FTPB write SetTPB;
    property TickTime: Integer read FTickTime;
    property SamplesPerTick: Integer read FSamplesPerTick;
    property SongVolume: Byte read FSongVolume write SetSongVolume;
  end;

implementation

uses SysUtils, MSConsts, ZLib;

{ TPattern }

procedure TPattern.SetChannels(AChannels: integer);
var i, j: Integer;
begin
  if (AChannels=0) and not InDestroy then
    Raise Exception.Create('Zero number of channels not allowed');
  if AChannels=FChannels then Exit;
  for i := AChannels+1 to FChannels do
    ReAllocMem(FData^[i], 0);
  ReAllocMem(FData, AChannels*SizeOf(PChannel));
  for i := FChannels+1 to AChannels do
    begin
      FData^[i] := nil;
      ReAllocMem(FData^[i], FLength*SizeOf(TCommand));
      if FLength>0 then
        //FillChar(FData^[i]^, FLength*SizeOf(TCommand), 0);
        for j := 1 to FLength do
          begin
            FData^[i]^[j].Note := 255;
            FData^[i]^[j].Instrument := 0;
            FData^[i]^[j].Volume := 255;
            FData^[i]^[j].Pan := 127;
          end;
   end;
  FChannels := AChannels;
end;

procedure TPattern.SetLength(ALength: Integer);
var i, j: Integer;
begin
  if (ALength=0) and not InDestroy then
    Raise Exception.Create('Zero pattern length not allowed');
  if ALength=FLength then Exit;
  for i := 1 to FChannels do
    begin
      ReAllocMem(FData^[i], ALength*SizeOf(TCommand));
      if ALength>FLength then
      //FillChar(FData^[i]^[FLength+1],
      //         (ALength-FLength)*SizeOf(TCommand), 0);
        for j := FLength+1 to ALength do
          begin
            FData^[i]^[j].Note := 255;
            FData^[i]^[j].Instrument := 0;
            FData^[i]^[j].Volume := 255;
            FData^[i]^[j].Pan := 127;
          end;
    end;
  FLength := ALength;
end;

procedure TPattern.CheckPos(AChannel, APos: Integer);
begin
  if (AChannel<1) or (AChannel>FChannels) then
    Raise Exception.Create('Channel out of range');
  if (APos<1) or (APos>FLength) then
    Raise Exception.Create('Position out of range');
end;

function TPattern.GetCommand(AChannel, APos: Integer): TCommand;
begin
  CheckPos(AChannel, APos);
  result := FData^[AChannel]^[APos];
end;

procedure TPattern.SetCommand(AChannel, APos: Integer; ACommand: TCommand);
begin
  CheckPos(AChannel, APos);
  with ACommand do
    begin
      if not (Note in [0..119, NoteCut, NoteOff, NoNote]) then
        Raise Exception.Create('Invalid note');
      if not (Volume in [0..200, DVol]) then
        Raise Exception.Create('Invalid volume');
      if (Pan<>DPan) and ((Pan<-100) or (Pan>100)) then
        Raise Exception.Create('Invalid panning');
      if Instrument=0 then Note := NoNote;
      if Note in [NoteCut, NoteOff, NoNote] then
        begin
          Instrument := 0;
{          Volume := DVol;
          Pan := DPan;}
        end;
    end;
  FData^[AChannel]^[APos] := ACommand;
end;

constructor TPattern.Create;
begin
  inherited Create;
  FChannels := 0;
  FLength := 0;
  FData := nil;
  SetChannels(1);
  SetLength(1);
  InDestroy := false;
end;

destructor TPattern.Destroy;
begin
  InDestroy := true;
  SetChannels(0);
  SetLength(0);
  InDestroy := false;
  inherited Destroy;
end;

{ TSample }

procedure TSample.SetDefVolume(ADefVolume: Byte);
begin
  if FDefVolume=ADefVolume then Exit;
  if ADefVolume>200 then
    Raise Exception.Create('Volume must be from 0 to 200');
  FDefVolume := ADefVolume;
end;

procedure TSample.SetDefPan(ADefPan: ShortInt);
begin
  if FDefPan=ADefPan then Exit;
  if (ADefPan<-100) or (ADefPan>100) then
    Raise Exception.Create('Panning must be from -100 to 100');
  FDefPan := ADefPan;
end;

//invalid loops are simply ignored, no errors !
procedure TSample.SetLoopStart(ALoopStart: Integer);
begin
  if ALoopStart=FLoopStart then Exit;
  FLoopStart := ALoopStart;
end;

procedure TSample.SetLoopEnd(ALoopEnd: Integer);
begin
  if ALoopEnd=FLoopEnd then Exit;
  FLoopEnd := ALoopEnd;
end;

constructor TSample.Create;
begin
  inherited Create;
  FDefVolume := 100;
  FDefPan := 0;
  LoopType := lNone;
  FLoopStart := 0;
  FLoopEnd := 0;
  Note := 0;
  Freq := Rate;
end;

procedure TSample.CopyFrom(Source: TSample);
begin
  inherited CopyFrom(Source);
  FDefVolume := Source.FDefVolume;
  FDefPan := Source.FDefPan;
  LoopType := Source.LoopType;
  FLoopStart := Source.FLoopStart;
  FLoopEnd := Source.FLoopEnd;
  Track := Source.Track;
  Channel := Source.Channel;
end;

procedure TSample.Resample(ARate: Integer);
var SRate: Integer;
begin
  SRate := Rate;
  inherited Resample(ARate);
  SetLoopStart(FLoopStart*ARate div SRate);
  SetLoopEnd(FLoopEnd*ARate div SRate);
end;

{ TSampleList }

destructor TSampleList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

function TSampleList.GetSample(i: Integer): TSample;
begin
  result := TSample(Items[i-1]);
end;

procedure TSampleList.Delete(i: Integer);
begin
  TSample(Items[i-1]).Free;
  inherited Delete(i-1);
end;

procedure TSampleList.Clear;
begin
  while Count>0 do Delete(1);
end;

{ TPatternList }

destructor TPatternList.Destroy;
begin
  Clear;
  inherited Destroy;
end;

function TPatternList.GetPattern(i: Integer): TPattern;
begin
  result := TPattern(Items[i-1]);
end;

procedure TPatternList.Delete(i: Integer);
begin
  TPattern(Items[i-1]).Free;
  inherited Delete(i-1);
end;

procedure TPatternList.Clear;
begin
  while Count>0 do Delete(1);
end;

{ TSequence }

procedure TSequence.SetTracks(ATracks: Integer);
var i: Integer;
begin
  if (ATracks=0) and not InDestroy then
    Raise Exception.Create('Zero tracks number not allowed');
  if ATracks=FTracks then Exit;
  for i := ATracks+1 to FTracks do
    SetTrackLength(i, 0);
  ReAllocMem(FData, ATracks*SizeOf(TTrack));
  for i := FTracks+1 to ATracks do
    FillChar(FData^[i], SizeOf(TTrack), 0);
  FTracks := ATracks;
end;

procedure TSequence.CheckTrack(Track: Integer);
begin
  if (Track<1) or (Track>FTracks) then
    Raise Exception.Create('Track number out of range');
end;

function TSequence.GetTrackLength(Track: Integer): Integer;
begin
  CheckTrack(Track);
  result := FData^[Track].Count;
end;

procedure TSequence.SetTrackLength(Track, TrackLength: Integer);
begin
  CheckTrack(Track);
  ReAllocMem(FData^[Track].Data, SizeOf(Integer)*TrackLength);
  if TrackLength>FData^[Track].Count then
    FillChar(FData^[Track].Data[FData^[Track].Count+1],
             (TrackLength-FData^[Track].Count)*SizeOf(Integer), 0);
  FData^[Track].Count := TrackLength;
end;

procedure TSequence.CheckPos(Track, Pos: Integer);
begin
  CheckTrack(Track);
  if (Pos<1) or (Pos>GetTrackLength(Track)) then
    Raise Exception.Create('Track position out of range');
end;

function TSequence.GetPatternNum(Track, Pos: Integer): Integer;
begin
  CheckPos(Track, Pos);
  result := FData^[Track].Data^[Pos];
end;

procedure TSequence.SetPatternNum(Track, Pos, Pattern: Integer);
begin
  CheckPos(Track, Pos);
  FData^[Track].Data^[Pos] := Pattern;
end;

function TSequence.GetPos(Track: Integer): Integer;
begin
  CheckTrack(Track);
  result := FData^[Track].Pos;
end;

procedure TSequence.SetPos(Track, Pos: Integer);
begin
  CheckTrack(Track);
  FData^[Track].Pos := Pos;
end;

function TSequence.GetPPos(Track: Integer): Integer;
begin
  CheckTrack(Track);
  result := FData^[Track].PPos;
end;

procedure TSequence.SetPPos(Track, PPos: Integer);
begin
  CheckTrack(Track);
  FData^[Track].PPos := PPos;
end;

constructor TSequence.Create;
begin
  inherited Create;
  FTracks := 0;
  FData := nil;
  SetTracks(1);
  InDestroy := false;
end;

destructor TSequence.Destroy;
var i: Integer;
begin
  for i := 1 to FTracks do
    SetTrackLength(i, 0);
  InDestroy := true;
  SetTracks(0);
  InDestroy := false;
  inherited Destroy;
end;

procedure TSequence.InsertPattern(Track, Pos, PatNum: Integer);
var i: Integer;
begin
  CheckTrack(Track);
  if Pos<>TrackLength[Track]+1 then CheckPos(Track, Pos);
  TrackLength[Track] := TrackLength[Track]+1;
  for i := TrackLength[Track] downto Pos+1 do
    PatternNum[Track, i] := PatternNum[Track, i-1];
  PatternNum[Track, Pos] := PatNum;
end;

procedure TSequence.DeletePattern(Track, Pos: Integer);
var i: Integer;
begin
  CheckPos(Track, Pos);
  for i := Pos to TrackLength[Track]-1 do
    PatternNum[Track, i] := PatternNum[Track, i+1];
  TrackLength[Track] := TrackLength[Track]-1;
end;

procedure TSequence.InsertTrack(Track: Integer);
var i, j: Integer;
begin
  if Track<>Tracks+1 then CheckTrack(Track);
  Tracks := Tracks+1;
  for i := Tracks downto Track+1 do
    begin
      TrackLength[i] := TrackLength[i-1];
      for j := 1 to TrackLength[i] do
        PatternNum[i, j] := PatternNum[i-1, j];
    end;
end;

procedure TSequence.DeleteTrack(Track: Integer);
var i, j: Integer;
begin
  CheckTrack(Track);
  for i := Track to Tracks-1 do
    begin
      TrackLength[i] := TrackLength[i+1];
      for j := 1 to TrackLength[i] do
        PatternNum[i, j] := PatternNum[i+1, j];
    end;
  Tracks := Tracks-1;
end;

procedure TSequence.CopyTrack(TFrom, TTo: Integer);
var i: Integer;
begin
  CheckTrack(TFrom);
  CheckTrack(TTo);
  TrackLength[TTo] := TrackLength[TFrom];
  for i := 1 to TrackLength[TFrom] do
    PatternNum[TTo, i] := PatternNum[TFrom, i];
end;

{ TSong }

procedure TSong.SetSongVolume(ASongVolume: Byte);
begin
  if FSongVolume=ASongVolume then Exit;
  if (ASongVolume>100) then
    Raise Exception.Create('Invalid song volume');
  FSongVolume := ASongVolume;
end;

function CalcTickTime(Tempo, TPB: Integer): Integer;
begin
  result := 60000 div Tempo div TPB;
end;

function CalcSPT(Tempo, TPB: Integer): Integer;
begin
  result := 60*BaseRate div Tempo div TPB;
end;

procedure TSong.SetTempo(ATempo: Word);
begin
  if ATempo=FTempo then Exit;
  if (ATempo<10) or (ATempo>300) then
    Raise Exception.Create('Invalid tempo');
  FTempo := ATempo;
  FTickTime := CalcTickTime(FTempo, FTPB);
  FSamplesPerTick := CalcSPT(FTempo, FTPB);
end;

procedure TSong.SetTPB(ATPB: Byte);
begin
  if ATPB=FTPB then Exit;
  if (ATPB<4) or (ATPB>32) then
    Raise Exception.Create('Invalid tempo');
  FTPB := ATPB;
  FTickTime := CalcTickTime(FTempo, FTPB);
  FSamplesPerTick := CalcSPT(FTempo, FTPB);
end;

constructor TSong.Create;
begin
  inherited Create;
  FTempo := 1;
  FTPB := 1;
  FSongVolume := 100;
  Tempo := 140;
  TPB := 4;
  NNA := nnaChannelCut;
  Title := '';
  Author := '';
  Info := '';
  Samples := TSampleList.Create;
  Patterns := TPatternList.Create;
  Sequence := TSequence.Create;
end;

destructor TSong.Destroy;
begin
  Samples.Free;
  Patterns.Free;
  Sequence.Free;
  inherited Destroy;
end;

procedure TSong.Clear;
begin
  SongVolume := 100;
  Tempo := 140;
  TPB := 4;
  NNA := nnaChannelCut;
  Title := '';
  Author := '';
  Info := '';
  Samples.Clear;
  Patterns.Clear;
  Sequence.SetTracks(1);
  Sequence.SetTrackLength(1, 0);
end;

function TSong.LoadFromFile(FileName: String): Boolean;
var F: TStream;
    FF: TFileStream;
    DF: TDecompressionStream;
    Sign: String[4];

function ReadString: String;
var L: Integer;
begin
  F.Read(L, 4);
  SetLength(result, L);
  F.Read(result[1], L);
end;

procedure ReadInfo;
var W: Word; B: Byte; N: TNNA;
begin
  F.Read(W, 2);
  Tempo := W;
  F.Read(B, 1);
  TPB := B;
  F.Read(N, SizeOf(TNNA));
  NNA := N;
  Title := ReadString;
  Author := ReadString;
  Info := ReadString;
end;

procedure ReadSamples;
var i, j, L: Integer; B: Byte; SI: ShortInt; S: TSample;
begin
  F.Read(L, 4);
  for j := 1 to L do
    begin
      S := TSample.Create;
      try
        S.Name := ReadString;
        F.Read(B, 1);
        S.Bits := B;
        F.Read(B, 1);
        S.Channels := B;
        F.Read(i, 4);
        S.Rate := i;
        F.Read(i, 4);
        S.Length := i;
        F.Read(S.RawData^, S.DataLength);
        F.Read(B, 1);
        S.LoopType := TLoopType(B);
        F.Read(i, 4);
        S.LoopStart := i;
        F.Read(i, 4);
        S.LoopEnd := i;
        F.Read(B, 1);
        S.DefVolume := B;
        F.Read(SI, 1);
        S.DefPan := SI;
        Samples.Add(S);
      except
        S.Free;
      end;
    end;
end;

procedure ReadPatterns;
var i, j, L: Integer; P: TPattern;
begin
  F.Read(L, 4);
  for j := 1 to L do
    begin
      P := TPattern.Create;
      try
        P.Name := ReadString;
        F.Read(i, 4);
        P.Channels := i;
        F.Read(i, 4);
        P.Length := i;
        for i := 1 to P.Channels do
          F.Read(P.FData^[i]^, SizeOf(TCommand)*P.Length);
        Patterns.Add(P);
      except
        P.Free;
      end;
    end;
end;

procedure ReadSequence;
var L, i, j: Integer;
begin
  F.Read(L, 4);
  Sequence.SetTracks(L);
  for j := 1 to L do
    begin
      F.Read(i, 4);
      Sequence.SetTrackLength(j, i);
      F.Read(Sequence.FData[j].Data^, SizeOf(Integer)*i);
    end;
end;

begin
  result := false;
  DF := nil;
  FF := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  try
    SetLength(Sign, 4);
    FF.Read(Sign[1], 4);
    if Sign<>'SONG' then
      Raise Exception.Create('Invalid file format');
    FF.Read(Sign[1], 4);
    if Sign='0.12' then
      Raise Exception.Create('File version is 0.12, no need to convert');
    if Sign<>'0.11' then
      Raise Exception.Create('Invalid file version, please convert');
    FF.Read(Sign[1], 4);
    if SIGN='NORM' then
      begin
        F := FF;
        result := false;
      end
    else
    if SIGN='COMP' then
      begin
        DF := TDecompressionStream.Create(FF);
        F := DF;
        result := true;
      end
    else
      Raise Exception.Create('Invalid file type');
    Clear;
    //General info
    ReadInfo;
    //Samples
    ReadSamples;
    //Patterns
    ReadPatterns;
    //Sequence
    ReadSequence;
  finally
    if Assigned(DF) then DF.Free;
    FF.Free;
  end;
end;

procedure TSong.SaveToFile(FileName: String; Compress: Boolean);
var F: TStream;
    FF: TFileStream;
    CF: TCompressionStream;
    Sign: String[4];

procedure WriteString(S: String);
var L: Integer;
begin
  L := Length(S);
  F.Write(L, 4);
  F.Write(S[1], L);
end;

procedure WriteInfo;
begin
  F.Write(Tempo, 2);
  F.Write(TPB, 1);
  F.Write(NNA, SizeOf(TNNA));
  F.Write(SongVolume, 1);
  WriteString(Title);
  WriteString(Author);
  WriteString(Info);
end;

procedure WriteSamples;
var i, j, L: Integer; B: Byte; SI: ShortInt; S: TSample;
begin
  L := Samples.Count;
  F.Write(L, 4);
  for j := 1 to L do
    begin
      S := Samples[j];
      WriteString(S.Name);
      B := S.Bits;
      F.Write(B, 1);
      B := S.Channels;
      F.Write(B, 1);
      i := S.Rate;
      F.Write(i, 4);
      i := S.Length;
      F.Write(i, 4);
      F.Write(S.RawData^, S.DataLength);
      B := Byte(S.LoopType);
      F.Write(B, 1);
      i := S.LoopStart;
      F.Write(i, 4);
      i := S.LoopEnd;
      F.Write(i, 4);
      B := S.DefVolume;
      F.Write(B, 1);
      SI := S.DefPan;
      F.Write(SI, 1);
    end;
end;

procedure WritePatterns;
var i, j, L: Integer; P: TPattern;
begin
  L := Patterns.Count;
  F.Write(L, 4);
  for j := 1 to L do
    begin
      P := Patterns[j];
      WriteString(P.Name);
      i := P.Channels;
      F.Write(i, 4);
      i := P.Length;
      F.Write(i, 4);
      for i := 1 to P.Channels do
        F.Write(P.FData^[i]^, SizeOf(TCommand)*P.Length);
    end;
end;

procedure WriteSequence;
var L, i, j: Integer;
begin
  L := Sequence.FTracks;
  F.Write(L, 4);
  for j := 1 to L do
    begin
      i := Sequence.GetTrackLength(j);
      F.Write(i, 4);
      F.Write(Sequence.FData[j].Data^, SizeOf(Integer)*i);
    end;
end;

begin
  CF := nil;
  FF := TFileStream.Create(FileName, fmCreate or fmShareDenyWrite);
  try
    Sign := 'SONG';
    FF.Write(Sign[1], 4);
    //Version
    Sign := '0.12';
    FF.Write(Sign[1], 4);
    if Compress then
      begin
        Sign := 'COMP';
        FF.Write(Sign[1], 4);
        CF := TCompressionStream.Create(clMax, FF);
        F := CF;
      end
    else
      begin
        Sign := 'NORM';
        FF.Write(Sign[1], 4);
        F := FF;
      end;

    //General info
    WriteInfo;
    //Samples
    WriteSamples;
    //Patterns
    WritePatterns;
    //Sequence
    WriteSequence;
  finally
    if Assigned(CF) then CF.Free;
    FF.Free;
  end;
end;

end.