unit SngEdit;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ToolWin, Menus, StdCtrls, ExtCtrls, Song, Grids, Player;

type
  TKeyMode = (kmNone, kmSample, kmVolume, kmPan);
  TSongEditForm = class(TForm)
    EdPageControl: TPageControl;
    SamplesTabSheet: TTabSheet;
    PattTabSheet: TTabSheet;
    GenTabSheet: TTabSheet;
    MainMenu: TMainMenu;
    FileItem: TMenuItem;
    EditItem: TMenuItem;
    OptionsItem: TMenuItem;
    HelpItem: TMenuItem;
    SeqTabSheet: TTabSheet;
    ToolBar: TToolBar;
    StatusBar: TStatusBar;
    NewToolButton: TToolButton;
    OpenToolButton: TToolButton;
    SaveToolButton: TToolButton;
    SaveAsToolButton: TToolButton;
    ToolSep1: TToolButton;
    CutToolButton: TToolButton;
    CopyToolButton: TToolButton;
    PasteToolButton: TToolButton;
    ToolSep2: TToolButton;
    PlayToolButton: TToolButton;
    StopToolButton: TToolButton;
    Images: TImageList;
    InfoMemo: TMemo;
    SListBox: TListBox;
    OpenSongDialog: TOpenDialog;
    SaveSongDialog: TSaveDialog;
    OpenSampleDialog: TOpenDialog;
    SaveSampleDialog: TSaveDialog;
    FileNewItem: TMenuItem;
    AboutItem: TMenuItem;
    FileOpenItem: TMenuItem;
    FileSaveItem: TMenuItem;
    FileSaveAsItem: TMenuItem;
    FileSeparator1: TMenuItem;
    FileExitItem: TMenuItem;
    MainPanel: TPanel;
    TempoLabel: TLabel;
    TPBLabel: TLabel;
    NNALabel: TLabel;
    NNAComboBox: TComboBox;
    TitleEdit: TEdit;
    TitleLabel: TLabel;
    AuthorLabel: TLabel;
    AuthorEdit: TEdit;
    StatGroupBox: TGroupBox;
    SamplesLabel: TLabel;
    PatternsLabel: TLabel;
    TracksLabel: TLabel;
    SamplesNumLabel: TLabel;
    PatternsNumLabel: TLabel;
    TracksNumLabel: TLabel;
    InfoLabel: TLabel;
    SPanel: TPanel;
    SInfoPanel: TPanel;
    DVLabel: TLabel;
    DefVolLabel: TLabel;
    DPLabel: TLabel;
    DefPanLabel: TLabel;
    LoopLabel: TLabel;
    LFromLabel: TLabel;
    LToLabel: TLabel;
    LengthLabel: TLabel;
    LengthVLabel: TLabel;
    SRateLabel: TLabel;
    BitLabel: TLabel;
    ChannelsLabel: TLabel;
    DefVolTrackBar: TTrackBar;
    DefPanTrackBar: TTrackBar;
    LTypeComboBox: TComboBox;
    LFromEdit: TEdit;
    LToEdit: TEdit;
    SBPanel: TPanel;
    NewSButton: TButton;
    DelSButton: TButton;
    ImportSButton: TButton;
    ExportSButton: TButton;
    RenameSButton: TButton;
    PlaySButton: TButton;
    PListBox: TListBox;
    PBPanel: TPanel;
    NewPButton: TButton;
    DelPButton: TButton;
    RenamePButton: TButton;
    PlayPButton: TButton;
    PPanel: TPanel;
    PGrid: TDrawGrid;
    PInfoPanel: TPanel;
    SListPanel: TPanel;
    SListLabel: TLabel;
    SListComboBox: TComboBox;
    PCLabel: TLabel;
    PLLabel: TLabel;
    OctLabel: TLabel;
    PCTrackBar: TTrackBar;
    PLTrackBar: TTrackBar;
    PCNLabel: TLabel;
    PLNLabel: TLabel;
    TempoTrackBar: TTrackBar;
    TPBTrackBar: TTrackBar;
    TempoNLabel: TLabel;
    TPBNLabel: TLabel;
    ExportSongDialog: TSaveDialog;
    ToolSep3: TToolButton;
    ExportToolButton: TToolButton;
    PListListBox: TListBox;
    ClearToolButton: TToolButton;
    EditCutItem: TMenuItem;
    EditCopyItem: TMenuItem;
    EditPasteItem: TMenuItem;
    EditClearItem: TMenuItem;
    PlayItem: TMenuItem;
    PlayPlayItem: TMenuItem;
    PlayStopItem: TMenuItem;
    FileExportItem: TMenuItem;
    SpacingLabel: TLabel;
    SpacingTrackBar: TTrackBar;
    SpcLabel: TLabel;
    SeqPanel: TPanel;
    RBPanel: TPanel;
    SeqRBImage: TImage;
    SeqScrollBox: TScrollBox;
    CanvasPanel: TPanel;
    PosPanel: TPanel;
    PosImage: TImage;
    ToolSep4: TToolButton;
    OptionsToolButton: TToolButton;
    ToolSep5: TToolButton;
    MeterPanel: TPanel;
    MLPanel: TPanel;
    MRPanel: TPanel;
    MLVPanel: TPanel;
    MRVPanel: TPanel;
    ClipPanel: TPanel;
    WaveDataPanel: TPanel;
    WaveData: TPaintBox;
    CompCheckBox: TCheckBox;
    ClipToolButton: TToolButton;
    SongVolLabel: TLabel;
    SVNLabel: TLabel;
    SongVolTrackBar: TTrackBar;
    KeyTimer: TTimer;
    NormalizeSButton: TButton;
    EditSButton: TButton;
    TransposePButton: TButton;
    PlayPatternItem: TMenuItem;
    ClearSButton: TButton;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure NewToolButtonClick(Sender: TObject);
    procedure OpenToolButtonClick(Sender: TObject);
    procedure SaveAsToolButtonClick(Sender: TObject);
    procedure SaveToolButtonClick(Sender: TObject);
    procedure TempoTrackBarChange(Sender: TObject);
    procedure TPBTrackBarChange(Sender: TObject);
    procedure NNAComboBoxChange(Sender: TObject);
    procedure TitleEditChange(Sender: TObject);
    procedure AuthorEditChange(Sender: TObject);
    procedure InfoMemoChange(Sender: TObject);
    procedure AboutItemClick(Sender: TObject);
    procedure FileExitItemClick(Sender: TObject);
    procedure SListBoxClick(Sender: TObject);
    procedure NewSButtonClick(Sender: TObject);
    procedure ExportSButtonClick(Sender: TObject);
    procedure ImportSButtonClick(Sender: TObject);
    procedure DelSButtonClick(Sender: TObject);
    procedure RenameSButtonClick(Sender: TObject);
    procedure LTypeComboBoxChange(Sender: TObject);
    procedure LFromEditChange(Sender: TObject);
    procedure LToEditChange(Sender: TObject);
    procedure PlaySButtonClick(Sender: TObject);
    procedure DefVolTrackBarChange(Sender: TObject);
    procedure DefPanTrackBarChange(Sender: TObject);
    procedure BitLabelDblClick(Sender: TObject);
    procedure ChannelsLabelDblClick(Sender: TObject);
    procedure SRateLabelDblClick(Sender: TObject);
    procedure StopToolButtonClick(Sender: TObject);
    procedure WaveDataDblClick(Sender: TObject);
    procedure SListBoxDblClick(Sender: TObject);
    procedure NewPButtonClick(Sender: TObject);
    procedure PGridDrawCell(Sender: TObject; Col, Row: Integer;
      Rect: TRect; State: TGridDrawState);
    procedure DelPButtonClick(Sender: TObject);
    procedure RenamePButtonClick(Sender: TObject);
    procedure PListBoxClick(Sender: TObject);
    procedure PListBoxDblClick(Sender: TObject);
    procedure PGridKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure PCTrackBarChange(Sender: TObject);
    procedure PLTrackBarChange(Sender: TObject);
    procedure PlayPButtonClick(Sender: TObject);
    procedure PlayToolButtonClick(Sender: TObject);
    procedure ExportToolButtonClick(Sender: TObject);
    procedure CutToolButtonClick(Sender: TObject);
    procedure CopyToolButtonClick(Sender: TObject);
    procedure PasteToolButtonClick(Sender: TObject);
    procedure ClearToolButtonClick(Sender: TObject);
    procedure PGridDblClick(Sender: TObject);
    procedure SpacingTrackBarChange(Sender: TObject);
    procedure SeqRBImageDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure SeqRBImageDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure PListListBoxStartDrag(Sender: TObject;
      var DragObject: TDragObject);
    procedure CanvasPanelDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure CanvasPanelDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure SeqEndDrag(Sender, Target: TObject; X, Y: Integer);
    procedure OptionsToolButtonClick(Sender: TObject);
    procedure SeqRBImageDblClick(Sender: TObject);
    procedure WaveDataPaint(Sender: TObject);
    procedure ClipToolButtonClick(Sender: TObject);
    procedure SongVolTrackBarChange(Sender: TObject);
    procedure PGridClick(Sender: TObject);
    procedure KeyTimerTimer(Sender: TObject);
    procedure NormalizeSButtonClick(Sender: TObject);
    procedure EditSButtonClick(Sender: TObject);
    procedure TransposePButtonClick(Sender: TObject);
    procedure PlayPatternItemClick(Sender: TObject);
    procedure ListBoxKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure PListListBoxDblClick(Sender: TObject);
    procedure ClearSButtonClick(Sender: TObject);
    procedure FormResize(Sender: TObject);
  private
    { Private declarations }
    InternalChange: Boolean;
    FSong: TSong;
    FPlayer: TSongPlayer;
    FFileName: String;
    FIsUntitled: Boolean;
    FIsModified: Boolean;
    FLastFile: String;
    FCoeff: Double;
    FKeyMode: TKeyMode;
    Destroying: Boolean;
    procedure SetCaption;
    procedure SetFileName(AFileName: String);
    procedure SetIsUntitled(AIsUntitled: Boolean);
    procedure SetIsModified(AIsModified: Boolean);
    function SaveBeforeAction: Boolean;
    procedure LoadFile(AFileName: String);
    procedure SaveFile(AFileName: String; Compressed: Boolean);
    procedure ReloadSamples;
    procedure ReloadPatterns;
    procedure UpdateStats;
    procedure UpdateCommon;
    procedure UpdateGUI;
    procedure UpdateSample;
    procedure UpdatePattern;
    procedure UpdateSequence;
    procedure CutPattern;
    procedure CopyPattern;
    procedure PastePattern;
    procedure ClearPattern;
    procedure STPanelStartDrag(Sender: TObject; var DragObject: TDragObject);
    procedure STPanelDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure STPanelDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure SPPanelStartDrag(Sender: TObject; var DragObject: TDragObject);
    procedure SPPanelDragOver(Sender, Source: TObject; X, Y: Integer;
      State: TDragState; var Accept: Boolean);
    procedure SPPanelDragDrop(Sender, Source: TObject; X, Y: Integer);
    procedure SaveOptions;
    procedure LoadOptions;
    procedure ResetClip;

    //Message handling
    procedure WMStart(var Message: TMessage); message WM_START;
    procedure WMStop(var Message: TMessage); message WM_STOP;
    procedure WMSetPPos(var Message: TMessage); message WM_SETPPOS;
    procedure WMSetSPos(var Message: TMessage); message WM_SETSPOS;
    procedure WMSetSample(var Message: TMessage); message WM_SETSAMPLE;
    procedure WMClip(var Message: TMessage); message WM_CLIP;
    procedure WMExcept(var Message: TMessage); message WM_EXCEPT;

    property FileName: String read FFileName write SetFileName;
    property IsUntitled: Boolean read FIsUntitled write SetIsUntitled;
    property IsModified: Boolean read FIsModified write SetIsModified;
  public
    { Public declarations }
  end;

var
  SongEditForm: TSongEditForm;

implementation

uses MSConsts, Notes, EvEdit, XClpBrd, IniFiles, OptForm, AboutDlg, SUtils,
  Transp;

{$R *.DFM}

type
  TWhatDrag = (wdPatList, wdTrack, wdPatSeq);

var
  CF: Integer = 0;
  WhatDrag: TWhatDrag;
  DoMove: Boolean = false;

procedure TSongEditForm.SetCaption;
begin
  Caption := ExtractFileName(FileName);
  if IsModified then Caption := Caption+' (*)';
  Caption := Caption+' - '+ProductName;
  Application.Title := Caption;
end;

procedure TSongEditForm.SetFileName(AFileName: String);
begin
  FFileName := AFileName;
  SetCaption;
end;

procedure TSongEditForm.SetIsUntitled(AIsUntitled: Boolean);
begin
  FIsUntitled := AIsUntitled;
  if IsUntitled then FileName := 'Untitled';
end;

procedure TSongEditForm.SetIsModified(AIsModified: Boolean);
begin
  FIsModified := AIsModified;
  SetCaption;
end;

procedure TSongEditForm.FormCreate(Sender: TObject);
begin
  Destroying := false;
  EdPageControl.ActivePage := GenTabSheet;
  OpenSongDialog.DefaultExt := SongExt;
  OpenSongDialog.Filter := SongFilter;
  SaveSongDialog.DefaultExt := SongExt;
  SaveSongDialog.Filter := SongFilter;
  FKeyMode := kmNone;
  InternalChange := false;
  FSong := TSong.Create;
  FileName := 'Untitled';
  IsUntitled := true;
  IsModified := false;
  NewToolButton.Click;
  LoadOptions;
  try
    FPlayer := TSongPlayer.Create(FSong, WaveDevice, BufferCount, BufferSize,
      Handle, LoopPattern);
  except
    FPlayer := nil;
    Application.ShowException(ExceptObject as Exception);
  end;
  PostMessage(Handle, WM_STOP, 0, 0);
  if ParamStr(1)<>'' then LoadFile(ParamStr(1))
  else if FLastFile<>'' then LoadFile(FLastFile);
end;

procedure TSongEditForm.FormDestroy(Sender: TObject);
begin
  Destroying := true;
  FSong.Free;
  if Assigned(FPlayer) then FPlayer.Free;
  SaveOptions;
end;

function TSongEditForm.SaveBeforeAction: Boolean;
var mr: TModalResult;
begin
  result := true;
  if IsModified then
    begin
      mr := MessageDlg('Song '+FileName+' is modified, save ?', mtConfirmation, [mbYes, mbNo, mbCancel], 0);
      result := mr<>mrCancel;
      if mr=mrYes then
        try
          SaveToolButton.Click;
        except
          result := false;
        end;
    end;
end;

procedure TSongEditForm.UpdateStats;
begin
  SamplesNumLabel.Caption := IntToStr(FSong.Samples.Count);
  PatternsNumLabel.Caption := IntToStr(FSong.Patterns.Count);
  TracksNumLabel.Caption := IntToStr(FSong.Sequence.Tracks);
end;

procedure TSongEditForm.UpdateCommon;
begin
  OctLabel.Caption := 'Octave '+IntToStr(BaseOctave);
  SpacingTrackBar.Position := Spacing;
  SpcLabel.Caption := IntToStr(Spacing);
end;

procedure TSongEditForm.ReloadSamples;
var i: Integer;
begin
  SListBox.Items.Clear;
  for i := 1 to FSong.Samples.Count do
    SListBox.Items.Add(FSong.Samples[i].Name);
  if SListBox.Items.Count>0 then SListBox.ItemIndex := 0
  else SListBox.ItemIndex := -1;
end;

procedure TSongEditForm.ReloadPatterns;
var i: Integer;
begin
  PListBox.Items.Clear;
  for i := 1 to FSong.Patterns.Count do
    PListBox.Items.Add(FSong.Patterns[i].Name);
  if PListBox.Items.Count>0 then PListBox.ItemIndex := 0
  else PListBox.ItemIndex := -1;
end;

procedure TSongEditForm.UpdateGUI;
begin
  InternalChange := true;
    try
      TempoTrackBar.Position := FSong.Tempo;
      TempoNLabel.Caption := IntToStr(FSong.Tempo);
      TPBTrackBar.Position := FSong.TPB;
      TPBNLabel.Caption := IntToStr(FSong.TPB);
      NNAComboBox.ItemIndex := Ord(FSong.NNA);
      SongVolTrackBar.Position := FSong.SongVolume;
      SVNLabel.Caption := IntToStr(FSong.SongVolume);
      TitleEdit.Text := FSong.Title;
      AuthorEdit.Text := FSong.Author;
      InfoMemo.Lines.Text := FSong.Info;
      UpdateCommon;
      UpdateStats;
      //Samples
      ReloadSamples;
      UpdateSample;
      //Patterns
      ReloadPatterns;
      UpdatePattern;
      //UpdateSequence; updated in UpdatePattern
    finally
      InternalChange := false;
    end;
end;

procedure TSongEditForm.FormCloseQuery(Sender: TObject;
  var CanClose: Boolean);
begin
  CanClose := SaveBeforeAction;
end;

procedure TSongEditForm.NewToolButtonClick(Sender: TObject);
begin
  if not SaveBeforeAction then Exit;
  FSong.Clear;
  UpdateGUI;
  CompCheckBox.Checked := false;
  FileName := 'Untitled';
  IsUntitled := true;
  IsModified := false;
  ResetClip;
end;

procedure TSongEditForm.LoadFile(AFileName: String);
var SC: TCursor;
begin
  SC := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  try
    AFileName := ExpandFileName(AFileName);
    CompCheckBox.Checked := FSong.LoadFromFile(AFileName);
    UpdateGUI;
    FileName := AFileName;
    IsUntitled := false;
    IsModified := false;
    ResetClip;
  finally
    Screen.Cursor := SC;
  end;
end;

procedure TSongEditForm.SaveFile(AFileName: String; Compressed: Boolean);
var SC: TCursor;
begin
  SC := Screen.Cursor;
  Screen.Cursor := crHourGlass;
  try
    FSong.SaveToFile(AFileName, Compressed);
    IsUntitled := false;
    IsModified := false;
  finally
    Screen.Cursor := SC;
  end;
end;

procedure TSongEditForm.OpenToolButtonClick(Sender: TObject);
begin
  if not SaveBeforeAction then Exit;
  if OpenSongDialog.Execute then
    LoadFile(OpenSongDialog.FileName)
  else Abort;
end;

procedure TSongEditForm.SaveAsToolButtonClick(Sender: TObject);
begin
  if not IsUntitled then SaveSongDialog.FileName := FileName;
  if SaveSongDialog.Execute then
    begin
      SaveFile(SaveSongDialog.FileName, CompCheckBox.Checked);
      FileName := SaveSongDialog.FileName;
    end
  else Abort;
end;

procedure TSongEditForm.SaveToolButtonClick(Sender: TObject);
begin
  if IsUntitled then SaveAsToolButton.Click
  else SaveFile(FileName, CompCheckBox.Checked);
end;

procedure TSongEditForm.TempoTrackBarChange(Sender: TObject);
begin
  FSong.Tempo := TempoTrackBar.Position;
  IsModified := true;
  TempoNLabel.Caption := IntToStr(FSong.Tempo);
end;

procedure TSongEditForm.TPBTrackBarChange(Sender: TObject);
begin
  FSong.TPB := TPBTrackBar.Position;
  IsModified := true;
  TPBNLabel.Caption := IntToStr(FSong.TPB);
end;

procedure TSongEditForm.NNAComboBoxChange(Sender: TObject);
begin
  FSong.NNA := TNNA(NNAComboBox.ItemIndex);
  IsModified := true;
end;

procedure TSongEditForm.TitleEditChange(Sender: TObject);
begin
  if InternalChange then Exit;
  FSong.Title := TitleEdit.Text;
  IsModified := true;
end;

procedure TSongEditForm.AuthorEditChange(Sender: TObject);
begin
  if InternalChange then Exit;
  FSong.Author := AuthorEdit.Text;
  IsModified := true;
end;

procedure TSongEditForm.InfoMemoChange(Sender: TObject);
begin
  if InternalChange then Exit;
  FSong.Info := InfoMemo.Lines.Text;
  IsModified := true;
end;

procedure TSongEditForm.AboutItemClick(Sender: TObject);
begin
  About;
end;

procedure TSongEditForm.FileExitItemClick(Sender: TObject);
begin
  Close;
end;

procedure TSongEditForm.UpdateSample;
begin
  InternalChange := true;
    try
      if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then
        begin
          DelSButton.Enabled := false;
          ImportSButton.Enabled := false;
          ExportSButton.Enabled := false;
          RenameSButton.Enabled := false;
          PlaySButton.Enabled := false;
          NormalizeSButton.Enabled := false;
          ClearSButton.Enabled := false;
          EditSButton.Enabled := false;
          LengthLabel.Visible := false;
          LengthVLabel.Caption := '';
          SRateLabel.Caption := '';
          BitLabel.Caption := '';
          ChannelsLabel.Caption := '';
          DVLabel.Enabled := false;
          DPLabel.Enabled := false;
          DefVolLabel.Enabled := false;
          DefVolLabel.Caption := '100';
          DefPanLabel.Enabled := false;
          DefPanLabel.Caption := 'C';
          DefVolTrackBar.Enabled := false;
          DefVolTrackBar.Position := 100;
          DefPanTrackBar.Enabled := false;
          DefPanTrackBar.Position := 0;
          LoopLabel.Enabled := false;
          LTypeComboBox.Enabled := false;
          LTypeComboBox.ItemIndex := -1;
          LFromLabel.Enabled := false;
          LFromEdit.Enabled := false;
          LFromEdit.Text := '';
          LToLabel.Enabled := false;
          LToEdit.Enabled := false;
          LToEdit.Text := '';
        end
      else with FSong.Samples[SListBox.ItemIndex+1] do
        begin
          SListBox.Items[SListBox.ItemIndex] :=
            FSong.Samples[SListBox.ItemIndex+1].Name;
          DelSButton.Enabled := true;
          ImportSButton.Enabled := true;
          ExportSButton.Enabled := Length>0;
          RenameSButton.Enabled := true;
          PlaySButton.Enabled := Length>0;
          NormalizeSButton.Enabled := Length>0;
          ClearSButton.Enabled := Length>0;
          EditSButton.Enabled := WaveEditor<>'';
          LengthLabel.Visible := true;
          LengthVLabel.Caption := IntToStr(Length);
          SRateLabel.Caption := IntToStr(Rate)+' Hz';
          BitLabel.Caption := IntToStr(Bits)+' bit';
          if Channels=1 then
            ChannelsLabel.Caption := 'Mono'
          else
            ChannelsLabel.Caption := 'Stereo';
          DVLabel.Enabled := true;
          DPLabel.Enabled := true;
          DefVolLabel.Enabled := true;
          DefVolLabel.Caption := IntToStr(DefVolume);
          DefPanLabel.Enabled := true;
          if DefPan=0 then
            DefPanLabel.Caption := 'C'
          else
          if DefPan>0 then
            DefPanLabel.Caption := IntToStr(Abs(DefPan))+' R'
          else
          if DefPan<0 then
            DefPanLabel.Caption := IntToStr(Abs(DefPan))+' L';
          DefVolTrackBar.Enabled := true;
          DefVolTrackBar.Position := DefVolume;
          DefPanTrackBar.Enabled := true;
          DefPanTrackBar.Position := DefPan;
          LoopLabel.Enabled := true;
          LTypeComboBox.Enabled := true;
          LTypeComboBox.ItemIndex := Ord(LoopType);
          LFromLabel.Enabled := true;
          LFromEdit.Enabled := true;
          LFromEdit.Text := IntToStr(LoopStart);
          LToLabel.Enabled := true;
          LToEdit.Enabled := true;
          LToEdit.Text := IntToStr(LoopEnd);
        end;
      UpdateStats;
      SListComboBox.Items.Assign(SListBox.Items);
      SListComboBox.Items.Insert(0, 'No sample');
      SListComboBox.ItemIndex := SListBox.ItemIndex+1;
      WaveData.Repaint;
    finally
      InternalChange := false;
    end;
end;

procedure TSongEditForm.SListBoxClick(Sender: TObject);
begin
  UpdateSample;
end;

procedure TSongEditForm.NewSButtonClick(Sender: TObject);
begin
  FSong.Samples.Add(TSample.Create);
  FSong.Samples[FSong.Samples.Count].Name :=
    'Sample '+IntToStr(FSong.Samples.Count);
  SListBox.Items.Add(FSong.Samples[FSong.Samples.Count].Name);
  SListBox.ItemIndex := FSong.Samples.Count-1;
  UpdateSample;
  IsModified := true;
end;

procedure TSongEditForm.ExportSButtonClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  with FSong.Samples[SListBox.ItemIndex+1] do
    begin
      if Length=0 then
        Raise Exception.Create('Empty sample cannot be exported');
      SaveSampleDialog.FileName := Name;
      if SaveSampleDialog.Execute then
        SaveToFile(SaveSampleDialog.FileName);
    end;
end;

procedure TSongEditForm.ImportSButtonClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  with FSong.Samples[SListBox.ItemIndex+1] do
    begin
      if Length<>0 then
        if MessageDlg('Replace current sample ?', mtConfirmation,
                      [mbYes, mbNo], 0)<>mrYes then Exit;
      if OpenSampleDialog.Execute then
        begin
          LoadFromFile(OpenSampleDialog.FileName);
          FSong.Samples[SListBox.ItemIndex+1].Name :=
            ChangeFileExt(ExtractFileName(OpenSampleDialog.FileName), '');
          IsModified := true;
          UpdateSample;
        end;
    end;
end;

procedure TSongEditForm.DelSButtonClick(Sender: TObject);
var SI: Integer;
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  with FSong.Samples[SListBox.ItemIndex+1] do
    begin
      if Length<>0 then
        if MessageDlg('Delete current sample ?', mtConfirmation,
                      [mbYes, mbNo], 0)<>mrYes then Exit;
      FSong.Samples.Delete(SListBox.ItemIndex+1);
      SI := SListBox.ItemIndex;
      SListBox.Items.Delete(SListBox.ItemIndex);
      IsModified := true;
      if SI>=SListBox.Items.Count then SI := SI-1;
      SListBox.ItemIndex := SI;
      UpdateSample;
    end;
end;

procedure TSongEditForm.RenameSButtonClick(Sender: TObject);
var S: String;
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  S := FSong.Samples[SListBox.ItemIndex+1].Name;
  if InputQuery('Rename sample', 'New sample name', S) then
    begin
      FSong.Samples[SListBox.ItemIndex+1].Name := S;
      IsModified := true;
      UpdateSample;
    end;
end;

procedure TSongEditForm.LTypeComboBoxChange(Sender: TObject);
begin
  if InternalChange then Exit;
  InternalChange := true;
  try
    if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      begin
        LoopType := TLoopType(LTypeComboBox.ItemIndex);
        IsModified := true;
        UpdateSample;
      end;
  finally
    InternalChange := false;
  end;
end;

procedure TSongEditForm.LFromEditChange(Sender: TObject);
begin
  if InternalChange then Exit;
  InternalChange := true;
  try
    if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      try
        LoopStart := StrToInt(LFromEdit.Text);
        IsModified := true;
        UpdateSample;
      except
        LFromEdit.Text := IntToStr(LoopStart);
        Raise;
      end;
  finally
    InternalChange := false;
  end;
end;

procedure TSongEditForm.LToEditChange(Sender: TObject);
begin
  if InternalChange then Exit;
  InternalChange := true;
  try
    if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      try
        LoopEnd := StrToInt(LToEdit.Text);
        IsModified := true;
        UpdateSample;
      except
        LToEdit.Text := IntToStr(LoopEnd);
        Raise;
      end;
  finally
    InternalChange := false;
  end;
end;

procedure TSongEditForm.PlaySButtonClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  if Assigned(FPlayer) then
    FPlayer.StartPlaySample(SListBox.ItemIndex+1);
end;

procedure TSongEditForm.DefVolTrackBarChange(Sender: TObject);
begin
  if InternalChange then Exit;
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      begin
        DefVolume := DefVolTrackBar.Position;
        IsModified := true;
        UpdateSample;
      end;
end;

procedure TSongEditForm.DefPanTrackBarChange(Sender: TObject);
begin
  if InternalChange then Exit;
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      begin
        DefPan := DefPanTrackBar.Position;
        IsModified := true;
        UpdateSample;
      end;
end;

procedure TSongEditForm.BitLabelDblClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      if Bits<>16 then
        begin
          Bits := 16;
          IsModified := true;
          UpdateSample;
        end;
end;

procedure TSongEditForm.ChannelsLabelDblClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      if Channels<>2 then
        begin
          Channels := 2;
          IsModified := true;
          UpdateSample;
        end;
end;

procedure TSongEditForm.SRateLabelDblClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
    with FSong.Samples[SListBox.ItemIndex+1] do
      if Rate<>44100 then
        begin
          Resample(44100);
          IsModified := true;
          UpdateSample;
        end;
end;

procedure TSongEditForm.StopToolButtonClick(Sender: TObject);
begin
  if Assigned(FPlayer) then
    FPlayer.StopPlay;
end;

procedure TSongEditForm.WaveDataDblClick(Sender: TObject);
begin
  PlaySButton.Click;
end;

procedure TSongEditForm.SListBoxDblClick(Sender: TObject);
begin
  RenameSButton.Click;
end;

procedure TSongEditForm.UpdatePattern;
begin
  InternalChange := true;
    try
      if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then
        begin
          DelPButton.Enabled := false;
          RenamePButton.Enabled := false;
          PlayPButton.Enabled := false;
          TransposePButton.Enabled := false;
          PCLabel.Enabled := false;
          PCNLabel.Enabled := false;
          PCNLabel.Caption := '1';
          PCTrackBar.Enabled := false;
          PCTrackBar.Position := 1;
          PLLabel.Enabled := false;
          PLNLabel.Enabled := false;
          PLNLabel.Caption := '1';
          PLTrackBar.Enabled := false;
          PLTrackBar.Position := 1;
          PGrid.Enabled := false;
          PGrid.Enabled := false;
          PGrid.RowCount := 0;
          PGrid.ColCount := 0;
          PGrid.Repaint;
        end
      else with FSong.Patterns[PListBox.ItemIndex+1] do
        begin
          PListBox.Items[PListBox.ItemIndex] :=
            FSong.Patterns[PListBox.ItemIndex+1].Name;
          DelPButton.Enabled := true;
          RenamePButton.Enabled := true;
          PlayPButton.Enabled := Length>0;
          TransposePButton.Enabled := Length>0;
          PCLabel.Enabled := true;
          PCNLabel.Enabled := true;
          PCTrackBar.Enabled := true;
          PCNLabel.Caption := IntToStr(Channels);
          PCTrackBar.Position := Channels;
          PLLabel.Enabled := true;
          PLNLabel.Enabled := true;
          PLTrackBar.Enabled := true;
          PLNLabel.Caption := IntToStr(Length);
          PLTrackBar.Position := Length;
          PGrid.Enabled := true;
          PGrid.RowCount := Length+1;
          PGrid.ColCount := Channels+1;
          PGrid.ColWidths[0] := 30;
          PGrid.FixedRows := 1;
          PGrid.FixedCols := 1;
          PGrid.Repaint;
        end;
      FKeyMode := kmNone;
      UpdateStats;
      PListListBox.Items.Assign(PListBox.Items);
      PListListBox.ItemIndex := PListBox.ItemIndex;
      UpdateSequence;
    finally
      InternalChange := false;
    end;
end;

procedure TSongEditForm.NewPButtonClick(Sender: TObject);
begin
  FSong.Patterns.Add(TPattern.Create);
  FSong.Patterns[FSong.Patterns.Count].Name :=
    'Pattern '+IntToStr(FSong.Patterns.Count);
  FSong.Patterns[FSong.Patterns.Count].Length := FSong.TPB*4;
  PListBox.Items.Add(FSong.Patterns[FSong.Patterns.Count].Name);
  PListBox.ItemIndex := FSong.Patterns.Count-1;
  UpdatePattern;
  IsModified := true;
end;

procedure TSongEditForm.PGridDrawCell(Sender: TObject; Col,
  Row: Integer; Rect: TRect; State: TGridDrawState);
procedure MyDrawText(S: String; R: TRect);
var SW, SH, XX, YY: Integer;
begin
  if S='' then Exit;
  with PGrid.Canvas do
    begin
      SW := TextWidth(S);
      SH := TextHeight(S);
      XX := R.Left+((R.Right-R.Left+1)-SW) div 2;
      YY := R.Top+((R.Bottom-R.Top+1)-SH) div 2;
      TextRect(R, XX, YY, S);
    end;
end;
var Text: String; R: TRect;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  with PGrid.Canvas do
    begin
      if (Col=0) and (Row=0) then Text := '***'
      else
      if (Col=0) and (Row>0) then Text := IntToStr(Row)
      else
      if (Col>0) and (Row=0) then Text := IntToStr(Col)
      else
      if (Col<=FSong.Patterns[PListBox.ItemIndex+1].Channels) and
         (Row<=FSong.Patterns[PListBox.ItemIndex+1].Length) then
        Text := CommandToText(FSong.Patterns[PListBox.ItemIndex+1]
          [Col, Row])
      else Text := '';

      Font.Assign(PGrid.Font);
      if gdSelected in State then
        begin
          Brush.Color := ColorSet.PEditColorS;
          Font.Color := ColorSet.PEditColorST;
        end
      else
      if not (gdFixed in State) and ((Row-1) mod FSong.TPB=0) and
         ((Row-1) div FSong.TPB mod 4<>0) then
        begin
          Brush.Color := ColorSet.PEditColorB;
          Font.Color := ColorSet.PEditColorBT;
        end
      else
      if not (gdFixed in State) and ((Row-1) mod FSong.TPB=0) and
         ((Row-1) div FSong.TPB mod 4=0) then
        begin
          Brush.Color := ColorSet.PEditColorM;
          Font.Color := ColorSet.PEditColorMT;
        end
      else
        begin
          Brush.Color := ColorSet.PEditColorN;
          Font.Color := ColorSet.PEditColorNT;
        end;

      R := Rect;
      Brush.Style := bsSolid;
      if gdFixed in State then
        begin
          DrawFrameControl(Handle, Rect, DFC_BUTTON, DFCS_BUTTONPUSH);
          InflateRect(R, -2, -2);
          Brush.Color := clBtnFace;
          Font.Color := clBtnText;
        end
      else
        begin
          FillRect(Rect);
        end;

      Brush.Style := bsClear;
      MyDrawText(Text, R);

      Pen.Color := ColorSet.PEditColorL;
      Pen.Style := psSolid;
      if (Col>0) {(Col=PGrid.ColCount-1)}and (Row>0) then
        begin
          MoveTo(Rect.Right-1, Rect.Top);
          LineTo(Rect.Right-1, Rect.Bottom);
        end;
      if (Row=PGrid.RowCount-1) and (Col>0) then
        begin
          MoveTo(Rect.Left, Rect.Bottom-1);
          LineTo(Rect.Right, Rect.Bottom-1);
        end;
      if gdFocused in State then DrawFocusRect(Rect);
    end;
end;

procedure TSongEditForm.DelPButtonClick(Sender: TObject);
var SI: Integer;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  with FSong.Patterns[PListBox.ItemIndex+1] do
    begin
      if MessageDlg('Delete current pattern ?', mtConfirmation,
                    [mbYes, mbNo], 0)<>mrYes then Exit;
      FSong.Patterns.Delete(PListBox.ItemIndex+1);
      SI := PListBox.ItemIndex;
      PListBox.Items.Delete(PListBox.ItemIndex);
      IsModified := true;
      if SI>=PListBox.Items.Count then SI := SI-1;
      PListBox.ItemIndex := SI;
      UpdatePattern;
    end;
end;

procedure TSongEditForm.RenamePButtonClick(Sender: TObject);
var S: String;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  S := FSong.Patterns[PListBox.ItemIndex+1].Name;
  if InputQuery('Rename pattern', 'New pattern name', S) then
    begin
      FSong.Patterns[PListBox.ItemIndex+1].Name := S;
      IsModified := true;
      UpdatePattern;
    end;
end;

procedure TSongEditForm.PListBoxClick(Sender: TObject);
begin
  UpdatePattern;
end;

procedure TSongEditForm.PListBoxDblClick(Sender: TObject);
begin
  RenamePButton.Click;
end;

procedure TSongEditForm.PGridKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
var C: TCommand; i: Integer; KM: TKeyMode; S: String;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  KM := FKeyMode;
  FKeyMode := kmNone;
  if Shift=[] then
    begin
      if Key=VK_RETURN then
        begin
          PGridDblClick(PGrid);
          Exit;
        end;
      C := FSong.Patterns[PListBox.ItemIndex+1]
             [PGrid.Col, PGrid.Row];
      C.Note := KeyToNote(Key);
      case C.Note of
        NoteIns:
          begin
            for i := FSong.Patterns[PListBox.ItemIndex+1].Length
                downto PGrid.Row+1 do
              FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, i] :=
              FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, i-1];
            C := FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row];
            C.Note := NoNote;
            FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row] := C;
            IsModified := true;
            UpdatePattern;
          end;
        NoteDel:
          begin
            for i := PGrid.Row to
                FSong.Patterns[PListBox.ItemIndex+1].Length-1 do
              FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, i] :=
              FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, i+1];
            C := FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col,
                                FSong.Patterns[PListBox.ItemIndex+1].Length];
            C.Note := NoNote;
            FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col,
                           FSong.Patterns[PListBox.ItemIndex+1].Length] := C;
            IsModified := true;
            UpdatePattern;
          end;
        NoteOctUp:
          begin
            if BaseOctave<9 then
              begin
                Inc(BaseOctave);
                UpdateCommon;
              end;
          end;
        NoteOctDown:
          begin
            if BaseOctave>0 then
              begin
                Dec(BaseOctave);
                UpdateCommon;
              end;
          end;
        else
          if C.Note<>InvKeyNote then
            begin
              if C.Instrument=0 then
                C.Instrument := SListComboBox.ItemIndex;
{              if C.Instrument=0 then
                C.Note := NoNote;}
{              if (C.Note=NoNote) and (Shift=[ssShift]) then
                begin
                  C.Volume := DVol;
                  C.Pan := DPan;
                end;}
              FSong.Patterns[PListBox.ItemIndex+1]
                [PGrid.Col, PGrid.Row] := C;
              if C.Note=NoteCut then
                begin
                  if PlayWhileEdit and Assigned(FPlayer) then
                    FPlayer.StopNote(0, PGrid.Col);
                end
              else
              if C.Note=NoteOff then
                begin
                  if PlayWhileEdit and Assigned(FPlayer) then
                    FPlayer.XStopNote(0, PGrid.Col);
                end
              else
              if C.Note<>NoNote then
                begin
                  if PlayWhileEdit and Assigned(FPlayer) then
                    FPlayer.XStartNote(0, PGrid.Col, C);
                end;
              for i := 1 to Spacing do
                if PGrid.Row=PGrid.RowCount-1 then
                  PGrid.Row := 1
                else PGrid.Row := PGrid.Row+1;
              IsModified := true;
              UpdatePattern;
            end;
      end;
    end
  else
  if (Shift=[ssShift]) and (Key=VK_SPACE) then
    begin
      C := FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row];
      C.Note := NoNote;
      C.Instrument := 0;
      C.Volume := DVol;
      C.Pan := DPan;
      FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row] := C;
      IsModified := true;
      UpdatePattern;
    end
  else
  if (Shift=[ssShift]) and (Key>=Ord('0')) and (Key<=Ord('9')) then
    begin
      C := FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row];
      if KM=kmSample then
        S := IntToStr(C.Instrument)+Char(Key)
      else S := Char(Key);
      i := StrToInt(S);
      if (i<0) or (i>FSong.Samples.Count) then Exit;
      C.Instrument := i;
      FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row] := C;
      IsModified := true;
      UpdatePattern;
      FKeyMode := kmSample;
      KeyTimer.Enabled := false;
      KeyTimer.Enabled := true;
    end
  else
  if (Shift=[ssCtrl]) and ((Key>=Ord('0')) and (Key<=Ord('9')) or
                           (Key=K_DEFAULT)) then
    begin
      C := FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row];
      if Key=K_DEFAULT then
        S := IntToStr(DVol)
      else
      if KM=kmVolume then
        S := IntToStr(C.Volume)+Char(Key)
      else S := Char(Key);
      i := StrToInt(S);
      if ((i<0) or (i>200)) and (i<>DVol) then Exit;
      C.Volume := i;
      FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row] := C;
      IsModified := true;
      UpdatePattern;
      if Key=K_DEFAULT then Exit;
      FKeyMode := kmVolume;
      KeyTimer.Enabled := false;
      KeyTimer.Enabled := true;
    end
  else
  if (Shift=[ssAlt]) and ((Key>=Ord('0')) and (Key<=Ord('9')) or
                          (Key=K_REVERSE) or (Key=K_DEFAULT)) then
    begin
      C := FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row];
      if Key=K_DEFAULT then
        S := IntToStr(DPan)
      else
      if Key=K_REVERSE then
        S := IntToStr(-C.Pan)
      else
      if KM=kmPan then
        S := IntToStr(C.Pan)+Char(Key)
      else S := Char(Key);
      i := StrToInt(S);
      if ((i<-100) or (i>100)) and (i<>DPan) then Exit;
      C.Pan := i;
      FSong.Patterns[PListBox.ItemIndex+1][PGrid.Col, PGrid.Row] := C;
      IsModified := true;
      UpdatePattern;
      if (Key=K_REVERSE) or (Key=K_DEFAULT) then Exit;
      FKeyMode := kmPan;
      KeyTimer.Enabled := false;
      KeyTimer.Enabled := true;
    end
  else
  if ssAlt in Shift then
    begin
      if ssShift in Shift then i := 10 else i := 1;
      case Key of
        VK_UP:
          begin
            if FSong.Patterns[PListBox.ItemIndex+1].Length>=
              PLTrackBar.Min+i then
              FSong.Patterns[PListBox.ItemIndex+1].Length :=
                FSong.Patterns[PListBox.ItemIndex+1].Length-i;
          end;
        VK_DOWN:
          begin
            if FSong.Patterns[PListBox.ItemIndex+1].Length<=
              PLTrackBar.Max-i then
              FSong.Patterns[PListBox.ItemIndex+1].Length :=
                FSong.Patterns[PListBox.ItemIndex+1].Length+i;
          end;
        VK_LEFT:
          begin
            if FSong.Patterns[PListBox.ItemIndex+1].Channels>=
              PCTrackBar.Min+i then
              FSong.Patterns[PListBox.ItemIndex+1].Channels :=
                FSong.Patterns[PListBox.ItemIndex+1].Channels-i;
          end;
        VK_RIGHT:
          begin
            if FSong.Patterns[PListBox.ItemIndex+1].Channels<=
              PCTrackBar.Max-i then
              FSong.Patterns[PListBox.ItemIndex+1].Channels :=
                FSong.Patterns[PListBox.ItemIndex+1].Channels+i;
          end;
        else Exit;
      end;
      Key := 0;
      IsModified := true;
      UpdatePattern;
    end;
end;

procedure TSongEditForm.PCTrackBarChange(Sender: TObject);
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  try
    FSong.Patterns[PListBox.ItemIndex+1].Channels :=
      PCTrackBar.Position;
    IsModified := true;
    UpdatePattern;
  finally
    InternalChange := false;
  end;
end;

procedure TSongEditForm.PLTrackBarChange(Sender: TObject);
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  try
    FSong.Patterns[PListBox.ItemIndex+1].Length :=
      PLTrackBar.Position;
    IsModified := true;
    UpdatePattern;
  finally
    InternalChange := false;
  end;
end;

procedure TSongEditForm.PlayPButtonClick(Sender: TObject);
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  if Assigned(FPlayer) then
    FPlayer.StartPlayPattern(PListBox.ItemIndex+1);
end;

procedure TSongEditForm.PlayToolButtonClick(Sender: TObject);
begin
  if Assigned(FPlayer) then
    FPlayer.StartPlaySong;
end;

procedure TSongEditForm.ExportToolButtonClick(Sender: TObject);
var SC: TCursor;
begin
  if not Assigned(FPlayer) then Exit;
  if not IsUntitled then ExportSongDialog.FileName :=
    ChangeFileExt(FileName, '.wav');
  if ExportSongDialog.Execute then
    begin
      SC := Screen.Cursor;
      Screen.Cursor := crHourGlass;
      try
        if Assigned(FPlayer) then
          FPlayer.Render(ExportSongDialog.FileName);
      finally
        Screen.Cursor := SC;
      end;
    end;
end;

procedure TSongEditForm.UpdateSequence;
var i, j, XX, SL, ST: Integer; PT, PP: TPanel;

function CalcWidth: Integer;
var i, j, L: Integer;
begin
  result := 0;
  with FSong do
    for i := 1 to Sequence.Tracks do
      begin
        L := 0;
        for j := 1 to Sequence.TrackLength[i] do
          if (Sequence[i, j]>0) and
             (Sequence[i, j]<=Patterns.Count) then
            L := L+Patterns[FSong.Sequence[i, j]].Length
          else L := L+TPEmpty;
        L := L+TPFree;
        if L>result then result := L;
      end;
  result := result*PPTick+PosImage.Width;
end;

function CalcHeight: Integer;
begin
  result := (FSong.Sequence.Tracks+1)*PPTrack+PosPanel.Height;
end;

procedure ClearInvalid;
var i, j: Integer;
begin
  i := 1;
  while i<=FSong.Sequence.Tracks do
    begin
      j := 1;
      while j<=FSong.Sequence.TrackLength[i] do
        if (FSong.Sequence[i, j]<1) or
           (FSong.Sequence[i, j]>FSong.Patterns.Count) then
          begin
            FSong.Sequence.DeletePattern(i, j);
            IsModified := true;
          end
        else Inc(j);
      if (FSong.Sequence.Tracks>1) and (FSong.Sequence.TrackLength[i]=0) then
        begin
          FSong.Sequence.DeleteTrack(i);
          IsModified := true;
        end
      else Inc(i);
    end;
end;

begin
  SL := CanvasPanel.Width-SeqScrollBox.HorzScrollBar.Position;
  ST := CanvasPanel.Height-SeqScrollBox.VertScrollBar.Position;
//  CanvasPanel.Visible := false;
  try
    while CanvasPanel.ControlCount>1 do
      with TPanel(CanvasPanel.Controls[1]) do
        begin
          while ControlCount>0 do Controls[0].Free;
          Free;
        end;
    ClearInvalid;
    CanvasPanel.Width := CalcWidth;
    CanvasPanel.Height := CalcHeight;
    //Now scrollbars updated and client width is correct
    if CanvasPanel.Width<SeqScrollBox.ClientWidth then
      CanvasPanel.Width := SeqScrollBox.ClientWidth;
    if CanvasPanel.Height<SeqScrollBox.ClientHeight then
      CanvasPanel.Height := SeqScrollBox.ClientHeight;

    with FSong do
      for i := 1 to Sequence.Tracks do
        begin
          PT := TPanel.Create(CanvasPanel);
          PT.Parent := CanvasPanel;
          PT.DragMode := dmAutomatic;
          PT.BevelOuter := bvLowered;
          PT.Tag := i;
          PT.OnStartDrag := STPanelStartDrag;
          PT.OnDragOver := STPanelDragOver;
          PT.OnDragDrop := STPanelDragDrop;
          PT.OnEndDrag := SeqEndDrag;
          PT.Caption := '';
          PT.Color := ColorSet.SeqColorT;
          PT.Left := 0;
          PT.Top := (i-1)*PPTrack+PosPanel.Height;
          PT.Width := CanvasPanel.Width;
          PT.Height := PPTrack;
          XX := 0;
          for j := 1 to Sequence.TrackLength[i] do
            begin
              PP := TPanel.Create(PT);
              PP.Parent := PT;
              PP.DragMode := dmAutomatic;
              PP.BevelOuter := bvRaised;
              PP.Tag := j;
              PP.OnStartDrag := SPPanelStartDrag;
              PP.OnDragOver := SPPanelDragOver;
              PP.OnDragDrop := SPPanelDragDrop;
              PP.OnEndDrag := SeqEndDrag;
              PP.Top := 0;
              PP.Left := XX+PosImage.Width;
              PP.Height := PPTrack;
              if (Sequence[i, j]>0) and
                 (Sequence[i, j]<=Patterns.Count) then
                begin
                  PP.Color := ColorSet.SeqColorP;
                  PP.Font.Color := ColorSet.SeqColorPT;
                  PP.Width := Patterns[Sequence[i, j]].Length*PPTick;
                  PP.Caption := Patterns[Sequence[i, j]].Name;
                end
              else
                begin
                  PP.Color := ColorSet.SeqColorI;
                  PP.Font.Color := ColorSet.SeqColorIT;
                  PP.Width := TPEmpty*PPTick;
                  PP.Caption := EmptyP;
                end;
              XX := XX+PP.Width;
            end;
        end;
    UpdateStats;
  finally
//    CanvasPanel.Visible := true;
    SeqScrollBox.HorzScrollBar.Position := CanvasPanel.Width-SL;
    SeqScrollBox.VertScrollBar.Position := CanvasPanel.Height-ST;
  end;
end;

procedure TSongEditForm.CutPattern;
begin
  CopyPattern;
  ClearPattern;
end;

procedure TSongEditForm.CopyPattern;
var i, j: Integer; C: TCommand; S: String;

function C2S(C: TCommand): String;
begin
  SetLength(result, SizeOf(TCommand));
  Move(C, result[1], SizeOf(TCommand));
end;

function I2S(I: Integer): String;
begin
  SetLength(result, SizeOf(Integer));
  Move(I, result[1], SizeOf(Integer));
end;

begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  with FSong.Patterns[PListBox.ItemIndex+1] do
    begin
      S := I2S(PGrid.Selection.Right-PGrid.Selection.Left+1)+
           I2S(PGrid.Selection.Bottom-PGrid.Selection.Top+1);
      for i := PGrid.Selection.Left to PGrid.Selection.Right do
        for j := PGrid.Selection.Top to PGrid.Selection.Bottom do
          begin
            C := Command[i, j];
            S := S+C2S(C);
          end;
      SetClipboardStr(CF, S);
    end;
end;

procedure TSongEditForm.PastePattern;
var i, j, k, CC, RR, NC, NR: Integer; C: TCommand; S: String;

function S2C(S: String): TCommand;
begin
  Move(S[1], result, Length(S));
end;

function S2I(S: String): Integer;
begin
  Move(S[1], result, Length(S));
end;

function GetStr(N: Integer): String;
begin
  if k>Length(S) then Exit;
  result := Copy(S, k, N);
  k := k+N;
end;

begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  with FSong.Patterns[PListBox.ItemIndex+1] do
    begin
      S := GetClipboardStr(CF);
      if S='' then Exit;
      CC := PGrid.Col;
      RR := PGrid.Row;
      k := 1;
      NC := S2I(GetStr(SizeOf(Integer)));
      NR := S2I(GetStr(SizeOf(Integer)));
      for i := 1 to NC do
        for j := 1 to NR do
          begin
            if k<=System.Length(S) then
              begin
                C := S2C(GetStr(SizeOf(TCommand)));
                if (CC+i-1<=Channels) and (RR+j-1<=Length) then
                  Command[CC+i-1, RR+j-1] := C;
              end;
          end;
      IsModified := true;
      UpdatePattern;
    end;
end;

procedure TSongEditForm.ClearPattern;
var i, j: Integer; C: TCommand;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  with FSong.Patterns[PListBox.ItemIndex+1] do
    begin
      C.Note := NoNote;
      C.Instrument := 0;
      C.Volume := DVol;
      C.Pan := DPan;
      for i := PGrid.Selection.Left to PGrid.Selection.Right do
        for j := PGrid.Selection.Top to PGrid.Selection.Bottom do
          Command[i, j] := C;
      IsModified := true;
      UpdatePattern;
    end;
end;

procedure TSongEditForm.CutToolButtonClick(Sender: TObject);
begin
  if not Assigned(ActiveControl) then Exit;
  if (ActiveControl is TEdit) then
    (ActiveControl as TEdit).CutToClipboard
  else
  if (ActiveControl is TMemo) then
    (ActiveControl as TMemo).CutToClipboard
  else
  if ActiveControl=PGrid then CutPattern;
end;

procedure TSongEditForm.CopyToolButtonClick(Sender: TObject);
begin
  if not Assigned(ActiveControl) then Exit;
  if (ActiveControl is TEdit) then
    (ActiveControl as TEdit).CopyToClipboard
  else
  if (ActiveControl is TMemo) then
    (ActiveControl as TMemo).CopyToClipboard
  else
  if ActiveControl=PGrid then CopyPattern;
end;

procedure TSongEditForm.PasteToolButtonClick(Sender: TObject);
begin
  if not Assigned(ActiveControl) then Exit;
  if (ActiveControl is TEdit) then
    (ActiveControl as TEdit).PasteFromClipboard
  else
  if (ActiveControl is TMemo) then
    (ActiveControl as TMemo).PasteFromClipboard
  else
  if ActiveControl=PGrid then PastePattern;
end;

procedure TSongEditForm.ClearToolButtonClick(Sender: TObject);
begin
  if not Assigned(ActiveControl) then Exit;
  if (ActiveControl is TEdit) then
    (ActiveControl as TEdit).ClearSelection
  else
  if (ActiveControl is TMemo) then
    (ActiveControl as TMemo).ClearSelection
  else
  if ActiveControl=PGrid then ClearPattern;
end;

procedure TSongEditForm.PGridDblClick(Sender: TObject);
var FR, FC: Integer; C: TCommand;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  FR := PGrid.Row;
  FC := PGrid.Col;
  C := FSong.Patterns[PListBox.ItemIndex+1][FC, FR];
  if EditEvent(FSong, C) then
    begin
      FSong.Patterns[PListBox.ItemIndex+1][FC, FR] := C;
      IsModified := true;
      UpdatePattern;
    end;
end;

procedure TSongEditForm.SpacingTrackBarChange(Sender: TObject);
begin
  Spacing := SpacingTrackBar.Position;
  UpdateCommon;
end;

procedure TSongEditForm.SeqRBImageDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  Accept := (WhatDrag<>wdPatList) and (Source is TPanel);{ and
            ((WhatDrag<>wdTrack) or (FSong.Sequence.Tracks>1));}
end;

procedure TSongEditForm.SeqRBImageDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var DP, DT: Integer;
begin
  if WhatDrag=wdPatList then Exit;
  if WhatDrag=wdTrack then
    begin
      DT := TPanel(Source).Tag;
      if FSong.Sequence.Tracks>1 then 
        FSong.Sequence.DeleteTrack(DT)
      else FSong.Sequence.TrackLength[DT] := 0;
    end
  else
  if WhatDrag=wdPatSeq then
    begin
      DP := TPanel(Source).Tag;
      DT := TPanel(TPanel(Source).Owner).Tag;
      FSong.Sequence.DeletePattern(DT, DP);
    end;
  IsModified := true;
end;

procedure TSongEditForm.PListListBoxStartDrag(Sender: TObject;
  var DragObject: TDragObject);
begin
  WhatDrag := wdPatList;
end;

procedure TSongEditForm.CanvasPanelDragOver(Sender, Source: TObject; X,
  Y: Integer; State: TDragState; var Accept: Boolean);
begin
  Accept := (Source<>PListListBox) or
            (PListListBox.ItemIndex<>-1);
end;

procedure TSongEditForm.CanvasPanelDragDrop(Sender, Source: TObject; X,
  Y: Integer);
var DT, DP: Integer;
begin
  if WhatDrag=wdTrack then
    begin
      DT := TPanel(Source).Tag;
      FSong.Sequence.Tracks := FSong.Sequence.Tracks+1;
      FSong.Sequence.CopyTrack(DT, FSong.Sequence.Tracks);
      if DoMove then FSong.Sequence.DeleteTrack(DT);
      IsModified := true;
    end
  else
  if WhatDrag=wdPatSeq then
    begin
      DP := TPanel(Source).Tag;
      DT := TPanel(TPanel(Source).Owner).Tag;
      FSong.Sequence.Tracks := FSong.Sequence.Tracks+1;
      FSong.Sequence.TrackLength[FSong.Sequence.Tracks] := 1;
      FSong.Sequence[FSong.Sequence.Tracks, 1] := FSong.Sequence[DT, DP];
      if DoMove then FSong.Sequence.DeletePattern(DT, DP);
      IsModified := true;
    end
  else
  if WhatDrag=wdPatList then
    begin
      DP := PListListBox.ItemIndex+1;
      if DP=0 then Exit;
      FSong.Sequence.Tracks := FSong.Sequence.Tracks+1;
      FSong.Sequence.TrackLength[FSong.Sequence.Tracks] := 1;
      FSong.Sequence[FSong.Sequence.Tracks, 1] := DP;
      IsModified := true;
    end
end;

procedure TSongEditForm.STPanelStartDrag(Sender: TObject;
  var DragObject: TDragObject);
begin
  WhatDrag := wdTrack;
end;

procedure TSongEditForm.STPanelDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  Accept := (Source<>Sender) and
            ((Source<>PListListBox) or
             (PListListBox.ItemIndex<>-1)) and
            ((X>=PosImage.Width) or
             (@TPanel(Source).OnDragOver=@TPanel(Sender).OnDragOver));
end;

procedure TSongEditForm.STPanelDragDrop(Sender, Source: TObject; X, Y: Integer);
var T, DP, DT: Integer;
begin
  if (Source=Sender) or
     ((X<PosImage.Width) and
      (@TPanel(Source).OnDragOver<>@TPanel(Sender).OnDragOver)) then Exit;
  T := TPanel(Sender).Tag;
  if WhatDrag=wdTrack then
    begin
      DT := TPanel(Source).Tag;
      if T<DT then Inc(DT);
      FSong.Sequence.InsertTrack(T);
      FSong.Sequence.CopyTrack(DT, T);
      if DoMove then FSong.Sequence.DeleteTrack(DT);
      IsModified := true;
    end
  else
  if WhatDrag=wdPatSeq then
    begin
      DP := TPanel(Source).Tag;
      DT := TPanel(TPanel(Source).Owner).Tag;
      FSong.Sequence.TrackLength[T] := FSong.Sequence.TrackLength[T]+1;
      FSong.Sequence[T, FSong.Sequence.TrackLength[T]] :=
        FSong.Sequence[DT, DP];
      if DoMove then FSong.Sequence.DeletePattern(DT, DP);
      IsModified := true;
    end
  else
  if WhatDrag=wdPatList then
    begin
      DP := PListListBox.ItemIndex+1;
      if DP=0 then Exit;
      FSong.Sequence.TrackLength[T] := FSong.Sequence.TrackLength[T]+1;
      FSong.Sequence[T, FSong.Sequence.TrackLength[T]] :=
        DP;
      IsModified := true;
    end
end;

procedure TSongEditForm.SPPanelStartDrag(Sender: TObject;
  var DragObject: TDragObject);
begin
  WhatDrag := wdPatSeq;
end;

procedure TSongEditForm.SPPanelDragOver(Sender, Source: TObject; X, Y: Integer;
  State: TDragState; var Accept: Boolean);
begin
  Accept := (Source<>Sender) and
            ((Source<>PListListBox) or
             (PListListBox.ItemIndex<>-1));
end;

procedure TSongEditForm.SPPanelDragDrop(Sender, Source: TObject; X, Y: Integer);
var T, P, DP, DT: Integer;
begin
  if Source=Sender then Exit;
  P := TPanel(Sender).Tag;
  T := TPanel(TPanel(Sender).Owner).Tag;
  if WhatDrag=wdTrack then
    begin
      DT := TPanel(Source).Tag;
      if T<DT then Inc(DT);
      FSong.Sequence.InsertTrack(T);
      FSong.Sequence.CopyTrack(DT, T);
      if DoMove then FSong.Sequence.DeleteTrack(DT);
      IsModified := true;
    end
  else
  if WhatDrag=wdPatSeq then
    begin
      DP := TPanel(Source).Tag;
      DT := TPanel(TPanel(Source).Owner).Tag;
      FSong.Sequence.InsertPattern(T, P, FSong.Sequence[DT, DP]);
      if DoMove then FSong.Sequence.DeletePattern(DT, DP);
      IsModified := true;
    end
  else
  if WhatDrag=wdPatList then
    begin
      DP := PListListBox.ItemIndex+1;
      if DP=0 then Exit;
      FSong.Sequence.InsertPattern(T, P, DP);
      IsModified := true;
    end
end;

procedure TSongEditForm.SeqEndDrag(Sender, Target: TObject; X,
  Y: Integer);
begin
  UpdateSequence;
end;

procedure TSongEditForm.SaveOptions;
var CF: TIniFile; i: Integer;
begin
  CF := TIniFile.Create(ChangeFileExt(Application.ExeName, '.cfg'));
  try
    //Sound card
    CF.WriteInteger('SoundCard', 'WaveDevice', WaveDevice);
    CF.WriteInteger('SoundCard', 'BufferCount', BufferCount);
    CF.WriteInteger('SoundCard', 'BufferSize', BufferSize);
    //Color set
    CF.WriteInteger('ColorSet', 'PEditColorN', ColorSet.PEditColorN);
    CF.WriteInteger('ColorSet', 'PEditColorNT', ColorSet.PEditColorNT);
    CF.WriteInteger('ColorSet', 'PEditColorS', ColorSet.PEditColorS);
    CF.WriteInteger('ColorSet', 'PEditColorST', ColorSet.PEditColorST);
    CF.WriteInteger('ColorSet', 'PEditColorB', ColorSet.PEditColorB);
    CF.WriteInteger('ColorSet', 'PEditColorBT', ColorSet.PEditColorBT);
    CF.WriteInteger('ColorSet', 'PEditColorM', ColorSet.PEditColorM);
    CF.WriteInteger('ColorSet', 'PEditColorMT', ColorSet.PEditColorMT);
    CF.WriteInteger('ColorSet', 'PEditColorL', ColorSet.PEditColorL);
    CF.WriteInteger('ColorSet', 'SeqColorT', ColorSet.SeqColorT);
    CF.WriteInteger('ColorSet', 'SeqColorP', ColorSet.SeqColorP);
    CF.WriteInteger('ColorSet', 'SeqColorPT', ColorSet.SeqColorPT);
    CF.WriteInteger('ColorSet', 'SeqColorI', ColorSet.SeqColorI);
    CF.WriteInteger('ColorSet', 'SeqColorIT', ColorSet.SeqColorIT);
    CF.WriteInteger('ColorSet', 'ClipColor', ColorSet.ClipColor);
    CF.WriteInteger('ColorSet', 'WaveDataColor', ColorSet.WaveDataColor);
    CF.WriteInteger('ColorSet', 'WaveAxisColor', ColorSet.WaveAxisColor);
    CF.WriteInteger('ColorSet', 'WaveColor', ColorSet.WaveColor);
    //Editor misc.
    CF.WriteBool('Editor', 'FollowSong', FollowSong);
    CF.WriteBool('Editor', 'FollowPattern', FollowPattern);
    CF.WriteBool('Editor', 'EnableMeter', EnableMeter);
    CF.WriteBool('Editor', 'PlayWhileEdit', PlayWhileEdit);
    CF.WriteInteger('Editor', 'MeterRate', MeterRate);
    CF.WriteInteger('Editor', 'PPTrack', PPTrack);
    CF.WriteInteger('Editor', 'PPTick', PPTick);
    CF.WriteInteger('Editor', 'TPEmpty', TPEmpty);
    CF.WriteInteger('Editor', 'TPFree', TPFree);
    CF.WriteInteger('Editor', 'BaseOctave', BaseOctave);
    CF.WriteInteger('Editor', 'Spacing', Spacing);
    //Keyboard
    for i := 1 to KeysNum do
      begin
        CF.WriteInteger('Keys', 'Key'+IntToStr(i), KeyTable[i].Ofs);
      end;
    //Environment
    if IsUntitled then FLastFile := '' else FLastFile := FileName;
    CF.WriteString('Environment', 'WaveEditor', WaveEditor);
    CF.WriteString('Environment', 'LastFile', FLastFile);
    //Window
    if WindowState = wsNormal then
      begin
        CF.WriteInteger('Window', 'Left', Left);
        CF.WriteInteger('Window', 'Top', Top);
        CF.WriteInteger('Window', 'Width', Width);
        CF.WriteInteger('Window', 'Height', Height);
      end;
    CF.WriteInteger('Window', 'State', Integer(WindowState));
    CF.WriteInteger('Window', 'Page', EdPageControl.ActivePage.PageIndex);
  finally
    CF.Free;
  end;
end;

procedure TSongEditForm.LoadOptions;
var CF: TIniFile; i: Integer;
begin
  CF := TIniFile.Create(ChangeFileExt(Application.ExeName, '.cfg'));
  try
    //Sound card
    WaveDevice := CF.ReadInteger('SoundCard', 'WaveDevice', WaveDevice);
    BufferCount := CF.ReadInteger('SoundCard', 'BufferCount', BufferCount);
    BufferSize := CF.ReadInteger('SoundCard', 'BufferSize', BufferSize);
    //Color set
    ColorSet.PEditColorN := CF.ReadInteger('ColorSet', 'PEditColorN', ColorSet.PEditColorN);
    ColorSet.PEditColorNT := CF.ReadInteger('ColorSet', 'PEditColorNT', ColorSet.PEditColorNT);
    ColorSet.PEditColorS := CF.ReadInteger('ColorSet', 'PEditColorS', ColorSet.PEditColorS);
    ColorSet.PEditColorST := CF.ReadInteger('ColorSet', 'PEditColorST', ColorSet.PEditColorST);
    ColorSet.PEditColorB := CF.ReadInteger('ColorSet', 'PEditColorB', ColorSet.PEditColorB);
    ColorSet.PEditColorBT := CF.ReadInteger('ColorSet', 'PEditColorBT', ColorSet.PEditColorBT);
    ColorSet.PEditColorM := CF.ReadInteger('ColorSet', 'PEditColorM', ColorSet.PEditColorM);
    ColorSet.PEditColorMT := CF.ReadInteger('ColorSet', 'PEditColorMT', ColorSet.PEditColorMT);
    ColorSet.PEditColorL := CF.ReadInteger('ColorSet', 'PEditColorL', ColorSet.PEditColorL);
    ColorSet.SeqColorT := CF.ReadInteger('ColorSet', 'SeqColorT', ColorSet.SeqColorT);
    ColorSet.SeqColorP := CF.ReadInteger('ColorSet', 'SeqColorP', ColorSet.SeqColorP);
    ColorSet.SeqColorPT := CF.ReadInteger('ColorSet', 'SeqColorPT', ColorSet.SeqColorPT);
    ColorSet.SeqColorI := CF.ReadInteger('ColorSet', 'SeqColorI', ColorSet.SeqColorI);
    ColorSet.SeqColorIT := CF.ReadInteger('ColorSet', 'SeqColorIT', ColorSet.SeqColorIT);
    ColorSet.ClipColor := CF.ReadInteger('ColorSet', 'ClipColor', ColorSet.ClipColor);
    ColorSet.WaveDataColor := CF.ReadInteger('ColorSet', 'WaveDataColor', ColorSet.WaveDataColor);
    ColorSet.WaveAxisColor := CF.ReadInteger('ColorSet', 'WaveAxisColor', ColorSet.WaveAxisColor);
    ColorSet.WaveColor := CF.ReadInteger('ColorSet', 'WaveColor', ColorSet.WaveColor);
    //Editor misc.
    FollowSong := CF.ReadBool('Editor', 'FollowSong', FollowSong);
    FollowPattern := CF.ReadBool('Editor', 'FollowPattern', FollowPattern);
    EnableMeter := CF.ReadBool('Editor', 'EnableMeter', EnableMeter);
    PlayWhileEdit := CF.ReadBool('Editor', 'PlayWhileEdit', PlayWhileEdit);
    MeterRate := CF.ReadInteger('Editor', 'MeterRate', MeterRate);
    PPTrack := CF.ReadInteger('Editor', 'PPTrack', PPTrack);
    PPTick := CF.ReadInteger('Editor', 'PPTick', PPTick);
    TPEmpty := CF.ReadInteger('Editor', 'TPEmpty', TPEmpty);
    TPFree := CF.ReadInteger('Editor', 'TPFree', TPFree);
    BaseOctave := CF.ReadInteger('Editor', 'BaseOctave', BaseOctave);
    Spacing := CF.ReadInteger('Editor', 'Spacing', Spacing);
    //Keyboard
    for i := 1 to KeysNum do
      begin
        KeyTable[i].Ofs := CF.ReadInteger('Keys', 'Key'+IntToStr(i), KeyTable[i].Ofs);
      end;
    //Environment
    WaveEditor := CF.ReadString('Environment', 'WaveEditor', WaveEditor);
    FLastFile := CF.ReadString('Environment', 'LastFile', FLastFile);
    //Window
    Left := CF.ReadInteger('Window', 'Left', Left);
    Top := CF.ReadInteger('Window', 'Top', Top);
    Width := CF.ReadInteger('Window', 'Width', Width);
    Height := CF.ReadInteger('Window', 'Height', Height);
    WindowState :=
      TWindowState(CF.ReadInteger('Window', 'State', Integer(WindowState)));
    EdPageControl.ActivePage :=
      EdPageControl.Pages[
        CF.ReadInteger('Window', 'Page', EdPageControl.ActivePage.PageIndex)];
    FormResize(Self);
  finally
    CF.Free;
  end;
end;

procedure TSongEditForm.OptionsToolButtonClick(Sender: TObject);
begin
  if SetOptions then
    begin
      FPlayer.Free;
      try
        FPlayer := TSongPlayer.Create(FSong, WaveDevice, BufferCount, BufferSize,
          Handle, LoopPattern);
      except
        FPlayer := nil;
        Application.ShowException(ExceptObject as Exception);
      end;
      UpdateGUI;
    end;
end;

procedure TSongEditForm.SeqRBImageDblClick(Sender: TObject);
begin
  if (FSong.Sequence.Tracks>1) or
     (FSong.Sequence.TrackLength[1]>0) then
    if MessageDlg('Clear sequence ?', mtConfirmation,
                  [mbYes, mbNo], 0)=mrYes then
    begin
      FSong.Sequence.Tracks := 1;
      FSong.Sequence.TrackLength[1] := 0;
      IsModified := true;
      UpdateSequence;
    end;
end;

procedure TSongEditForm.WMStart(var Message: TMessage);
begin
  //PlayToolButton.Enabled := false;
  StopToolButton.Enabled := true;
  ResetClip;
end;

procedure TSongEditForm.WMStop(var Message: TMessage);
begin
  //PlayToolButton.Enabled := true;
  StopToolButton.Enabled := false;
  PostMessage(Handle, WM_SETSPOS, 1, 0);
  MLVPanel.Width := 0;
  MRVPanel.Width := 0;
//  ClipPanel.Color := MeterPanel.Color;
end;

procedure TSongEditForm.WMSetPPos(var Message: TMessage);
begin
  if EdPageControl.ActivePage<>PattTabSheet then Exit;
  if Message.WParam=PListBox.ItemIndex+1 then
    PGrid.Row := Message.LParam;
end;

procedure TSongEditForm.WMSetSPos(var Message: TMessage);
var i: Integer;
begin
  if (EdPageControl.ActivePage<>SeqTabSheet) and (Message.WParam<>1) then Exit;
  PosImage.Left := (Message.WParam-1)*PPTick-PosImage.Width div 2+PosImage.Width;
  i := PosImage.Left+PosImage.Width-SeqScrollBox.ClientWidth;
  if SeqScrollBox.HorzScrollBar.Position<i then
    SeqScrollBox.HorzScrollBar.Position := i;
  if SeqScrollBox.HorzScrollBar.Position>PosImage.Left-PosImage.Width then
    SeqScrollBox.HorzScrollBar.Position := PosImage.Left-PosImage.Width;
end;

procedure TSongEditForm.WMSetSample(var Message: TMessage);
begin
  MLVPanel.Width := Round((MLPanel.ClientWidth-2)*Abs(Message.WParam)/32768);
  MRVPanel.Width := Round((MRPanel.ClientWidth-2)*Abs(Message.LParam)/32768);
end;

procedure TSongEditForm.WMClip(var Message: TMessage);
var CC: Double;
begin
  CC := Message.WParam/Message.LParam;
  if CC<FCoeff then FCoeff := CC;
  ClipPanel.Color := ColorSet.ClipColor;
end;

procedure TSongEditForm.WMExcept(var Message: TMessage);
begin
  if Assigned(FPlayer) then
    Raise Exception.Create(FPlayer.ExceptMessage);
end;

procedure TSongEditForm.WaveDataPaint(Sender: TObject);
var CX, CY: Double; i, X, Y, YY, SH: Integer; S: SmallInt;
begin
  WaveData.Canvas.Brush.Color := ColorSet.WaveDataColor;
  WaveData.Canvas.FillRect(WaveData.ClientRect);
  if (SListBox.ItemIndex<0) or
     (SListBox.ItemIndex>=FSong.Samples.Count) then Exit;
  with FSong.Samples[SListBox.ItemIndex+1], WaveData.Canvas, WaveData do
    begin
      SH := ClientHeight div Channels;
      CX := Length/ClientWidth;
      if CX<1 then CX := 1;
      CY := (SH-10)/2/32768;
      for i := 1 to Channels do
        begin
          Pen.Color := ColorSet.WaveAxisColor;
          YY := SH*(i-1)+SH div 2;
          MoveTo(0, YY);
          LineTo(ClientWidth-1, YY);
          Pen.Color := ColorSet.WaveColor;
          if Length>0 then
            for X := 1 to ClientWidth do
              if X*CX<=Length then
                begin
                  if i=1 then
                    S := XGetSampleL(Round(X*CX))
                  else
                    S := XGetSampleR(Round(X*CX));
                  Y := YY-Round(CY*S);
                  if X=1 then MoveTo(X-1, Y) else LineTo(X-1, Y);
                end;
        end;
      //Loop
      if (LoopType<>lNone) and
         (LoopStart>0) and (LoopStart<=Length) and
         (LoopEnd>0) and (LoopEnd<=Length) and
         (LoopEnd>=LoopStart) then
        begin
          Pen.Color := ColorSet.WaveAxisColor;
          MoveTo(Round(LoopStart/CX), 0);
          LineTo(Round(LoopStart/CX), ClientHeight-1);
          MoveTo(Round(LoopEnd/CX), 0);
          LineTo(Round(LoopEnd/CX), ClientHeight-1);
        end;
    end;
end;

procedure TSongEditForm.ResetClip;
begin
  FCoeff := 1;
  ClipPanel.Color := MeterPanel.Color;
end;

procedure TSongEditForm.ClipToolButtonClick(Sender: TObject);
begin
  if FCoeff=1 then Exit;
  FSong.SongVolume := Trunc(FSong.SongVolume*FCoeff);
  ResetClip;
  IsModified := true;
  SongVolTrackBar.Position := FSong.SongVolume;
  SVNLabel.Caption := IntToStr(FSong.SongVolume);
end;

procedure TSongEditForm.SongVolTrackBarChange(Sender: TObject);
begin
  FSong.SongVolume := SongVolTrackBar.Position;
  IsModified := true;
  SVNLabel.Caption := IntToStr(FSong.SongVolume);
end;

procedure TSongEditForm.PGridClick(Sender: TObject);
begin
  FKeyMode := kmNone;
end;

procedure TSongEditForm.KeyTimerTimer(Sender: TObject);
begin
  FKeyMode := kmNone;
end;

procedure TSongEditForm.NormalizeSButtonClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  FSong.Samples[SListBox.ItemIndex+1].Normalize;
  IsModified := true;
  UpdateSample;
end;

procedure TSongEditForm.EditSButtonClick(Sender: TObject);
var TFN: String; FA: Integer;
begin
  if WaveEditor='' then Exit;
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  TFN := GetTempName;
  FSong.Samples[SListBox.ItemIndex+1].SaveToFile(TFN);
  try
  FA := FileAge(TFN);
  Exec(WaveEditor, TFN);
  if FileAge(TFN)<>FA then
    begin
      FSong.Samples[SListBox.ItemIndex+1].LoadFromFile(TFN);
      IsModified := true;
      UpdateSample;
    end;
  finally
    if not DeleteFile(TFN) then RaiseLastWin32Error;
  end;
end;

procedure TSongEditForm.TransposePButtonClick(Sender: TObject);
var T: Integer; i, j: Integer; C: TCommand; NT: Boolean;
begin
  if (PListBox.Items.Count=0) or (PListBox.ItemIndex=-1) then Exit;
  T := GetTranspose;
  if T=0 then Exit;
  NT := false;
  for i := PGrid.Selection.Left to PGrid.Selection.Right do
    for j := PGrid.Selection.Top to PGrid.Selection.Bottom do
      begin
        C := FSong.Patterns[PListBox.ItemIndex+1][i, j];
        if (C.Note<=MaxNote) then
          begin
            NT := true;
            if (C.Note+T<0) or (C.Note+T>MaxNote) then Exit;
          end;
      end;
  if not NT then Exit;
  for i := PGrid.Selection.Left to PGrid.Selection.Right do
    for j := PGrid.Selection.Top to PGrid.Selection.Bottom do
      begin
        C := FSong.Patterns[PListBox.ItemIndex+1][i, j];
        if (C.Note<=MaxNote) then C.Note := C.Note+T;
        FSong.Patterns[PListBox.ItemIndex+1][i, j] := C;
      end;
  IsModified := true;
  UpdatePattern;
end;

procedure TSongEditForm.PlayPatternItemClick(Sender: TObject);
begin
  if EdPageControl.ActivePage=PattTabSheet then PlayPButton.Click;
end;

procedure TSongEditForm.ListBoxKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key=VK_RETURN) and (Shift=[]) then
    begin
      if Assigned((Sender as TListBox).OnDblClick) then
        (Sender as TListBox).OnDblClick(Sender);
      Key := 0;
    end;
end;

procedure TSongEditForm.PListListBoxDblClick(Sender: TObject);
begin
  if (PListListBox.Items.Count=0) or (PListListBox.ItemIndex=-1) then Exit;
  if Assigned(FPlayer) then
    FPlayer.StartPlayPattern(PListListBox.ItemIndex+1);
end;

procedure TSongEditForm.ClearSButtonClick(Sender: TObject);
begin
  if (SListBox.Items.Count=0) or (SListBox.ItemIndex=-1) then Exit;
  with FSong.Samples[SListBox.ItemIndex+1] do
    begin
      Length := 0;
      LoopType := lNone;
      LoopStart := 0;
      LoopEnd := 0;
    end;
  IsModified := true;
  UpdateSample;
end;

procedure TSongEditForm.FormResize(Sender: TObject);
begin
  if not Destroying then UpdateSequence;
end;

initialization
  CF := RegisterClipboardFormat(ClipboardFormat);
end.
