unit Wave;

interface

uses MMSystem;

type
  TS8Mono = Byte;
  TS8Stereo = packed record L, R: Byte end;
  TS16Mono = SmallInt;
  TS16Stereo = packed record L, R: SmallInt end;
  TS8MonoArray = packed Array[1..MaxInt div SizeOf(TS8Mono)] of TS8Mono;
  TS8StereoArray = packed Array[1..MaxInt div SizeOf(TS8Stereo)] of TS8Stereo;
  TS16MonoArray = packed Array[1..MaxInt div SizeOf(TS16Mono)] of TS16Mono;
  TS16StereoArray = packed Array[1..MaxInt div SizeOf(TS16Stereo)] of TS16Stereo;
  PS8MonoArray = ^TS8MonoArray;
  PS8StereoArray = ^TS8StereoArray;
  PS16MonoArray = ^TS16MonoArray;
  PS16StereoArray = ^TS16StereoArray;

  TWave = class
   private
    FBits: Byte;
    FChannels: Byte;
    FRate: Integer;
    FDataLength: Integer; //Length in bytes
    FData: Pointer;
    procedure SetBits(ABits: Byte);
    procedure SetChannels(AChannels: Byte);
    procedure SetRate(ARate: Integer);
    function GetLength: Integer;
    procedure SetLength(ALength: Integer);
    procedure SetDataLength(ADataLength: Integer);
    procedure CheckPos(Pos: Integer);
    function GetSampleL(Pos: Integer): SmallInt;
    function GetSampleR(Pos: Integer): SmallInt;
    procedure SetSampleL(Pos: Integer; Sample: SmallInt);
    procedure SetSampleR(Pos: Integer; Sample: SmallInt);
   protected
    procedure ProcessChunk(Sign: String; Size: Integer; var Data); virtual;
   public
    constructor Create;
    destructor Destroy; override;
    function BytesPerSample: Byte;
    procedure LoadFromFile(FileName: String);
    procedure SaveToFile(FileName: String);
    procedure CopyFrom(Source: TWave);
    procedure ApplyVolume(AVolume: Byte);
    procedure ApplyPan(APan: ShortInt);
    procedure MixWith(Source: TWave);
    procedure XMixWith(Source: TWave; SDivider: Integer);
    procedure GetSliceFrom(Source: TWave; SFrom, STo: Integer);
    procedure Delete(SFrom, STo: Integer);
    procedure AddFrom(Source: TWave);
    procedure Resample(ARate: Integer);
    procedure Normalize;

    function XGetSampleL(Pos: Integer): SmallInt;
    function XGetSampleR(Pos: Integer): SmallInt;

    property Bits: Byte read FBits write SetBits;
    property Channels: Byte read FChannels write SetChannels;
    property Rate: Integer read FRate write SetRate;
    property Length: Integer read GetLength write SetLength;
    property DataLength: Integer read FDataLength;
    property SampleL[Pos: Integer]: SmallInt read GetSampleL write SetSampleL;
    property SampleR[Pos: Integer]: SmallInt read GetSampleR write SetSampleR;
    property RawData: Pointer read FData;
  end;

implementation
uses Classes, SysUtils;

function _8to16(b: Byte): SmallInt;
begin
  result := (b-128)*256;
end;

function _16to8(s: SmallInt): Byte;
begin
  result := s div 256+128;
end;

procedure TWave.SetBits(ABits: Byte);
var SecData: Pointer; i, OL: Integer; SBits: Byte;
begin
  if ABits=FBits then Exit;
  if not (ABits in [8, 16]) then
    Raise Exception.Create('Invalid bits: '+IntToStr(ABits));
  OL := Length;
  SBits := FBits;
  FBits := ABits;
  if FDataLength<>0 then
    begin
      GetMem(SecData, FDataLength);
      try
        Move(FData^, SecData^, FDataLength);
        SetLength(OL);
        for i := 1 to Length do
          if (SBits=8) and (ABits=16) then
            begin
              if FChannels=1 then
                PS16MonoArray(FData)^[i] := _8to16(PS8MonoArray(SecData)^[i])
              else
              if FChannels=2 then
                begin
                  PS16StereoArray(FData)^[i].L := _8to16(PS8StereoArray(SecData)^[i].L);
                  PS16StereoArray(FData)^[i].R := _8to16(PS8StereoArray(SecData)^[i].R);
                end;
            end
          else
          if (SBits=16) and (ABits=8) then
            begin
              if FChannels=1 then
                PS8MonoArray(FData)^[i] := _16to8(PS16MonoArray(SecData)^[i])
              else
              if FChannels=2 then
                begin
                  PS8StereoArray(FData)^[i].L := _16to8(PS16StereoArray(SecData)^[i].L);
                  PS8StereoArray(FData)^[i].R := _16to8(PS16StereoArray(SecData)^[i].R);
                end;
            end
      finally
        FreeMem(SecData);
      end;
    end;
end;

procedure TWave.SetChannels(AChannels: Byte);
var SecData: Pointer; i, OL: Integer; SChannels: Byte;
begin
  if AChannels=FChannels then Exit;
  if not (AChannels in [1, 2]) then
    Raise Exception.Create('Invalid channels: '+IntToStr(AChannels));
  OL := Length;
  SChannels := FChannels;
  FChannels := AChannels;
  if FDataLength<>0 then
    begin
      GetMem(SecData, FDataLength);
      try
        Move(FData^, SecData^, FDataLength);
        SetLength(OL);
        for i := 1 to Length do
          if (SChannels=1) and (AChannels=2) then
            begin
              if FBits=8 then
                begin
                  PS8StereoArray(FData)^[i].L := PS8MonoArray(SecData)^[i];
                  PS8StereoArray(FData)^[i].R := PS8MonoArray(SecData)^[i];
                end
              else
              if FBits=16 then
                begin
                  PS16StereoArray(FData)^[i].L := PS16MonoArray(SecData)^[i];
                  PS16StereoArray(FData)^[i].R := PS16MonoArray(SecData)^[i];
                end;
            end
          else
          if (SChannels=2) and (AChannels=1) then
            begin
              if FBits=8 then
                begin
                  PS8MonoArray(FData)^[i] :=
                    PS8StereoArray(SecData)^[i].L div 2+
                    PS8StereoArray(SecData)^[i].R div 2;
                end
              else
              if FBits=16 then
                begin
                  PS16MonoArray(FData)^[i] :=
                    PS16StereoArray(SecData)^[i].L div 2+
                    PS16StereoArray(SecData)^[i].R div 2;
                end;
            end
      finally
        FreeMem(SecData);
      end;
    end;
end;

procedure TWave.SetRate(ARate: Integer);
begin
  if ARate=FRate then Exit;
  if ARate=0 then
    Raise Exception.Create('Samplerate cannot be zero');
  FRate := ARate;
end;

function TWave.GetLength: Integer;
begin
  result := FDataLength div BytesPerSample;
end;

procedure TWave.SetLength(ALength: Integer);
begin
  SetDataLength(ALength*BytesPerSample);
end;

procedure TWave.SetDataLength(ADataLength: Integer);
begin
  if ADataLength=FDataLength then Exit;
  while ADataLength mod BytesPerSample<>0 do
    Inc(ADataLength);
  ReAllocMem(FData, ADataLength);
  if ADataLength>FDataLength then
    FillChar(Pointer(Integer(FData)+FDataLength)^, ADataLength-FDataLength, 0);
  FDataLength := ADataLength;
end;

procedure TWave.CheckPos(Pos: Integer);
begin
  if (Pos<1) and (Pos>Length) then
    Raise Exception.Create('Position out of range');
end;

function TWave.GetSampleL(Pos: Integer): SmallInt;
begin
  result := 0;
  CheckPos(Pos);
  if (FBits=8) and (FChannels=1) then
    begin
      result := PS8MonoArray(FData)^[Pos];
    end
  else
  if (FBits=8) and (FChannels=2) then
    begin
      result := PS8StereoArray(FData)^[Pos].L;
    end
  else
  if (FBits=16) and (FChannels=1) then
    begin
      result := PS16MonoArray(FData)^[Pos];
    end
  else
  if (FBits=16) and (FChannels=2) then
    begin
      result := PS16StereoArray(FData)^[Pos].L;
    end
end;

function TWave.GetSampleR(Pos: Integer): SmallInt;
begin
  result := 0;
  CheckPos(Pos);
  if (FBits=8) and (FChannels=1) then
    begin
      result := PS8MonoArray(FData)^[Pos];
    end
  else
  if (FBits=8) and (FChannels=2) then
    begin
      result := PS8StereoArray(FData)^[Pos].R;
    end
  else
  if (FBits=16) and (FChannels=1) then
    begin
      result := PS16MonoArray(FData)^[Pos];
    end
  else
  if (FBits=16) and (FChannels=2) then
    begin
      result := PS16StereoArray(FData)^[Pos].R;
    end
end;

procedure TWave.SetSampleL(Pos: Integer; Sample: SmallInt);
begin
  CheckPos(Pos);
  if (FBits=8) and (FChannels=1) then
    begin
      PS8MonoArray(FData)^[Pos] := Sample;
    end
  else
  if (FBits=8) and (FChannels=2) then
    begin
      PS8StereoArray(FData)^[Pos].L := Sample;
    end
  else
  if (FBits=16) and (FChannels=1) then
    begin
      PS16MonoArray(FData)^[Pos] := Sample;
    end
  else
  if (FBits=16) and (FChannels=2) then
    begin
      PS16StereoArray(FData)^[Pos].L := Sample;
    end
end;

procedure TWave.SetSampleR(Pos: Integer; Sample: SmallInt);
begin
  CheckPos(Pos);
  if (FBits=8) and (FChannels=1) then
    begin
      PS8MonoArray(FData)^[Pos] := Sample;
    end
  else
  if (FBits=8) and (FChannels=2) then
    begin
      PS8StereoArray(FData)^[Pos].R := Sample;
    end
  else
  if (FBits=16) and (FChannels=1) then
    begin
      PS16MonoArray(FData)^[Pos] := Sample;
    end
  else
  if (FBits=16) and (FChannels=2) then
    begin
      PS16StereoArray(FData)^[Pos].R := Sample;
    end
end;

constructor TWave.Create;
begin
  inherited Create;
  FBits := 16;
  FChannels := 2;
  FRate := 44100;
  FDataLength := 0;
  FData := nil;
end;

destructor TWave.Destroy;
begin
  SetDataLength(0);
  inherited Destroy;
end;

function TWave.BytesPerSample: Byte;
begin
  result := FBits div 8*FChannels;
end;

type
  WaveFmt = packed record
    wFormatTag: Word;           // Format category
    wChannels: Word;            // Number of channels
    dwSamplesPerSec: LongInt;   // Sampling rate
    dwAvgBytesPerSec: LongInt;  // For buffer estimation
    wBlockAlign: Word;          // Data block size
    case LongInt of
      WAVE_FORMAT_PCM:
        (
          wBitsPerSample: Word; // Sample size
        )
      // Other wave formats are not supported now
  end;

procedure TWave.ProcessChunk(Sign: String; Size: Integer; var Data);
begin
end;

procedure TWave.LoadFromFile(FileName: String);
var F: TFileStream; Sign: String[4]; L: LongInt; Fmt: WaveFmt; P: Pointer;
begin
  F := TFileStream.Create(FileName, fmOpenRead or fmShareDenyWrite);
  try
    System.SetLength(Sign, 4);
    F.Read(Sign[1], 4);
    if Sign<>'RIFF' then
      Raise Exception.Create('RIFF signature not found');
    F.Read(L, 4);
    F.Read(Sign[1], 4);
    if Sign<>'WAVE' then
      Raise Exception.Create('WAVE signature not found');
    SetDataLength(0);
    repeat
      F.Read(Sign[1], 4);
      F.Read(L, 4);
      if Sign='fmt ' then
        begin
          if L>SizeOf(Fmt) then
            begin
              F.Read(Fmt, SizeOf(Fmt));
              F.Position := F.Position+(L-SizeOf(Fmt));
            end
          else F.Read(Fmt, L);
          if Fmt.wFormatTag<>WAVE_FORMAT_PCM then
            Raise Exception.Create('Unsupported wave file format');
          if not (Fmt.wChannels in [1, 2]) then
            Raise Exception.Create('Unsupported number of channels: '+
                                   IntToStr(Fmt.wChannels));
          FChannels := Fmt.wChannels;
          FRate := Fmt.dwSamplesPerSec;
          if not (Fmt.wBitsPerSample in [8, 16]) then
            Raise Exception.Create('Unsupported bits per sample: '+
                                   IntToStr(Fmt.wBitsPerSample));
          FBits := Fmt.wBitsPerSample;
        end
      else
      if Sign='data' then
        begin
          SetDataLength(L);
          F.Read(FData^, L);
        end
      else
        begin
          //F.Position := F.Position+L;
          GetMem(P, L);
          try
            F.Read(P^, L);
            ProcessChunk(Sign, L, P^);
          finally
            FreeMem(P);
          end;
        end;
    until (L=0) or (F.Position>=F.Size);
    while FDataLength mod BytesPerSample<>0 do
      SetDataLength(FDataLength+1);
  finally
    F.Free;
  end;
end;

procedure TWave.SaveToFile(FileName: String);
var F: TFileStream; Sign: String[4]; L: LongInt; Fmt: WaveFmt;
begin
{  if FDataLength=0 then
    Raise Exception.Create('Empty wave data');}
  while FDataLength mod BytesPerSample<>0 do
    SetDataLength(FDataLength+1);
  F := TFileStream.Create(FileName, fmCreate or fmShareDenyWrite);
  try
    Sign := 'RIFF';
    F.Write(Sign[1], 4);
    L := 12+SizeOf(Fmt)+8+FDataLength;
    F.Write(L, 4);
    Sign := 'WAVE';
    F.Write(Sign[1], 4);
    Sign := 'fmt ';
    F.Write(Sign[1], 4);
    L := SizeOf(Fmt);
    F.Write(L, 4);
    Fmt.wFormatTag := WAVE_FORMAT_PCM;
    Fmt.wChannels := FChannels;
    Fmt.dwSamplesPerSec := FRate;
    Fmt.dwAvgBytesPerSec := FRate*BytesPerSample;
    Fmt.wBlockAlign := BytesPerSample;
    Fmt.wBitsPerSample := FBits;
    F.Write(Fmt, SizeOf(Fmt));
    Sign := 'data';
    F.Write(Sign[1], 4);
    L := FDataLength;
    F.Write(L, 4);
    F.Write(FData^, L);
  finally
    F.Free;
  end;
end;

procedure TWave.CopyFrom(Source: TWave);
begin
  SetDataLength(0);
  FBits := Source.FBits;
  FChannels := Source.FChannels;
  FRate := Source.FRate;
  SetDataLength(Source.FDataLength);
  Move(Source.FData^, FData^, FDataLength);
end;

procedure TWave.ApplyVolume(AVolume: Byte);
var i: Integer;
begin
  if AVolume=100 then Exit; //100%
  if AVolume=255 then Exit; //Special volume
  if AVolume>200 then
    Raise Exception.Create('Volume must be from 0 to 200');
  for i := 1 to Length do
    if (FBits=8) and (FChannels=1) then
      begin
        PS8MonoArray(FData)^[i] := (PS8MonoArray(FData)^[i]-128)*AVolume div 100+128;
      end
    else
    if (FBits=8) and (FChannels=2) then
      begin
        PS8StereoArray(FData)^[i].L := (PS8StereoArray(FData)^[i].L-128)*AVolume div 100+128;
        PS8StereoArray(FData)^[i].R := (PS8StereoArray(FData)^[i].R-128)*AVolume div 100+128;
      end
    else
    if (FBits=16) and (FChannels=1) then
      begin
        PS16MonoArray(FData)^[i] := PS16MonoArray(FData)^[i]*AVolume div 100;
      end
    else
    if (FBits=16) and (FChannels=2) then
      begin
        PS16StereoArray(FData)^[i].L := PS16StereoArray(FData)^[i].L*AVolume div 100;
        PS16StereoArray(FData)^[i].R := PS16StereoArray(FData)^[i].R*AVolume div 100;
      end
end;

procedure TWave.ApplyPan(APan: ShortInt);
var i: Integer;
begin
  if FChannels<>2 then
    Raise Exception.Create('To apply panning, wave must be stereo');
  if APan=0 then Exit; //Center
  if APan=127 then Exit; //Special pan
  if (APan<-100) or (APan>100) then
    Raise Exception.Create('Panning mast be from -100 to 100');
  for i := 1 to Length do
    if (FBits=8) and (FChannels=2) then
      begin
        if (APan>0) then
          PS8StereoArray(FData)^[i].L :=
            (PS8StereoArray(FData)^[i].L-128)*(100-Abs(APan)) div 100+128
        else
        if (APan<0) then
          PS8StereoArray(FData)^[i].R :=
            (PS8StereoArray(FData)^[i].R-128)*(100-Abs(APan)) div 100+128
      end
    else
    if (FBits=16) and (FChannels=2) then
      begin
        if (APan>0) then
          PS16StereoArray(FData)^[i].L := PS16StereoArray(FData)^[i].L*(100-Abs(APan)) div 100
        else
        if (APan<0) then
          PS16StereoArray(FData)^[i].R := PS16StereoArray(FData)^[i].R*(100-Abs(APan)) div 100
      end
end;

procedure TWave.MixWith(Source: TWave);
var i, e: Integer;

function Mix(i1, i2: Integer): Integer;
begin
  result := (i1+i2) div 2;
end;

begin
  if (FBits<>Source.FBits) or (FChannels<>Source.FChannels) then
    Raise Exception.Create('Mixed waves must have same bits and channels');
  if (Bits<>16) then
    Raise Exception.Create('16 bit wave expected');
  e := Length;
  if Source.Length<e then e := Source.Length;
  for i := 1 to e do
    if (FBits=8) and (FChannels=1) then
      begin
        PS8MonoArray(FData)^[i] := Mix(
          PS8MonoArray(FData)^[i],
          PS8MonoArray(Source.FData)^[i]);
      end
    else
    if (FBits=8) and (FChannels=2) then
      begin
        PS8StereoArray(FData)^[i].L := Mix(
          PS8StereoArray(FData)^[i].L,
          PS8StereoArray(Source.FData)^[i].L);

        PS8StereoArray(FData)^[i].R := Mix(
          PS8StereoArray(FData)^[i].R,
          PS8StereoArray(Source.FData)^[i].R);
      end
    else
    if (FBits=16) and (FChannels=1) then
      begin
        PS16MonoArray(FData)^[i] := Mix(
          PS16MonoArray(FData)^[i],
          PS16MonoArray(Source.FData)^[i]);
      end
    else
    if (FBits=16) and (FChannels=2) then
      begin
        PS16StereoArray(FData)^[i].L := Mix(
          PS16StereoArray(FData)^[i].L,
          PS16StereoArray(Source.FData)^[i].L);

        PS16StereoArray(FData)^[i].R := Mix(
          PS16StereoArray(FData)^[i].R,
          PS16StereoArray(Source.FData)^[i].R);
      end
end;

procedure TWave.XMixWith(Source: TWave; SDivider: Integer);
var i, e: Integer;

function Mix(i1, i2, i2div: Integer): Integer;
begin
  result := i1+i2 div i2div;
end;

begin
  if (FBits<>Source.FBits) or (FChannels<>Source.FChannels) then
    Raise Exception.Create('Mixed waves must have same bits and channels');
  if (Bits<>16) then
    Raise Exception.Create('16 bit wave expected');
  if SDivider=0 then
    Raise Exception.Create('Zero divider');
  e := Length;
  if Source.Length<e then e := Source.Length;
  for i := 1 to e do
    if (FBits=8) and (FChannels=1) then
      begin
        PS8MonoArray(FData)^[i] := Mix(
          PS8MonoArray(FData)^[i],
          PS8MonoArray(Source.FData)^[i], SDivider);
      end
    else
    if (FBits=8) and (FChannels=2) then
      begin
        PS8StereoArray(FData)^[i].L := Mix(
          PS8StereoArray(FData)^[i].L,
          PS8StereoArray(Source.FData)^[i].L, SDivider);

        PS8StereoArray(FData)^[i].R := Mix(
          PS8StereoArray(FData)^[i].R,
          PS8StereoArray(Source.FData)^[i].R, SDivider);
      end
    else
    if (FBits=16) and (FChannels=1) then
      begin
        PS16MonoArray(FData)^[i] := Mix(
          PS16MonoArray(FData)^[i],
          PS16MonoArray(Source.FData)^[i], SDivider);
      end
    else
    if (FBits=16) and (FChannels=2) then
      begin
        PS16StereoArray(FData)^[i].L := Mix(
          PS16StereoArray(FData)^[i].L,
          PS16StereoArray(Source.FData)^[i].L, SDivider);

        PS16StereoArray(FData)^[i].R := Mix(
          PS16StereoArray(FData)^[i].R,
          PS16StereoArray(Source.FData)^[i].R, SDivider);
      end
end;

procedure TWave.GetSliceFrom(Source: TWave; SFrom, STo: Integer);
begin
  if (SFrom<1) or (STo<1) or (SFrom>STo) then
    Raise Exception.Create('Invalid slice interval');
  SetDataLength(0);
  FBits := Source.FBits;
  FChannels := Source.FChannels;
  FRate := Source.FRate;
  if (SFrom>Source.Length) and (STo>Source.Length) then Exit;
  if STo>Source.Length then STo := Source.Length;
  SetLength(STo-SFrom+1);
  Move(Pointer(Integer(Source.FData)+(SFrom-1)*BytesPerSample)^, FData^, FDataLength);
end;

procedure TWave.Delete(SFrom, STo: Integer);
begin
  if (SFrom<1) or (STo<1) or (SFrom>STo) then
    Raise Exception.Create('Invalid slice interval');
  if (SFrom>Length) and (STo>Length) then Exit;
  if STo>Length then STo := Length;
  Move(Pointer(Integer(FData)+(STo)*BytesPerSample)^,
       Pointer(Integer(FData)+(SFrom-1)*BytesPerSample)^,
       (Length-STo)*BytesPerSample);
  SetLength(Length-(STo-SFrom+1));
end;

procedure TWave.AddFrom(Source: TWave);
var SL: Integer;
begin
  if (FBits<>Source.FBits) or (FChannels<>Source.FChannels) then
    Raise Exception.Create('Added waves must have same bits and channels');
  if Source.Length=0 then Exit;
  SL := FDataLength;
  SetLength(Length+Source.Length);
  Move(Source.FData^, Pointer(Integer(FData)+SL)^, Source.FDataLength);
end;

procedure TWave.Resample(ARate: Integer);
var SecData: Pointer; i, OL: Integer; SRate: Integer; Coeff: Double;
begin
  if ARate=FRate then Exit;
  SRate := FRate;
  SetRate(ARate);
  OL := Length;
  if FDataLength<>0 then
    begin
      GetMem(SecData, FDataLength);
      try
        Move(FData^, SecData^, FDataLength);
        //Less rate - more length
        Coeff := ARate/SRate;
        SetLength(Round(OL*Coeff));

        for i := 1 to Length do
          if (FBits=8) and (FChannels=1) then
            PS8MonoArray(FData)^[i] := PS8MonoArray(SecData)^[Round((i-1)/Coeff+1)]
          else
          if (FBits=8) and (FChannels=2) then
            PS8StereoArray(FData)^[i] := PS8StereoArray(SecData)^[Round((i-1)/Coeff+1)]
          else
          if (FBits=16) and (FChannels=1) then
            PS16MonoArray(FData)^[i] := PS16MonoArray(SecData)^[Round((i-1)/Coeff+1)]
          else
          if (FBits=16) and (FChannels=2) then
            PS16StereoArray(FData)^[i] := PS16StereoArray(SecData)^[Round((i-1)/Coeff+1)];
      finally
        FreeMem(SecData);
      end;
    end;
end;

procedure TWave.Normalize;
var Max, i: Integer; V: Double;
begin
  Max := 0;
  if FBits=16 then
    begin
      for i := 1 to Length do
        begin
          if Abs(GetSampleL(i))>Max then Max := Abs(GetSampleL(i));
          if Abs(GetSampleR(i))>Max then Max := Abs(GetSampleR(i));
        end;
      if (Max>10) and (Max<32767) then
        begin
          V := 32767/Max;
          for i := 1 to Length do
            begin
              SetSampleL(i, Round(GetSampleL(i)*V));
              if FChannels=2 then
                SetSampleR(i, Round(GetSampleR(i)*V));
            end;
        end;
    end
  else
  if FBits=8 then
    begin
      for i := 1 to Length do
        begin
          if Abs(GetSampleL(i)-128)>Max then Max := Abs(GetSampleL(i)-128);
          if Abs(GetSampleR(i)-128)>Max then Max := Abs(GetSampleR(i)-128);
        end;
      if (Max>5) and (Max<127) then
        begin
          V := 127/Max;
          for i := 1 to Length do
            begin
              SetSampleL(i, Round((GetSampleL(i)-128)*V)+128);
              if FChannels=2 then
                SetSampleR(i, Round((GetSampleR(i)-128)*V)+128);
            end;
        end;
    end;
end;

function TWave.XGetSampleL(Pos: Integer): SmallInt;
begin
  result := 0;
  CheckPos(Pos);
  if (FBits=8) and (FChannels=1) then
    begin
      result := _8to16(PS8MonoArray(FData)^[Pos]);
    end
  else
  if (FBits=8) and (FChannels=2) then
    begin
      result := _8to16(PS8StereoArray(FData)^[Pos].L);
    end
  else
  if (FBits=16) and (FChannels=1) then
    begin
      result := PS16MonoArray(FData)^[Pos];
    end
  else
  if (FBits=16) and (FChannels=2) then
    begin
      result := PS16StereoArray(FData)^[Pos].L;
    end
end;

function TWave.XGetSampleR(Pos: Integer): SmallInt;
begin
  result := 0;
  CheckPos(Pos);
  if (FBits=8) and (FChannels=1) then
    begin
      result := _8to16(PS8MonoArray(FData)^[Pos]);
    end
  else
  if (FBits=8) and (FChannels=2) then
    begin
      result := _8to16(PS8StereoArray(FData)^[Pos].R);
    end
  else
  if (FBits=16) and (FChannels=1) then
    begin
      result := PS16MonoArray(FData)^[Pos];
    end
  else
  if (FBits=16) and (FChannels=2) then
    begin
      result := PS16StereoArray(FData)^[Pos].R;
    end
end;

end.