unit Player;

interface

uses Song, WM, AE, Windows, Messages, Classes;

const
  WM_START = WM_USER+1;
  WM_STOP = WM_USER+2;
  WM_SETPPOS = WM_USER+3;
  WM_SETSPOS = WM_USER+4;
  WM_SETSAMPLE = WM_USER+5;
  WM_CLIP = WM_USER+6;
  WM_EXCEPT = WM_USER+7;

type
  //Sample wrapper for resampling, apply volume and pan, etc.
  TSampleWrapper = class
   private
    FFSong: TSong;
    FSample: Integer;
    FVolume: Byte;
    FPan: ShortInt;
    FPosition: Integer;
    FRealPos: Integer;
    FTrack: Integer;
    FChannel: Integer;
    FNote: Byte;
    FCoef: Double;
    FRate: Integer;
    FLoopBack: Boolean;
    FEnded: Boolean;
    procedure P2R;
    procedure R2P;
    //According to Position etc.
    function InRange: Boolean;
    procedure GetSamples(var SL, SR: SmallInt);
    procedure NextStep;
  end;

  TPlayingSamples = class(TList)
   private
    function GetSample(i: Integer): TSampleWrapper;
   public
    destructor Destroy; override;
    procedure Delete(i: Integer);
    procedure Clear;
    property Sample[i: Integer]: TSampleWrapper read GetSample; default;
  end;

  TPlayMode = (pmNone, pmSample, pmPattern, pmSong);

  TSongPlayer = class
   private
    //These are from main program
    FSong: TSong;
    FHandle: HWND;
    //These are created and destroyed automatically
    FMixer: TWaveMixer;
    FAudioEngine: TAudioEngine;
    FPlayingSamples: TPlayingSamples;
    //Variables :)
    FPLayMode: TPlayMode;
    FPattern: Integer;
    FPos: Integer;
    FSPos: Integer;
    FLoopPattern: Boolean;
    FWaveDevice: Integer;
    FPlaying: Boolean;
    FTickCount: Integer;
    FTickCountM: Integer;
    FInRender: Boolean;

    //Handlers
    procedure M_NextStep;
    function M_GetSamplesCount: Integer;
    procedure M_GetSample(Render: Boolean; SNum: Integer; var SL: SmallInt; var SR: SmallInt);
    procedure M_SetSample(Render: Boolean; SL, SR: SmallInt);
    procedure M_Clip(Render: Boolean; CV: Integer);
    procedure AE_StartEngine;
    procedure AE_StopEngine;
    function AE_Exception(Message: String): Boolean;

    //Internal procedures
    procedure AdvanceSong;
   public
    ExceptMessage: String;
    constructor Create(ASong: TSong; AWaveDevice: Integer;
      ABufCount: Integer; ABufSize: Integer; AHandle: HWND;
      ALoopPattern: Boolean);
    destructor Destroy; override;
    procedure StartPlaySong;
    procedure StartPlayPattern(Pattern: Integer);
    procedure StartPlaySample(Sample: Integer);
    //For editing audition
    procedure StartNote(Track, Channel: Integer; C: TCommand);
    procedure XStartNote(Track, Channel: Integer; C: TCommand);
    procedure StopNote(Track, Channel: Integer);
    procedure XStopNote(Track, Channel: Integer); //ends loop
    procedure SetVolPan(Track, Channel: Integer; Vol: Byte; Pan: ShortInt);
    procedure SetNote(Track, Channel: Integer; Note: Byte);
    procedure StopPlay;
    procedure Render(FileName: String);
  end;

implementation

uses MSConsts, Notes, SysUtils;

{ TSampleWrapper }

procedure TSampleWrapper.P2R;
var FCC: Double;
begin
  if (FSample<1) or (FSample>FFSong.Samples.Count) then Exit;
  with FFSong.Samples[FSample] do
    begin
      FCC := FFSong.Samples[FSample].Rate/FRate;
      FRealPos := Round((FPosition-1)/(FCoef/FCC)+1);
    end;
end;

procedure TSampleWrapper.R2P;
var FCC: Double;
begin
  if (FSample<1) or (FSample>FFSong.Samples.Count) then Exit;
  with FFSong.Samples[FSample] do
    begin
      FCC := FFSong.Samples[FSample].Rate/FRate;
      FPosition := Round((FRealPos-1)*(FCoef/FCC)+1);
    end;
end;

function TSampleWrapper.InRange: Boolean;
begin
  if (FSample<1) or (FSample>FFSong.Samples.Count) then
    result := false
  else
    result := FRealPos<=FFSong.Samples[FSample].Length;
end;

procedure TSampleWrapper.GetSamples(var SL, SR: SmallInt);
begin
  if (FSample<1) or (FSample>FFSong.Samples.Count) or not InRange then
    begin
      SL := 0;
      SR := 0;
    end
  else
    begin
      SL := FFSong.Samples[FSample].XGetSampleL(FRealPos);
      SR := FFSong.Samples[FSample].XGetSampleR(FRealPos);
      if (FPan>0) then
        SL := SL*(100-Abs(FPan)) div 100
      else
      if (FPan<0) then
        SR := SR*(100-Abs(FPan)) div 100;
      if FVolume<>100 then
        begin
          SL := SL*FVolume div 100;
          SR := SR*FVolume div 100;
        end;
    end;
end;

procedure TSampleWrapper.NextStep;
begin
  if (FSample<1) or (FSample>FFSong.Samples.Count) then Exit;
  with FFSong.Samples[FSample] do
    begin
      if (LoopType<>lNone) and
         (LoopStart>0) and (LoopStart<=Length) and
         (LoopEnd>0) and (LoopEnd<=Length) and
         (LoopEnd>=LoopStart) then
        begin
          if (LoopType=lBidi) and FLoopBack then
            FPosition := FPosition-1
          else FPosition := FPosition+1;
          P2R;
          if (LoopType=lNormal) and (FRealPos>LoopEnd) and
             not FEnded then
            begin
              FRealPos := LoopStart;
              R2P;
            end
          else
          if (LoopType=lBidi) and (FRealPos>LoopEnd) and
             not FLoopBack and not FEnded then
            begin
              FLoopBack := true;
              FPosition := FPosition-1;
              P2R;
            end
          else
          if (LoopType=lBidi) and (FRealPos<LoopStart) and FLoopBack then
            begin
              FLoopBack := false;
              FPosition := FPosition+1;
              P2R;
            end;
        end
      else
        begin
          FPosition := FPosition+1;
          P2R;
        end;
    end;
end;

{ TPlayingSamples }

function TPlayingSamples.GetSample(i: Integer): TSampleWrapper;
begin
  result := TSampleWrapper(Items[i-1]);
end;

destructor TPlayingSamples.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TPlayingSamples.Delete(i: Integer);
begin
  TSampleWrapper(Items[i-1]).Free;
  inherited Delete(i-1);
end;

procedure TPlayingSamples.Clear;
begin
  while Count>0 do Delete(1);
end;

{ TSongPlayer }

procedure TSongPlayer.AdvanceSong;
var CP: Boolean; i, j: Integer;
begin
  CP := false;
  if FollowSong and not FInRender and (FHandle<>0) then
    PostMessage(FHandle, WM_SETSPOS, FSPos, 0);
  with FSong.Sequence do
    for i := 1 to FSong.Sequence.Tracks do
      begin
        //Skip not existing patterns
        while (Pos[i]>0) and (Pos[i]<=TrackLength[i]) and
              ((PatternNum[i, Pos[i]]<1) or
               (PatternNum[i, Pos[i]]>FSong.Patterns.Count)) do
        Pos[i] := Pos[i]+1;

        //Process every track
        if (Pos[i]>0) and (Pos[i]<=TrackLength[i]) then
          with FSong.Patterns[PatternNum[i, Pos[i]]] do
            begin
              //Pattern position OK
              if (PPos[i]>0) and (PPos[i]<=Length) then

                //Process every channel
                for j := 1 to Channels do
                  StartNote(i, j, Command[j, PPos[i]]);

              //Continue play
              CP := true;

              //Advance position in pattern
              PPos[i] := PPos[i]+1;

              //If pattern end, advance position in track
              if PPos[i]>Length then
                begin
                  PPos[i] := 1;
                  Pos[i] := Pos[i]+1;
                end;
            end;
        end;
  Inc(FSPos);
  //If song over - stop play
  if not CP then FPlaying := false;
end;

procedure TSongPlayer.M_NextStep;
var i: Integer;
begin
  if not FPlaying {and (FTickCount=0)} and
     ((FPlayMode=pmPattern) or (FPlayMode=pmSong)) then
    begin
      FPlayingSamples.Clear;
    end;
  if FPlaying and (FTickCount=0) then
    if FPlayMode=pmPattern then //Process standalone pattern
      begin
        if (FPattern<1) or (FPattern>FSong.Patterns.Count) or
           (FPos<1) or (FPos>FSong.Patterns[FPattern].Length) then
          FPlaying := false
        else
          begin
          if FollowPattern and not FInRender and (FHandle<>0) then
            PostMessage(FHandle, WM_SETPPOS, FPattern, FPos);
          i := 1;
          while i<=FSong.Patterns[FPattern].Channels do
            begin
              StartNote(1, i, FSong.Patterns[FPattern][i, FPos]);
              Inc(i);
            end;
            Inc(FPos);
            if FPos>FSong.Patterns[FPattern].Length then
              if FLoopPattern then FPos := 1;
          end;
      end
    else
    if FPlayMode=pmSong then //Process entire song
      AdvanceSong;
  //Process samples
  i := 1;
  while i<=FPlayingSamples.Count do
    begin
      FPlayingSamples[i].NextStep;
      if not FPlayingSamples[i].InRange then
        FPlayingSamples.Delete(i)
      else Inc(i);
    end;
  Inc(FTickCount);
  if FTickCount>=FSong.SamplesPerTick then FTickCount := 0;
end;

function TSongPlayer.M_GetSamplesCount: Integer;
begin
  result := FPlayingSamples.Count;
  if (result=0) and FPlaying then
    result := 1;
end;

procedure TSongPlayer.M_GetSample(Render: Boolean; SNum: Integer; var SL: SmallInt; var SR: SmallInt);
begin
  if (SNum>0) and (SNum<=FPlayingSamples.Count) then
    begin
      if FPlayingSamples[SNum].FVolume=0 then
        begin
          SL := 0;
          SR := 0;
          Exit;
        end;
      FPlayingSamples[SNum].GetSamples(SL, SR);
      if FSong.SongVolume<>100 then
        begin
          SL := SL*FSong.SongVolume div 100;
          SR := SR*FSong.SongVolume div 100;
        end;
    end
  else
    begin
      SL := 0;
      SR := 0;
    end;
end;

procedure TSongPlayer.M_SetSample(Render: Boolean; SL, SR: SmallInt);
begin
  if EnableMeter and (FTickCountM=0) and not Render and (FHandle<>0) then
    PostMessage(FHandle, WM_SETSAMPLE, SL, SR);
  Inc(FTickCountM);
  if FTickCountM>=MeterRate then FTickCountM := 0;
end;

procedure TSongPlayer.M_Clip(Render: Boolean; CV: Integer);
var CC: Integer;
begin
  if CV<0 then CC := -32768 else CC := 32767;
  if not Render and (FHandle<>0) then
    PostMessage(FHandle, WM_CLIP, CC, CV);
end;

procedure TSongPlayer.AE_StartEngine;
begin
  FTickCount := 0;
  FTickCountM := 0;
  if FHandle<>0 then
    PostMessage(FHandle, WM_START, Ord(FPlayMode), 0);
end;

procedure TSongPlayer.AE_StopEngine;
begin
  FPlayMode := pmNone;
  FPlaying := false;
  FInRender := false;
  FTickCount := 0;
  if FHandle<>0 then
    PostMessage(FHandle, WM_STOP, 0, 0);
end;

function TSongPlayer.AE_Exception(Message: String): Boolean;
begin
  result := true;
  ExceptMessage := Message;
  if FHandle<>0 then
    PostMessage(FHandle, WM_EXCEPT, 0, 0)
  else Raise Exception.Create(Message);
end;

procedure TSongPlayer.StartNote(Track, Channel: Integer; C: TCommand);
var SW: TSampleWrapper;
begin
  if C.Note=NoteCut then
    begin
      StopNote(Track, Channel);
      Exit;
    end;
  if C.Note=NoteOff then
    begin
      XStopNote(Track, Channel);
      Exit;
    end;
  if (C.Note=NoNote) or (C.Note>MaxNote) then
    begin
      SetVolPan(Track, Channel, C.Volume, C.Pan);
      Exit;
    end;
  if (C.Instrument=0) and (C.Note>=0) and (C.Note<=MaxNote) then
    begin
      SetNote(Track, Channel, C.Note);
      Exit;
    end;
  if (C.Instrument<1) or (C.Instrument>FSong.Samples.Count) then Exit;
  if FSong.Samples[C.Instrument].Length=0 then Exit;
  if FSong.NNA=nnaChannelCut then StopNote(Track, Channel);
  if FSong.NNA=nnaChannelOff then XStopNote(Track, Channel);
  SW := TSampleWrapper.Create;
  with SW do
    try
      FFSong := FSong;
      FSample := C.Instrument;
      FVolume := C.Volume;
      FPan := C.Pan;
      if FVolume=DVol then FVolume := FFSong.Samples[FSample].DefVolume;
      if FPan=DPan then FPan := FFSong.Samples[FSample].DefPan;
      if (FVolume>200) then FVolume := 100;
      if (FPan<-100) or (FPan>100) then FPan := 0;
      FPosition := 0;
      FRealPos := 0;
      FTrack := Track;
      FChannel := Channel;
      FNote := C.Note;
      FRate := BaseRate;
      //FRate := FFSong.Samples[FSample].Rate;
      FCoef := NoteCoef(C.Note);
      FLoopBack := false;
      FEnded := false;
      FPlayingSamples.Add(SW);
    except
      SW.Free;
    end;
end;

procedure TSongPlayer.StopNote(Track, Channel: Integer);
var i: Integer;
begin
  i := 1;
  while i<=FPlayingSamples.Count do
    begin
      if (FPlayingSamples[i].FTrack=Track) and
         (FPlayingSamples[i].FChannel=Channel) then
        FPlayingSamples.Delete(i)
      else Inc(i);
    end;
end;

procedure TSongPlayer.XStopNote(Track, Channel: Integer);
var i: Integer;
begin
  i := 1;
  while i<=FPlayingSamples.Count do
    begin
      if (FPlayingSamples[i].FTrack=Track) and
         (FPlayingSamples[i].FChannel=Channel) then
        FPlayingSamples[i].FEnded := true;
      Inc(i);
    end;
end;

procedure TSongPlayer.SetVolPan(Track, Channel: Integer; Vol: Byte; Pan: ShortInt);
var i: Integer;
begin
  i := 1;
  while i<=FPlayingSamples.Count do
    begin
      if (FPlayingSamples[i].FTrack=Track) and
         (FPlayingSamples[i].FChannel=Channel) then
        begin
          if (Vol>=0) and (Vol<=200) then
            FPlayingSamples[i].FVolume := Vol;
          if (Pan>=-100) and (Pan<=100) then
            FPlayingSamples[i].FPan := Pan;
        end;
      Inc(i);
    end;
end;

procedure TSongPlayer.SetNote(Track, Channel: Integer; Note: Byte);
var i: Integer;
begin
  i := 1;
  while i<=FPlayingSamples.Count do
    begin
      if (FPlayingSamples[i].FTrack=Track) and
         (FPlayingSamples[i].FChannel=Channel) then
        if (Note>=0) and (Note<=MaxNote) then
          begin
            FPlayingSamples[i].FNote := Note;
            FPlayingSamples[i].FCoef := NoteCoef(Note);
            FPlayingSamples[i].R2P;
          end;
      Inc(i);
    end;
end;

constructor TSongPlayer.Create(ASong: TSong; AWaveDevice: Integer;
  ABufCount: Integer; ABufSize: Integer; AHandle: HWND;
  ALoopPattern: Boolean);
begin
  FSong := ASong;
  FHandle := AHandle;
  FWaveDevice := AWaveDevice;
  FPlayingSamples := TPlayingSamples.Create;
  FAudioEngine := TAudioEngine.Create(FWaveDevice, ABufCount, ABufSize);
  FMixer := TWaveMixer.Create(FAudioEngine);
  FPLayMode := pmNone;
  FPlaying := false;
  FPattern := 0;
  FPos := 1;
  FLoopPattern := ALoopPattern;

  FMixer.OnNextStep := M_NextStep;
  FMixer.OnGetSamplesCount := M_GetSamplesCount;
  FMixer.OnGetSample := M_GetSample;
  FMixer.OnSetSample := M_SetSample;
  FMixer.OnClip := M_Clip;
  FAudioEngine.OnStartEngine := AE_StartEngine;
  FAudioEngine.OnStopEngine := AE_StopEngine;
  FAudioEngine.OnException := AE_Exception;
end;

destructor TSongPlayer.Destroy;
begin
  StopPlay;
  if Assigned(FMixer) then FMixer.Free;
  if Assigned(FAudioEngine) then FAudioEngine.Free;
  if Assigned(FPlayingSamples) then FPlayingSamples.Free;
end;

procedure TSongPlayer.StartPlaySong;
var i: Integer;
begin
  StopPlay;
  FSPos := 1;
  for i := 1 to FSong.Sequence.Tracks do
    begin
      FSong.Sequence.Pos[i] := 1;
      FSong.Sequence.PPos[i] := 1;
    end;
  FPlaying := true;
  FPlayMode := pmSong;
  FInRender := false;
  FAudioEngine.Start;
end;

procedure TSongPlayer.StartPlayPattern(Pattern: Integer);
begin
  StopPlay;
  FPattern := Pattern;
  FPos := 1;
  FPlaying := true;
  FPlayMode := pmPattern;
  FInRender := false;
  FAudioEngine.Start;
end;

procedure TSongPlayer.XStartNote(Track, Channel: Integer; C: TCommand);
begin
  FAudioEngine.Pause;
  try
    StartNote(Track, Channel, C);
    if FPlayMode=pmNone then
      begin
        FPlaying := false;
        FPlayMode := pmSample;
        FInRender := false;
        FAudioEngine.Start;
      end;
  finally
    FAudioEngine.Resume;
  end;
end;

procedure TSongPlayer.StartPlaySample(Sample: Integer);
var C: TCommand;
begin
  C.Note := BaseNote;
  C.Instrument := Sample;
  C.Volume := DVol;
  C.Pan := DPan;
  XStartNote(0, 0, C);
end;

procedure TSongPlayer.StopPlay;
begin
  if not Assigned(FAudioEngine) then Exit;
  FAudioEngine.Pause;
  try
    if FPlayMode=pmNone then Exit;
    FPlayingSamples.Clear;
  finally
    FAudioEngine.Resume;
  end;
  FAudioEngine.Stop;
end;

procedure TSongPlayer.Render(FileName: String);
var i: Integer;
begin
  StopPlay;
  FSPos := 1;
  for i := 1 to FSong.Sequence.Tracks do
    begin
      FSong.Sequence.Pos[i] := 1;
      FSong.Sequence.PPos[i] := 1;
    end;
  FPlaying := true;
  FPlayMode := pmSong;
  FInRender := true;
  FAudioEngine.Render(FileName);
end;

end.
