unit Music;

interface

uses Classes, SysUtils, Midi;

type
  TFS=(None, Flat, Sharp);

  TEventType =
    (
      evText,
      evLyric,
      evMarker,
      evTrackName,
      evInstrumentName,
      evCopyright
    );

  TTextCallback = procedure(Event: TEventType; Text: String) of Object;

  TSong = class;

  TTrack = class
   private
    FData: String;
    FTranslatedData: String;
    FSleeping: Boolean;
    FSleepUntil: Integer;
    FPlaying: Boolean;
    FDataPos: Integer;
    FChannel: Byte;
    FTempo: Word;
    FDuration: Byte;
    FOctave: Byte;
    FVelocity: Byte;
    FLastNote: Byte;
    FLastVelocity: Byte;
    FLastChannel: Byte;
    FLastTime: Integer;
    FMidiOut: TAbstractMidiOut;
    FSong: TSong;
    procedure SetData(AData: String);
    function CalcTime(Duration: Byte; Dotted: Boolean): Integer;
    procedure StopNote;
    procedure Init;
    procedure Error(S: String);
    function Translate(S: String; Patterns: TStrings): String;
    procedure CheckChannel(Channel: Byte);
    procedure CheckTempo(Tempo: Word);
    procedure CheckDuration(Duration: Byte);
    procedure CheckOctave(Octave: Byte);
    procedure CheckVelocity(Velocity: Byte);
    procedure CheckBank(Bank: Byte);
    procedure CheckPatch(Patch: Byte);
    procedure CheckNote(Note: Byte);
    procedure CheckBend(Bend: Word);
    procedure CheckBendUp(Bend: Word);
    procedure CheckBendDown(Bend: Word);
    function CalcNote(O: Byte; N: Char; F: TFS): Byte;
    function GetTickCount: Integer;
   public
    DropoutDetect: Boolean;
    DropoutCount: Integer;
    BankChangeControl: Byte;
    TextCallback: TTextCallback;
    Patterns: TStrings;
    constructor Create(MidiOut: TAbstractMidiOut);
    destructor Destroy; override;
    procedure ProcessCommand;
    procedure Play;
    procedure Stop;
    function Export: String;
    property Data: String read FData write SetData;
    property TranslatedData: String read FTranslatedData;
    property Sleeping: Boolean read FSleeping;
    property SleepUntil: Integer read FSleepUntil;
    property Playing: Boolean read FPlaying;
    property DataPos: Integer read FDataPos;
    property Channel: Byte read FChannel;
    property Tempo: Word read FTempo;
    property Duration: Byte read FDuration;
    property Octave: Byte read FOctave;
    property Velocity: Byte read FVelocity;
  end;

  TSong = class
   private
    FPatterns: TStrings;
    FTracks: TList;
    FMidiOut: TAbstractMidiOut;
    procedure Error(S: String);
    function GetTrackData(Track: Integer): String;
    function GetTrackTranslatedData(Track: Integer): String;
    procedure SetTrackData(Track: Integer; Data: String);
    procedure SetTempo(Tempo: Word);
   public
    DropoutDetect: Boolean;
    DropoutCount: Integer;
    BankChangeControl: Byte;
    TextCallback: TTextCallback;
    constructor Create(MidiOut: TAbstractMidiOut);
    destructor Destroy; override;
    function TracksCount: Integer;
    procedure DeleteTracks;
    procedure AddTrack(Data: String);
    procedure Play;
    procedure Export(FileName: String);
    property Patterns: TStrings read FPatterns;
    property TrackData[Track: Integer]: String
      read GetTrackData write SetTrackData;
    property TrackTranslatedData[Track: Integer]: String
      read GetTrackTranslatedData;
  end;

  EMusicError = class(Exception);
  ESongError = class(Exception);

implementation

uses Windows;

{$R-}

//MIDI ticks per beat (quarter note)
const TPB: Word = 128;

{ TTrack }

procedure TTrack.Init;
begin
  FSleeping := false;
  FSleepUntil := 0;
  FDataPos := 1;
  FChannel := 1;
  FTempo := 140;
  FDuration := 16;
  FOctave := 5;
  FVelocity := 127;
  FLastNote := $FF;
  FLastVelocity := FVelocity;
  FLastChannel := FChannel;
  FLastTime := 0;
end;

constructor TTrack.Create(MidiOut: TAbstractMidiOut);
begin
  inherited Create;
  FData := '';
  FTranslatedData := '';
  FPlaying := false;
  Patterns := nil;
  FMidiOut := MidiOut;
  DropoutDetect := false;
  DropoutCount := 100;
  BankChangeControl := 0;
  TextCallback := nil;
  Init;
end;

destructor TTrack.Destroy;
begin
  Stop;
  inherited Destroy;
end;

procedure TTrack.Error(S: String);
begin
  Raise EMusicError.Create(S);
end;

function TTrack.Translate(S: String; Patterns: TStrings): String;
var PatNum, strPos: Integer; SS: String;
begin
  StrPos := 1;
  while StrPos<=Length(S) do
    begin
      if S[StrPos]='%' then
        begin
          Inc(StrPos);
          SS := '';
          while (StrPos<=Length(S)) and (S[StrPos] in ['0'..'9']) do
            begin
              SS := SS+S[StrPos];
              Inc(StrPos);
            end;
          if not Assigned(Patterns) then Error('No patterns list')
          else
          if (SS='') then Error('Pattern number not specified')
          else
            begin
              PatNum := StrToInt(SS);
              if (PatNum<0) or (PatNum>Patterns.Count-1) then
                Error('Invalid pattern number: '+SS)
              else result := result+Patterns[PatNum];
            end;
        end
      else
        begin
          result := result+S[StrPos];
          Inc(StrPos);
        end;
    end;
end;

procedure TTrack.SetData(AData: String);
begin
  if FPlaying then Stop;
  FData := AData;
  Init;
  FTranslatedData := Translate(FData, Patterns);
end;

procedure TTrack.CheckChannel(Channel: Byte);
begin
  if (Channel=0) or (Channel>16) then
    Error('Invalid channel: '+IntToStr(Channel));
end;

procedure TTrack.CheckTempo(Tempo: Word);
begin
  if (Tempo<10) or (Tempo>500) then
    Error('Invalid tempo: '+IntToStr(Tempo));
end;

procedure TTrack.CheckDuration(Duration: Byte);
begin
  if not (Duration in [1,2,4,8,16,32,64]) then
    Error('Invalid duration: '+IntToStr(Duration));
end;

procedure TTrack.CheckOctave(Octave: Byte);
begin
  if Octave>10 then
    Error('Invalid octave: '+IntToStr(Octave));
end;

procedure TTrack.CheckVelocity(Velocity: Byte);
begin
  if Velocity>127 then
    Error('Invalid velocity: '+IntToStr(Velocity));
end;

procedure TTrack.CheckBank(Bank: Byte);
begin
  if Bank>127 then
    Error('Invalid bank: '+IntToStr(Bank));
end;

procedure TTrack.CheckPatch(Patch: Byte);
begin
  if Patch>127 then
    Error('Invalid patch: '+IntToStr(Patch));
end;

procedure TTrack.CheckNote(Note: Byte);
begin
  if Note>127 then
    Error('Invalid note: '+IntToStr(Note));
end;

procedure TTrack.CheckBend(Bend: Word);
begin
  if Bend>$3FFF then
    Error('Invalid pitch bend: '+IntToStr(Bend));
end;

procedure TTrack.CheckBendUp(Bend: Word);
begin
  if Bend>$1FFF then
    Error('Invalid pitch bend up: '+IntToStr(Bend));
end;

procedure TTrack.CheckBendDown(Bend: Word);
begin
  if Bend>$2000 then
    Error('Invalid pitch bend down: '+IntToStr(Bend));
end;

function TTrack.CalcTime(Duration: Byte; Dotted: Boolean): Integer;
//var R: double;
begin
  CheckDuration(Duration);
  result := 128 div Duration;
  if Dotted then result := result*3 div 2;
{  R := 60000*4/FTempo/Duration;
  if Dotted then R := R*3/2;
  result := Round(R);}
end;

procedure TTrack.StopNote;
begin
  if FLastNote<>$FF then
    if Assigned(FMidiOut) then
      FMidiOut.NoteOff(FLastChannel-1, FLastNote, FLastVelocity);
end;

function TTrack.CalcNote(O: Byte; N: Char; F: TFS): Byte;
begin
  result := 0;
  CheckOctave(O);
  Case UpCase(N) of
    'C': result := 0;
    'D': result := 2;
    'E': result := 4;
    'F': result := 5;
    'G': result := 7;
    'A': result := 9;
    'B': result := 11;
    else Error('Invalid note symbol: '+N);
  end;
  result := result+O*12;
  if F=Flat then result := result-1
  else
  if F=Sharp then result := result+1;
  CheckNote(result);
end;

function TTrack.GetTickCount: Integer;
begin
//  result := Windows.GetTickCount;
  result := Round(Windows.GetTickCount/60000/4*FTempo*128);
end;

procedure TTrack.ProcessCommand;
var CurChar, Cmd: Char; O, N, L: Byte; F: TFS; D: Boolean; T: Word; S: String;
    NewTime: Integer;

procedure GetChar;
begin
  if FDataPos>Length(FTranslatedData) then CurChar := #0
  else
    begin
      CurChar := FTranslatedData[FDataPos];
      Inc(FDataPos);
    end;
end;

function GetInteger: Integer;
var S: String;
begin
  result := 0;
  while CurChar in ['0'..'9'] do
    begin
      S := S+CurChar;
      GetChar;
    end;
  if S='' then Error('Number expected')
  else result := StrToInt(S);
end;

begin
  if FPlaying and FSleeping and (GetTickCount<SleepUntil) then Exit;
  if FPlaying and FSleeping then
    begin
      FSleeping := false;
      if DropoutDetect and (GetTickCount-SleepUntil>DropoutCount) then
        Error('Dropout');
      FSleepUntil := 0;
    end;

  if FDataPos>Length(FTranslatedData) then
    begin
      Stop;
      Exit;
    end;

  if not FPlaying then FPlaying := true;

  GetChar;
  Cmd := CurChar;

  GetChar;
  case UpCase(Cmd) of
    #0..' ', '_': ;
    'C','D','E','F','G','A','B':
      begin
        F := None;
        if CurChar in ['#', '+'] then
          begin
            F := Sharp;
            GetChar;
          end
        else
        if CurChar='-' then
          begin
            F := Flat;
            GetChar;
          end;
        if CurChar in ['0'..'9'] then O := GetInteger
        else O := FOctave;
        CheckOctave(O);
        if CurChar=',' then
          begin
            GetChar;
            L := GetInteger;
          end
        else L := FDuration;
        CheckDuration(L);
        D := false;
        if CurChar='.' then
          begin
            D := true;
            GetChar;
          end;
        N := CalcNote(O, Cmd, F);
        StopNote;
        FLastNote := N;
        FLastVelocity := FVelocity;
        FLastChannel := FChannel;
        FLastTime := CalcTime(L, D);
//        writeln(format('%p', [pointer(self)]), '->',gettickcount, ' ', flasttime);
        if Assigned(FMidiOut) then
          FMidiOut.NoteOn(FChannel-1, N, FVelocity);
        FSleeping := true;
        FSleepUntil := GetTickCount+FLastTime;
      end;
    'N':
      begin
        N := GetInteger;
        CheckNote(N);
        if CurChar=',' then
          begin
            GetChar;
            L := GetInteger;
          end
        else L := FDuration;
        CheckDuration(L);
        D := false;
        if CurChar='.' then
          begin
            D := true;
            GetChar;
          end;
        StopNote;
        FLastNote := N;
        FLastVelocity := FVelocity;
        FLastChannel := FChannel;
        FLastTime := CalcTime(L, D);
        if Assigned(FMidiOut) then
          FMidiOut.NoteOn(FChannel-1, N, FVelocity);
        FSleeping := true;
        FSleepUntil := GetTickCount+FLastTime;
      end;
    'P':
      begin
        if CurChar in ['0'..'9'] then L := GetInteger
        else L := FDuration;
        CheckDuration(L);
        D := false;
        if CurChar='.' then
          begin
            D := true;
            GetChar;
          end;
        StopNote;
        FLastNote := $FF;
        FLastTime := CalcTime(L, D);
//        writeln(format('%p', [pointer(self)]), 'p ',gettickcount, ' ', flasttime);
        FSleeping := true;
        FSleepUntil := GetTickCount+FLastTime
      end;
    '=':
      begin
        D := false;
        if CurChar in ['0'..'9'] then
          begin
            L := GetInteger;
            CheckDuration(L);
            D := false;
            if CurChar='.' then
              begin
                D := true;
                GetChar;
              end;
            NewTime := CalcTime(L, D);
            D := false;
            if CurChar='*' then
              begin
                D := true;
                GetChar;
              end;
          end
        else NewTime := 0;
        if FLastTime<>0 then
          begin
            FSleeping := true;
            if NewTime<>0 then
              begin
//        writeln(format('%p', [pointer(self)]), '= ',gettickcount, ' ', newtime);
                FSleepUntil := GetTickCount+NewTime;
                if D then FLastTime := NewTime;
              end
            else
              {begin }FSleepUntil := GetTickCount+FLastTime;
//        writeln(format('%p', [pointer(self)]), '= ',gettickcount, ' ', flasttime); end;

          end;
      end;
    'O':
      begin
        O := GetInteger;
        CheckOctave(O);
        FOctave := O;
      end;
    'L':
      begin
        L := GetInteger;
        CheckDuration(L);
        FDuration := L;
      end;
    'V':
      begin
        O := GetInteger;
        CheckVelocity(O);
        FVelocity := O;
      end;
    '!':
      begin
        O := GetInteger;
        CheckChannel(O);
        FChannel := O;
      end;
    'T':
      begin
        T := GetInteger;
        CheckTempo(T);
        FTempo := T;
        if Assigned(FSong) then FSong.SetTempo(T);
      end;
    'S':
      begin
        O := GetInteger;
        CheckBank(O);
        if Assigned(FMidiOut) then
          FMidiOut.ControlChange(FChannel-1, BankChangeControl, O)
      end;
    'I':
      begin
        O := GetInteger;
        CheckPatch(O);
        if Assigned(FMidiOut) then
          FMidiOut.PatchChange(FChannel-1, O)
      end;
    '>':
      begin
        O := FOctave+1;
        CheckOctave(O);
        FOctave := O;
      end;
    '<':
      begin
        O := FOctave-1;
        CheckOctave(O);
        FOctave := O;
      end;
    '~':
      begin
        T := GetInteger;
        CheckBend(T);
        if Assigned(FMidiOut) then
          FMidiOut.PitchBend(FChannel-1, T)
      end;
    '/':
      begin
        T := GetInteger;
        CheckBendUp(T);
        if Assigned(FMidiOut) then
          FMidiOut.PitchBend(FChannel-1, $2000+T)
      end;
    '\':
      begin
        T := GetInteger;
        CheckBendDown(T);
        if Assigned(FMidiOut) then
          FMidiOut.PitchBend(FChannel-1, $2000-T)
      end;
    '|':
      begin
        if Assigned(FMidiOut) then
          FMidiOut.PitchBend(FChannel-1, $2000)
      end;
    '{':
      begin
        S := '';
        while not (CurChar in ['}', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          if Assigned(TextCallback) then
            TextCallback(evText, S);
        GetChar;
      end;
    '[':
      begin
        S := '';
        while not (CurChar in [']', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          if Assigned(TextCallback) then
            TextCallback(evLyric, S);
        GetChar;
      end;
    '(':
      begin
        S := '';
        while not (CurChar in [')', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          if Assigned(TextCallback) then
            TextCallback(evMarker, S);
        GetChar;
      end;
    '"':
      begin
        S := '';
        while not (CurChar in ['"', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          if Assigned(TextCallback) then
            TextCallback(evTrackName, S);
        GetChar;
      end;
    '`':
      begin
        S := '';
        while not (CurChar in ['`', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          if Assigned(TextCallback) then
            TextCallback(evInstrumentName, S);
        GetChar;
      end;
    '''':
      begin
        S := '';
        while not (CurChar in ['''', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          if Assigned(TextCallback) then
            TextCallback(evCopyright, S);
        GetChar;
      end;
    '?':
      begin
        O := GetInteger;
        CheckBank(O);
        if CurChar=':' then
          begin
            GetChar;
            while not (CurChar in [';', #0]) do
              begin
                S := S+CurChar;
                GetChar;
              end;
            GetChar;
            if Assigned(FMidiOut) then FMidiOut.LoadBank(O, S);
          end
        else
        if CurChar=';' then
          begin
            GetChar;
            if Assigned(FMidiOut) then FMidiOut.ClearBank(O);
          end
      end;
    else Error('Invalid command: '+Cmd);
  end;
  if CurChar<>#0 then FDataPos := FDataPos-1;
end;

procedure TTrack.Play;
begin
  Init;
  repeat
    try
      ProcessCommand;
    except
      Stop;
      Raise;
    end;
  until not FPlaying;
end;

procedure TTrack.Stop;
begin
  if not FPlaying then Exit;
  StopNote;
  FPlaying := false;
  Init;
end;

function TTrack.Export: String;
var Cmds: TStrings; Ended: Boolean; Pos: Integer; WasTempo, WasChannel: Boolean;

function CalcTimeMidi(Duration: Byte; Dotted: Boolean): Integer;
var R: Double;
begin
  CheckDuration(Duration);
  R := TPB*4/Duration;
  if Dotted then R := R*1.5;
  result := Round(R);
end;

function CalcMidiTempo: String;
var I: Integer;
begin
  I := 60000000 div FTempo;
  result := Char(I shl 8 shr 24)+Char(I shl 16 shr 24)+Char(I shl 24 shr 24);
end;

function EncodedTime(Time: Integer): String;
var B: Byte; R: Integer; S: String;
begin
  S := '';
  R := Time;
  repeat
    B := R mod 128;
    R := R div 128;
    S := S+Char(B or 128);
  until R=0;
  result := '';
  S[1] := Char(Byte(S[1]) and not 128);
  for R := Length(S) downto 1 do
    result := result+S[R];
end;

procedure CheckDTempo;
begin
  if not WasTempo then
    begin
      Cmds.Add('C'#$FF#$51+#3+CalcMidiTempo);
      WasTempo := true;
    end;
end;

procedure CheckDChannel;
begin
  if not WasChannel then
    begin
      Cmds.Add('C'#$FF#$20+#1+Char(FChannel-1));
      WasChannel := true;
    end;
end;

procedure ExportCommand;
var CurChar, Cmd: Char; O, N, L: Byte; F: TFS; D: Boolean; T: Word; S: String;
    NewTime: Integer;

procedure GetChar;
begin
  if Pos>Length(FTranslatedData) then CurChar := #0
  else
    begin
      CurChar := FTranslatedData[Pos];
      Inc(Pos);
    end;
end;

function GetInteger: Integer;
var S: String;
begin
  result := 0;
  while CurChar in ['0'..'9'] do
    begin
      S := S+CurChar;
      GetChar;
    end;
  if S='' then Error('Number expected')
  else result := StrToInt(S);
end;

procedure AddTime(T: Integer);
begin
  Cmds.Add('T'+IntToStr(T));
end;

procedure AddCmd(Cmd, Chn, D1, D2: Byte);
begin
  Cmds.Add('C'+Char(Cmd shl 4 or Chn)+Char(D1)+Char(D2));
end;

procedure AddCmd3(Cmd, Chn, D1: Byte);
begin
  Cmds.Add('C'+Char(Cmd shl 4 or Chn)+Char(D1));
end;

procedure NoteOn(Channel, Note, Velocity: Byte);
begin
  AddCmd($9, Channel, Note, Velocity);
end;

procedure NoteOff(Channel, Note, Velocity: Byte);
begin
  AddCmd($8, Channel, Note, Velocity);
end;

procedure ControlChange(Channel, Control, Value: Byte);
begin
  AddCmd($B, Channel, Control, Value);
end;

procedure PatchChange(Channel, Patch: Byte);
begin
  AddCmd3($C, Channel, Patch);
end;

procedure PitchBend(Channel: Byte; Bend: Word);
begin
  AddCmd($E, Channel, Bend mod 128, Bend div 128);
end;

procedure EndNote;
begin
  if FLastNote<>$FF then
    NoteOff(FLastChannel-1, FLastNote, FLastVelocity);
end;

procedure EndExport;
begin
  EndNote;
  Ended := true;
  Init;
end;

begin
  if Pos>Length(FTranslatedData) then
    begin
      EndExport;
      Exit;
    end;

  GetChar;
  Cmd := CurChar;

  GetChar;
  case UpCase(Cmd) of
    #0..' ', '_': ;
    'C','D','E','F','G','A','B':
      begin
        CheckDTempo;
        F := None;
        if CurChar in ['#', '+'] then
          begin
            F := Sharp;
            GetChar;
          end
        else
        if CurChar='-' then
          begin
            F := Flat;
            GetChar;
          end;
        if CurChar in ['0'..'9'] then O := GetInteger
        else O := FOctave;
        CheckOctave(O);
        if CurChar=',' then
          begin
            GetChar;
            L := GetInteger;
          end
        else L := FDuration;
        CheckDuration(L);
        D := false;
        if CurChar='.' then
          begin
            D := true;
            GetChar;
          end;
        N := CalcNote(O, Cmd, F);
        EndNote;
        FLastNote := N;
        FLastVelocity := FVelocity;
        FLastChannel := FChannel;
        FLastTime := CalcTimeMidi(L, D);
        NoteOn(FChannel-1, N, FVelocity);
        AddTime(FLastTime);
      end;
    'N':
      begin
        CheckDTempo;
        N := GetInteger;
        CheckNote(N);
        if CurChar=',' then
          begin
            GetChar;
            L := GetInteger;
          end
        else L := FDuration;
        CheckDuration(L);
        D := false;
        if CurChar='.' then
          begin
            D := true;
            GetChar;
          end;
        EndNote;
        FLastNote := N;
        FLastVelocity := FVelocity;
        FLastChannel := FChannel;
        FLastTime := CalcTimeMidi(L, D);
        NoteOn(FChannel-1, N, FVelocity);
        AddTime(FLastTime);
      end;
    'P':
      begin
        CheckDTempo;
        if CurChar in ['0'..'9'] then L := GetInteger
        else L := FDuration;
        CheckDuration(L);
        D := false;
        if CurChar='.' then
          begin
            D := true;
            GetChar;
          end;
        EndNote;
        FLastNote := $FF;
        FLastTime := CalcTimeMidi(L, D);
        AddTime(FLastTime);
      end;
    '=':
      begin
        D := false;
        if CurChar in ['0'..'9'] then
          begin
            L := GetInteger;
            CheckDuration(L);
            D := false;
            if CurChar='.' then
              begin
                D := true;
                GetChar;
              end;
            NewTime := CalcTimeMidi(L, D);
            D := false;
            if CurChar='*' then
              begin
                D := true;
                GetChar;
              end;
          end
        else NewTime := 0;
        if FLastTime<>0 then
          begin
            CheckDTempo;
            FSleeping := true;
            if NewTime<>0 then
              begin
                AddTime(NewTime);
                if D then FLastTime := NewTime;
              end
            else AddTime(FLastTime);
          end;
      end;
    'O':
      begin
        O := GetInteger;
        CheckOctave(O);
        FOctave := O;
      end;
    'L':
      begin
        L := GetInteger;
        CheckDuration(L);
        FDuration := L;
      end;
    'V':
      begin
        O := GetInteger;
        CheckVelocity(O);
        FVelocity := O;
      end;
    '!':
      begin
        O := GetInteger;
        CheckChannel(O);
        FChannel := O;
        Cmds.Add('C'#$FF#$20+#1+Char(FChannel-1));
        WasChannel := true;
      end;
    'T':
      begin
        T := GetInteger;
        CheckTempo(T);
        FTempo := T;
        if Assigned(FSong) then FSong.SetTempo(T);
        Cmds.Add('C'#$FF#$51+#3+CalcMidiTempo);
        WasTempo := true;
      end;
    'S':
      begin
        O := GetInteger;
        CheckBank(O);
        ControlChange(FChannel-1, BankChangeControl, O)
      end;
    'I':
      begin
        O := GetInteger;
        CheckPatch(O);
        PatchChange(FChannel-1, O)
      end;
    '>':
      begin
        O := FOctave+1;
        CheckOctave(O);
        FOctave := O;
      end;
    '<':
      begin
        O := FOctave-1;
        CheckOctave(O);
        FOctave := O;
      end;
    '~':
      begin
        T := GetInteger;
        CheckBend(T);
        PitchBend(FChannel-1, T)
      end;
    '/':
      begin
        T := GetInteger;
        CheckBendUp(T);
        PitchBend(FChannel-1, $2000+T)
      end;
    '\':
      begin
        T := GetInteger;
        CheckBendDown(T);
        PitchBend(FChannel-1, $2000-T)
      end;
    '|':
      begin
        PitchBend(FChannel-1, $2000)
      end;
    '{':
      begin
        S := '';
        while not (CurChar in ['}', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          begin
            CheckDChannel;
            OEMToChar(PChar(S), PChar(S));
            Cmds.Add('C'+#$FF#1+EncodedTime(Length(S))+S);
          end;
        GetChar;
      end;
    '[':
      begin
        S := '';
        while not (CurChar in [']', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          begin
            CheckDChannel;
            OEMToChar(PChar(S), PChar(S));
            Cmds.Add('C'+#$FF#5+EncodedTime(Length(S))+S);
          end;
        GetChar;
      end;
    '(':
      begin
        S := '';
        while not (CurChar in [')', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          begin
            CheckDChannel;
            OEMToChar(PChar(S), PChar(S));
            Cmds.Add('C'+#$FF#6+EncodedTime(Length(S))+S);
          end;
        GetChar;
      end;
    '"':
      begin
        S := '';
        while not (CurChar in ['"', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          begin
            CheckDChannel;
            OEMToChar(PChar(S), PChar(S));
            Cmds.Add('C'+#$FF#3+EncodedTime(Length(S))+S);
          end;
        GetChar;
      end;
    '`':
      begin
        S := '';
        while not (CurChar in ['`', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          begin
            CheckDChannel;
            OEMToChar(PChar(S), PChar(S));
            Cmds.Add('C'+#$FF#4+EncodedTime(Length(S))+S);
          end;
        GetChar;
      end;
    '''':
      begin
        S := '';
        while not (CurChar in ['''', #0]) do
          begin
            S := S+CurChar;
            GetChar;
          end;
        if S<>'' then
          begin
            CheckDChannel;
            OEMToChar(PChar(S), PChar(S));
            Cmds.Add('C'+#$FF#2+EncodedTime(Length(S))+S);
          end;
        GetChar;
      end;
    '?':
      begin
        O := GetInteger;
        CheckBank(O);
        if CurChar=':' then
          begin
            GetChar;
            while not (CurChar in [';', #0]) do
              begin
                S := S+CurChar;
                GetChar;
              end;
            GetChar;
            S := 'LB'+Char(O)+S;
            Cmds.Add('C'+#$Ff#$7F+EncodedTime(Length(S))+S);
          end
        else
        if CurChar=';' then
          begin
            GetChar;
            S := 'CB'+Char(O);
            Cmds.Add('C'+#$Ff#$7F+EncodedTime(Length(S))+S);
          end
      end;
    else Error('Invalid command: '+Cmd);
  end;
  if CurChar<>#0 then Pos := Pos-1;
end;

var i, Time: Integer; S: String;
begin
  Ended := false;
  Pos := 1;
  WasTempo := false;
  Cmds := TStringList.Create;
  try
    //Time signature command 4/4, 8/32 per quarter
    Cmds.Add('C'+#$FF#$58#4#4#2#$18#8);
    //Key signature command
    Cmds.Add('C'+#$FF#$59#2#0#0);
    repeat
      ExportCommand;
    until Ended;
    result := '';
    Time := 0;
    for i := 0 to Cmds.Count-1 do
      begin
        S := Cmds[i];
        if S[1]='T' then Time := Time+StrToInt(Copy(S, 2, Length(S)));
        if S[1]='C' then
          begin
            if Time<>0 then
              begin
                result := result+EncodedTime(Time);
                Time := 0;
              end
            else result := result+#0;
            result := result+Copy(S, 2, Length(S));
          end;
      end;
      if Time<>0 then
        result := result+EncodedTime(Time)
      else result := result+#0;
      result := result+#$FF#$2F#0;
  finally
    Cmds.Free;
  end;
end;

{ TSong }

procedure TSong.Error(S: String);
begin
  Raise ESongError.Create(S);
end;

function TSong.GetTrackData(Track: Integer): String;
begin
  if (Track<0) or (Track>TracksCount-1) then
    Error('Invalid track number: '+IntToStr(Track));
  result := TTrack(FTracks[Track]).Data;
end;

function TSong.GetTrackTranslatedData(Track: Integer): String;
begin
  if (Track<0) or (Track>TracksCount-1) then
    Error('Invalid track number: '+IntToStr(Track));
  result := TTrack(FTracks[Track]).TranslatedData;
end;

procedure TSong.SetTrackData(Track: Integer; Data: String);
begin
  if (Track<0) or (Track>TracksCount-1) then
    Error('Invalid track number: '+IntToStr(Track));
  TTrack(FTracks[Track]).Data := Data;
end;

procedure TSong.SetTempo(Tempo: Word);
var i: Integer;
begin
  for i := 0 to TracksCount-1 do
    TTrack(FTracks[i]).FTempo := Tempo;
end;

constructor TSong.Create(MidiOut: TAbstractMidiOut);
begin
  FTracks := TList.Create;
  FPatterns := TStringList.Create;
  FMidiOut := MidiOut;
  DropoutDetect := false;
  DropoutCount := 100;
  BankChangeControl := 0;
  TextCallback := nil;
end;

destructor TSong.Destroy;
begin
  DeleteTracks;
  FTracks.Free;
  FPatterns.Free;
  inherited Destroy;
end;

function TSong.TracksCount: Integer;
begin
  result := FTracks.Count;
end;

procedure TSong.DeleteTracks;
var i: Integer;
begin
  for i := 0 to TracksCount-1 do
    TTrack(FTracks[i]).Free;
  FTracks.Clear;
end;

procedure TSong.AddTrack(Data: String);
var Track: TTrack;
begin
  Track := TTrack.Create(FMidiOut);
  Track.Patterns := FPatterns;
  Track.Data := Data;
  Track.DropoutDetect := DropoutDetect;
  Track.DropoutCount := DropoutCount;
  Track.BankChangeControl := BankChangeControl;
  Track.TextCallback := TextCallback;
  FTracks.Add(Track);
end;

function PlayTrack(Parameter: Pointer): Integer;
begin
  result := 0;
  TTrack(Parameter).Play;
end;

{procedure TSong.Play;
var i: Integer; Done: Boolean; TID: Integer; TL: TList; TP: Integer;
begin
  TL := TList.Create;
  TP := GetThreadPriority(GetCurrentThread);
  SetThreadPriority(GetCurrentThread, THREAD_PRIORITY_TIME_CRITICAL);
  try
    for i := 0 to FTracks.Count-1 do
      begin
        TL.Add(Pointer(BeginThread(NIL, 4096, @PlayTrack, FTracks[i],
                       CREATE_SUSPENDED, TID)));
        //SetThreadPriority(Integer(TL[i]), THREAD_PRIORITY_TIME_CRITICAL);
        SetThreadPriority(Integer(TL[i]), THREAD_PRIORITY_NORMAL);
      end;
    for i := 0 to TL.Count-1 do
      ResumeThread(Integer(TL[i]));
    SetThreadPriority(GetCurrentThread, THREAD_PRIORITY_LOWEST);
    Sleep(0);
    repeat
      Sleep(0);
      Done := true;
      for i := 0 to TracksCount-1 do
        with TTrack(FTracks[i]) do
          if Playing then Done := false;
    until Done;
  finally
    TL.Free;
    SetThreadPriority(GetCurrentThread, TP);
  end;
end;}

procedure TSong.Play;
var i: Integer; Done, First: Boolean;
begin
  First := true;
  repeat
    Done := true;
    for i := 0 to TracksCount-1 do
      with TTrack(FTracks[i]) do
        begin
          if First or Playing then ProcessCommand;
          if Playing then Done := false;
        end;
    First := false;
  until Done;
end;

procedure TSong.Export(FileName: String);
var F: TFileStream; S, T: String; i: Integer;

type
  TIntRec = Record
    case Boolean of
      false: (I: Integer);
      true: (B: Array[1..4] of Byte);
  end;

procedure Dump;
begin
  F.Write(S[1], Length(S));
end;

function GetByte(i: Integer; Num: Byte): Byte;
begin
  if (Num>4) or (Num=0) then result := 0
  else result := TIntRec(i).B[Num];
end;

begin
  if FTracks.Count>65535 then Error('Too many tracks: '+IntToStr(FTracks.Count));
  F := TFileStream.Create(FileName, fmCreate);
  try
    //Header
    S := #$4D#$54#$68#$64#0#0#0#$06+
      //Sync multiple tracks
      #0#1+
      //Number of tracks
      Char(FTracks.Count shr 8)+
      Char(FTracks.Count shl 8 shr 8)+
      //Ticks per quarter note
      Char(TPB shr 8)+Char(TPB shl 8 shr 8);
    Dump;
    for i := 0 to TracksCount-1 do
      with TTrack(FTracks[i]) do
        begin
          T := Export;
          S := #$4D#$54#$72#$6B+
            //Track length
            Char(Length(T) shr 24)+
            Char(Length(T) shl 8 shr 24)+
            Char(Length(T) shl 16 shr 24)+
            Char(Length(T) shl 24 shr 24)+
            //Track data
            T;
          Dump;
        end;
  finally
    F.Free;
  end;
end;

end.