unit Midi;

interface

uses MMSystem, SysUtils;

type
  TAbstractMidiOut = class
   protected
    function OutCmd(Cmd, Chn, D1, D2: Byte): MMResult; virtual;
    procedure Check(R: MMRESULT);
   public
    constructor Create;
    destructor Destroy; override;
    procedure NoteOn(Channel, Note, Velocity: Byte);
    procedure NoteOff(Channel, Note, Velocity: Byte);
    procedure ControlChange(Channel, Control, Value: Byte);
    procedure PatchChange(Channel, Patch: Byte);
    procedure PitchBend(Channel: Byte; Bend: Word);
  end;

  TMidiOut = class(TAbstractMidiOut)
   private
    MidiOut: HMIDIOUT;
   protected
    function OutCmd(Cmd, Chn, D1, D2: Byte): MMResult; override;
    procedure Check(R: MMRESULT);
   public
    constructor Create(DeviceID: Integer);
    constructor CreateMapper;
    destructor Destroy; override;
  end;

  EMidiError = class(Exception)
   public
    ErrorCode: MMResult;
    constructor Create(ErrCode: MMResult; AMessage: String);
  end;

implementation

{ EMidiError }

constructor EMidiError.Create(ErrCode: MMResult; AMessage: String);
begin
  inherited Create(AMessage);
  ErrorCode := ErrCode;
end;

{ TAbstractMidiOut }

function TAbstractMidiOut.OutCmd(Cmd, Chn, D1, D2: Byte): MMResult;
begin
  result := 0;
end;

procedure TAbstractMidiOut.Check(R: MMRESULT);
begin
end;

constructor TAbstractMidiOut.Create;
begin
  inherited Create;
end;

destructor TAbstractMidiOut.Destroy;
begin
  inherited Destroy;
end;

procedure TAbstractMidiOut.NoteOn(Channel, Note, Velocity: Byte);
begin
  Check(OutCmd($9, Channel, Note, Velocity));
end;

procedure TAbstractMidiOut.NoteOff(Channel, Note, Velocity: Byte);
begin
  Check(OutCmd($8, Channel, Note, Velocity));
end;

procedure TAbstractMidiOut.ControlChange(Channel, Control, Value: Byte);
begin
  Check(OutCmd($B, Channel, Control, Value));
end;

procedure TAbstractMidiOut.PatchChange(Channel, Patch: Byte);
begin
  Check(OutCmd($C, Channel, Patch, 0));
end;

procedure TAbstractMidiOut.PitchBend(Channel: Byte; Bend: Word);
begin
  Check(OutCmd($E, Channel, Bend mod 128, Bend div 128));
end;

{ TMidiOut }

function TMidiOut.OutCmd(Cmd, Chn, D1, D2: Byte): MMResult;
begin
  result := midiOutShortMsg(MidiOut,
    Cmd shl 4 or Chn or D1 shl 8 or D2 shl 16);
end;

procedure TMidiOut.Check(R: MMResult);
var C: Array[0..255] of Char;
begin
  if R=0 then Exit;
  if midiOutGetErrorText(R, C, SizeOf(C))=0 then
    Raise EMidiError.Create(R, C)
  else Raise EMidiError.Create(R, 'Unknown midi error '+IntToStr(R));
end;

constructor TMidiOut.Create(DeviceID: Integer);
begin
  inherited Create;
  Check(midiOutOpen(@MidiOut, DeviceID, 0, 0, CALLBACK_NULL));
end;

destructor TMidiOut.Destroy;
begin
  midiOutClose(MidiOut);
  inherited Destroy;
end;

constructor TMidiOut.CreateMapper;
begin
  Create(MIDI_MAPPER);
end;

end.