unit Player;

interface

uses Song, Windows, Messages, Classes, MIDI;

const
  WM_START = WM_USER+1;
  WM_STOP = WM_USER+2;
  WM_SETPPOS = WM_USER+3;
  WM_SETSPOS = WM_USER+4;
  WM_EXCEPT = WM_USER+7;

type
  TPlayingNote = class
   private
    FTrack: Integer;
    FChannel: Integer;
    FMIDIChannel: Byte;
    FBank: Byte;
    FPatch: Byte;
    FNote: Byte;
    FVelocity: Byte;
    procedure Start(MidiOut: TMidiOut);
    procedure Stop(MidiOut: TMidiOut);
  end;

  TPlayingNotes = class(TList)
   private
    function GetNote(i: Integer): TPlayingNote;
   public
    destructor Destroy; override;
    procedure Delete(i: Integer);
    procedure Clear;
    property Note[i: Integer]: TPlayingNote read GetNote; default;
  end;

  TPlayMode = (pmNone, pmInstrument, pmPattern, pmSong);

  TSongPlayer = class
   private
    //These are from main program
    FSong: TSong;
    FHandle: HWND;
    //These are created and destroyed automatically
    FMidiOut: TMidiOut;
    FPlayingNotes: TPlayingNotes;
    //Variables :)
    FPLayMode: TPlayMode;
    FPattern: Integer;
    FPos: Integer;
    FSPos: Integer;
    FLoopPattern: Boolean;
    FMidiDevice: Integer;
    FPlaying: Boolean;
    FTimerID: Integer;

    //Handlers
    procedure NextStep;
    procedure Start;
    procedure Stop;

    //Internal procedures
    procedure AdvanceSong;
    procedure DoAllNotesOff;
    procedure StartTimer;
   public
    ExceptMessage: String;
    constructor Create(ASong: TSong; AMidiDevice: Integer;
      AHandle: HWND; ALoopPattern: Boolean);
    destructor Destroy; override;
    procedure StartPlaySong;
    procedure StartPlayPattern(Pattern: Integer);
    procedure StartPlayInstrument(Instrument: Integer);
    //For editing audition
    procedure StartNote(Track, Channel: Integer; C: TCommand);
    procedure XStartNote(Track, Channel: Integer; C: TCommand);
    procedure StopNote(Track, Channel: Integer);
    procedure StopPlay;
    procedure ExportSong(FileName: String);
  end;

implementation

uses MSConsts, Notes, SysUtils, MMSystem;

{ TPlayingNote }

procedure TPlayingNote.Start(MidiOut: TMidiOut);
begin
  MidiOut.ControlChange(FMIDIChannel-1, BankSet, FBank);
  MidiOut.PatchChange(FMIDIChannel-1, FPatch);
  MidiOut.NoteOn(FMIDIChannel-1, FNote, FVelocity);
end;

procedure TPlayingNote.Stop(MidiOut: TMidiOut);
begin
  MidiOut.NoteOff(FMIDIChannel-1, FNote, FVelocity);
end;

{ TPlayingNotes }

function TPlayingNotes.GetNote(i: Integer): TPlayingNote;
begin
  result := TPlayingNote(Items[i-1]);
end;

destructor TPlayingNotes.Destroy;
begin
  Clear;
  inherited Destroy;
end;

procedure TPlayingNotes.Delete(i: Integer);
begin
  TPlayingNote(Items[i-1]).Free;
  inherited Delete(i-1);
end;

procedure TPlayingNotes.Clear;
begin
  while Count>0 do Delete(1);
end;

{ Timer handler procedure }

procedure TimerProc(uTimerID, uMessage: UINT;
                    dwUser, dw1, dw2: DWORD) stdcall;
begin
  with TSongPlayer(dwUser) do
    begin
      FTimerID := 0;
      NextStep;
    end;
end;

{ TSongPlayer }

procedure TSongPlayer.StartTimer;
begin
  FTimerID := timeSetEvent(FSong.TickTime, 0, TimerProc, Integer(Self),
    TIME_CALLBACK_FUNCTION or TIME_ONESHOT);
end;

procedure TSongPlayer.DoAllNotesOff;
var i: Byte;
begin
  for i := 0 to 15 do
    FMidiOut.ControlChange(i, AllNotesOff, 0);
  FPlayingNotes.Clear;
end;

procedure TSongPlayer.AdvanceSong;
var CP: Boolean; i, j: Integer;
begin
  CP := false;
  if FollowSong and (FHandle<>0) then
    PostMessage(FHandle, WM_SETSPOS, FSPos, 0);
  with FSong.Sequence do
    for i := 1 to FSong.Sequence.Tracks do
      begin
        //Skip not existing patterns
        while (Pos[i]>0) and (Pos[i]<=TrackLength[i]) and
              ((PatternNum[i, Pos[i]]<1) or
               (PatternNum[i, Pos[i]]>FSong.Patterns.Count)) do
        Pos[i] := Pos[i]+1;

        //Process every track
        if (Pos[i]>0) and (Pos[i]<=TrackLength[i]) then
          with FSong.Patterns[PatternNum[i, Pos[i]]] do
            begin
              //Pattern position OK
              if (PPos[i]>0) and (PPos[i]<=Length) then

                //Process every channel
                for j := 1 to Channels do
                  StartNote(i, j, Command[j, PPos[i]]);

              //Continue play
              CP := true;

              //Advance position in pattern
              PPos[i] := PPos[i]+1;

              //If pattern end, advance position in track
              if PPos[i]>Length then
                begin
                  PPos[i] := 1;
                  Pos[i] := Pos[i]+1;
                end;
            end;
        end;
  Inc(FSPos);
  //If song over - stop play
  if not CP then FPlaying := false;
end;

procedure TSongPlayer.NextStep;
var i: Integer;
begin
  if FPlaying then
    if FPlayMode=pmPattern then //Process standalone pattern
      begin
        if (FPattern<1) or (FPattern>FSong.Patterns.Count) or
           (FPos<1) or (FPos>FSong.Patterns[FPattern].Length) then
          begin
            FPlaying := false;
          end
        else
          begin
          if FollowPattern and (FHandle<>0) then
            PostMessage(FHandle, WM_SETPPOS, FPattern, FPos);
          i := 1;
          while i<=FSong.Patterns[FPattern].Channels do
            begin
              StartNote(1, i, FSong.Patterns[FPattern][i, FPos]);
              Inc(i);
            end;
            Inc(FPos);
            if FPos>FSong.Patterns[FPattern].Length then
              if FLoopPattern then FPos := 1;
          end;
      end
    else
    if FPlayMode=pmSong then //Process entire song
      AdvanceSong;
    if FPlaying then StartTimer
    else
      begin
        DoAllNotesOff;
        FPlayMode := pmNone;
        Stop;
      end;
end;

procedure TSongPlayer.Start;
begin
  if FHandle<>0 then
    PostMessage(FHandle, WM_START, Ord(FPlayMode), 0);
end;

procedure TSongPlayer.Stop;
begin
  FPlayMode := pmNone;
  FPlaying := false;
  if FHandle<>0 then
    PostMessage(FHandle, WM_STOP, 0, 0);
end;

procedure TSongPlayer.StartNote(Track, Channel: Integer; C: TCommand);
var PN: TPlayingNote;
begin
  if C.Note=NoteOff then
    begin
      StopNote(Track, Channel);
      Exit;
    end;
  if (C.Note=NoNote) or (C.Note>MaxNote) then Exit;
  if (C.Instrument<1) or (C.Instrument>FSong.Instruments.Count) then Exit;
  if FSong.NNA=nnaChannelOff then StopNote(Track, Channel);
  PN := TPlayingNote.Create;
  with PN do
    try
      FTrack := Track;
      FChannel := Channel;
      FMIDIChannel := FSong.Instruments[C.Instrument].Channel;
      FBank := FSong.Instruments[C.Instrument].Bank;
      FPatch := FSong.Instruments[C.Instrument].Patch;
      FNote := C.Note;
      FVelocity := C.Velocity;
      if FVelocity=DVel then FVelocity :=
        FSong.Instruments[C.Instrument].DefVelocity;
      PN.Start(FMidiOut);
      FPlayingNotes.Add(PN);
    except
      PN.Free;
    end;
end;

procedure TSongPlayer.StopNote(Track, Channel: Integer);
var i: Integer;
begin
  i := 1;
  while i<=FPlayingNotes.Count do
    begin
      if (FPlayingNotes[i].FTrack=Track) and
         (FPlayingNotes[i].FChannel=Channel) then
        begin
          FPlayingNotes[i].Stop(FMidiOut);
          FPlayingNotes.Delete(i)
        end
      else Inc(i);
    end;
  if (FPlayingNotes.Count=0) and (FPlayMode=pmInstrument) then Stop;
end;

constructor TSongPlayer.Create(ASong: TSong; AMidiDevice: Integer;
  AHandle: HWND; ALoopPattern: Boolean);
begin
  FSong := ASong;
  FHandle := AHandle;
  FMidiDevice := AMidiDevice;
  FPlayingNotes := TPlayingNotes.Create;
  try
    FMidiOut := TMidiOut.Create(FMidiDevice);
  except
    Raise Exception.Create('Unable to open MIDI device: '#13+
                           (ExceptObject as Exception).Message);
  end;
  FPLayMode := pmNone;
  FPlaying := false;
  FPattern := 0;
  FPos := 1;
  FLoopPattern := ALoopPattern;
  FTimerID := 0;
end;

destructor TSongPlayer.Destroy;
begin
  StopPlay;
  if Assigned(FMidiOut) then FMidiOut.Free;
  if Assigned(FPlayingNotes) then FPlayingNotes.Free;
end;

procedure TSongPlayer.StartPlaySong;
var i: Integer;
begin
  StopPlay;
  FSPos := 1;
  for i := 1 to FSong.Sequence.Tracks do
    begin
      FSong.Sequence.Pos[i] := 1;
      FSong.Sequence.PPos[i] := 1;
    end;
  FPlaying := true;
  FPlayMode := pmSong;
  Start;
  StartTimer;
end;

procedure TSongPlayer.StartPlayPattern(Pattern: Integer);
begin
  StopPlay;
  FPattern := Pattern;
  FPos := 1;
  FPlaying := true;
  FPlayMode := pmPattern;
  Start;
  StartTimer;
end;

procedure TSongPlayer.XStartNote(Track, Channel: Integer; C: TCommand);
begin
  StartNote(Track, Channel, C);
  if FPlayMode=pmNone then
    begin
      FPlaying := false;
      FPlayMode := pmInstrument;
      Start;
    end;
end;

procedure TSongPlayer.StartPlayInstrument(Instrument: Integer);
var C: TCommand;
begin
  C.Note := BaseNote;
  C.Instrument := Instrument;
  C.Velocity := DVel;
  XStartNote(0, 0, C);
end;

procedure TSongPlayer.StopPlay;
begin
  if FPlayMode=pmNone then Exit;
  DoAllNotesOff;
  if FTimerID<>0 then TimeKillEvent(FTimerID);
  Stop;
end;

procedure TSongPlayer.ExportSong(FileName: String);
const _TPB=128;
var F: TFileStream;
    S: String; i: Integer; ES: Boolean; Time: Integer;

function EncNum(Time: Integer): String;
var B: Byte; R: Integer; S: String;
begin
  S := '';
  R := Time;
  repeat
    B := R mod 128;
    R := R div 128;
    S := S+Char(B or 128);
  until R=0;
  result := '';
  S[1] := Char(Byte(S[1]) and not 128);
  for R := Length(S) downto 1 do
    result := result+S[R];
end;

procedure AddTrackHdr;
begin
  S :=
    //Time signature command 4/4, 8/32 per quarter
    #0#$FF#$58#4#4#2#$18#8+
    //Key signature command
    #0#$FF#$59#2#0#0+S;
  //Track end command
  S := S+EncNum(Time*_TPB div FSong.TPB)+#$FF#$2F#0;
  S := #$4D#$54#$72#$6B+
    //Track length
    Char(Length(S) shr 24)+
    Char(Length(S) shl 8 shr 24)+
    Char(Length(S) shl 16 shr 24)+
    Char(Length(S) shl 24 shr 24)+
    //Track data
    S;
end;

function CalcMidiTempo: String;
var I: Integer;
begin
  I := 60000000 div FSong.Tempo;
  result := Char(I shl 8 shr 24)+Char(I shl 16 shr 24)+Char(I shl 24 shr 24);
end;

procedure Dump;
begin
  if S<>'' then F.Write(S[1], Length(S));
end;

procedure AddCmd(Cmd, Chn, D1, D2: Byte);
begin
  S := S+Char(Cmd shl 4 or Chn)+Char(D1)+Char(D2);
end;

procedure AddCmd3(Cmd, Chn, D1: Byte);
begin
  S := S+Char(Cmd shl 4 or Chn)+Char(D1);
end;

procedure _NoteOn(Channel, Note, Velocity: Byte);
begin
  AddCmd($9, Channel, Note, Velocity);
end;

procedure _NoteOff(Channel, Note, Velocity: Byte);
begin
  AddCmd($8, Channel, Note, Velocity);
end;

procedure _ControlChange(Channel, Control, Value: Byte);
begin
  AddCmd($B, Channel, Control, Value);
end;

procedure _PatchChange(Channel, Patch: Byte);
begin
  AddCmd3($C, Channel, Patch);
end;

procedure StopNote(Track, Channel: Integer);
var i: Integer;
begin
  i := 1;
  while i<=FPlayingNotes.Count do
    begin
      if (FPlayingNotes[i].FTrack=Track) and
         (FPlayingNotes[i].FChannel=Channel) then
        begin
          S := S+EncNum(Time*_TPB div FSong.TPB);
          Time := 0;
          _NoteOff(FPlayingNotes[i].FMIDIChannel-1,
                  FPlayingNotes[i].FNote,
                  FPlayingNotes[i].FVelocity);
          FPlayingNotes.Delete(i)
        end
      else Inc(i);
    end;
  if (FPlayingNotes.Count=0) and (FPlayMode=pmInstrument) then Stop;
end;

procedure StartNote(Track, Channel: Integer; C: TCommand);
var PN: TPlayingNote;
begin
  if C.Note=NoteOff then
    begin
      StopNote(Track, Channel);
      Exit;
    end;
  if (C.Note=NoNote) or (C.Note>MaxNote) then Exit;
  if (C.Instrument<1) or (C.Instrument>FSong.Instruments.Count) then Exit;
  if FSong.NNA=nnaChannelOff then StopNote(Track, Channel);
  PN := TPlayingNote.Create;
  with PN do
    try
      FTrack := Track;
      FChannel := Channel;
      FMIDIChannel := FSong.Instruments[C.Instrument].Channel;
      FBank := FSong.Instruments[C.Instrument].Bank;
      FPatch := FSong.Instruments[C.Instrument].Patch;
      FNote := C.Note;
      FVelocity := C.Velocity;
      if FVelocity=DVel then FVelocity :=
        FSong.Instruments[C.Instrument].DefVelocity;

      S := S+EncNum(Time*_TPB div FSong.TPB);
      Time := 0;
      _ControlChange(FMIDIChannel-1, BankSet, FBank);
      S := S+#0;
      _PatchChange(FMIDIChannel-1, FPatch);
      S := S+#0;
      _NoteOn(FMIDIChannel-1, FNote, FVelocity);

      FPlayingNotes.Add(PN);
    except
      PN.Free;
    end;
end;

procedure ExportTrack(i: Integer);
var j: Integer; CP: Boolean;
begin
  S := '';
  Time := 0;
  repeat
    CP := false;
    //Skip not existing patterns
    with FSong.Sequence do
      begin
        while (Pos[i]>0) and (Pos[i]<=TrackLength[i]) and
          ((PatternNum[i, Pos[i]]<1) or
                 (PatternNum[i, Pos[i]]>FSong.Patterns.Count)) do
          Pos[i] := Pos[i]+1;

          //Process every track
          if (Pos[i]>0) and (Pos[i]<=TrackLength[i]) then
            with FSong.Patterns[PatternNum[i, Pos[i]]] do
              begin
                //Pattern position OK
                if (PPos[i]>0) and (PPos[i]<=Length) then

                  //Process every channel
                  for j := 1 to Channels do
                  StartNote(i, j, Command[j, PPos[i]]);

                Time := Time+1;

                //Continue play
                CP := true;

                //Advance position in pattern
                PPos[i] := PPos[i]+1;

                //If pattern end, advance position in track
                if PPos[i]>Length then
                  begin
                    PPos[i] := 1;
                    Pos[i] := Pos[i]+1;
                  end;
              end;
      end;
  until not CP;

  if S<>'' then AddTrackHdr;
end;

procedure MakeHeader;
begin
  S := #$4D#$54#$68#$64#0#0#0#$06+
    //Sync multiple tracks
    #0#1+
    //Number of tracks
    Char((FSong.Sequence.Tracks+1) shr 8)+
    Char((FSong.Sequence.Tracks+1) shl 8 shr 8)+
    //Ticks per quarter note
    Char(_TPB shr 8)+Char(_TPB shl 8 shr 8);
end;

procedure MakeInfoTrack;
begin
  S := '';
  //Author
  if FSong.Author<>'' then
    S := S+#0#$FF#$02+EncNum(Length(FSong.Author))+FSong.Author;
  //Title
  if FSong.Title<>'' then
    S := S+#0#$FF#$03+EncNum(Length(FSong.Title))+FSong.Title;
  //Info
  if FSong.Info<>'' then
    S := S+#0#$FF#$01+EncNum(Length(FSong.Info))+FSong.Info;
  S := S+
    #0#$FF#$51+#3+CalcMidiTempo;
  if S<>'' then AddTrackHdr;
end;

begin
  if FSong.Sequence.Tracks>65535 then
    Raise Exception.Create('Too many tracks: '+IntToStr(FSong.Sequence.Tracks));
  ES := true;
  for i := 1 to FSong.Sequence.Tracks do
    if FSong.Sequence.TrackLength[i]>0 then
      begin
        ES := false;
        Break;
      end;
  if ES then Exit;
  StopPlay;
  for i := 1 to FSong.Sequence.Tracks do
    begin
      FSong.Sequence.Pos[i] := 1;
      FSong.Sequence.PPos[i] := 1;
    end;

  F := TFileStream.Create(FileName, fmCreate or fmShareDenyWrite);
  try
    MakeHeader;
    Dump;
    MakeInfoTrack;
    Dump;
    for i := 1 to FSong.Sequence.Tracks do
      begin
        ExportTrack(i);
        Dump;
      end;
    StopPlay;
  finally
    F.Free;
  end;
end;

end.
