Unit SFMan;

interface

uses SFMan_h, Windows;

type
  TSFMan = class
    private
      DLL: HMODULE;
      FuncTable: PSFMANAGER;
      SFMan: PSFMANL101API;
      Opened: Boolean;
      DevID: Integer;
      procedure Error(Msg: String);
    public
     constructor Create;
     destructor Destroy; override;
     function SFManager: SFMANL101API;
     function GetErrorText(ErrorCode: Integer): String;
     procedure Check(ErrorCode: Integer);
     function GetNumDevs: Integer;
     function DevName(ID: Integer): String;
     procedure Open(ID: Integer);
     procedure Close;
     function IsMIDIBankUsed(BankNo: Byte): Word;
     procedure LoadBank(BankNo: Byte; FileName: String);
     procedure LoadBankMem(BankNo: Byte; P: Pointer; Size: Integer);
     procedure ClearBank(BankNo: Byte);
     procedure ClearAllBanks;
     function BankName(BankNo: Byte): String;
     function BankFileName(BankNo: Byte): String;
     function PatchName(BankNo, PatchNo: Byte): String;
     function PercPatchName(PatchNo: Byte): String;
     property IsOpened: Boolean read Opened;
  end;

implementation

uses SysUtils;

procedure TSFMan.Error(Msg: String);
begin
  Raise Exception.Create(Msg);
end;

constructor TSFMan.Create;
begin
  inherited Create;
  Opened := false;
  DevID := -1;

  DLL := LoadLibrary(SF_MASTER_MANAGER_FILENAME);
  if DLL=INVALID_HANDLE_VALUE then
    Check(SFERR_LOAD_DLL_FAIL);

  FuncTable := GetProcAddress(DLL, SF_FUNCTION_TABLE_NAME);
  if FuncTable=nil then
    Check(SFERR_GET_ENTRYPOINT_FAIL);

  if FuncTable^.SF_QueryInterface(ID_SFMANL101API, @SFMan)<>0 then
    Check(SFERR_INTERFACE_NOT_SUPPORTED);
end;

destructor TSFMan.Destroy;
begin
  if Opened then Close;
  if DLL<>INVALID_HANDLE_VALUE then FreeLibrary(DLL);
end;

function TSFMan.SFManager: SFMANL101API;
begin
  result := SFMan^;
end;

function TSFMan.GetErrorText(ErrorCode: Integer): String;
begin
  case ErrorCode of
    SFERR_NOERR:                   result := '';
    SFERR_UNDEFINE_ERROR:          result := 'Undefined Error';
    SFERR_LOAD_DLL_FAIL:           result := 'Load Dll Failed';
    SFERR_GET_ENTRYPOINT_FAIL:     result := 'Get Entrypoint Fail';
    SFERR_DEVICE_NOT_FOUND:        result := 'Device Not Found';
    SFERR_INTERFACE_NOT_SUPPORTED: result := 'Interface Not Supported';
    SFERR_LAST_API:                result := 'Last API';
    SFERR_SYNTH_INVALID:           result := 'Synth Invalid';
    SFERR_BANK_INDEX_INVALID:      result := 'Bank Index Invalid';
    SFERR_PATHNAME_INVALID:        result := 'Pathname Invalid';
    SFERR_FORMAT_INVALID:          result := 'Format Invalid';
    SFERR_PRESET_INDEX_INVALID:    result := 'Preset Index Invalid';
    SFERR_ACCESS_NOT_PERMITTED:    result := 'Access Not Permitted';
    SFERR_VERSION_INVALID:         result := 'Version Invalid';
    SFERR_DEVICE_INVALID:          result := 'Device Invalid';
    SFERR_SUPPORT_INVALID:         result := 'Support Invalid';
    SFERR_INVALID_SOUND_ROM:       result := 'Invalid Sound Rom';
    SFERR_NOT_STATIC_MEMORY:       result := 'Not Static Memory';
    SFERR_LAST_INVALID:            result := 'Last Invalid';
    SFERR_DLL_BUSY:                result := 'Dll Busy';
    SFERR_DEVICE_BUSY:             result := 'Device Busy';
    SFERR_RESOURCE_INSUFFICIENT:   result := 'Resource Insufficient';
    SFERR_SYSMEM_INSUFFICIENT:     result := 'System Memory Insufficient';
    SFERR_SOUNDMEM_INSUFFICIENT:   result := 'Sound Memory Insufficient';
    SFERR_LAST_CONTENTION:         result := 'Last Contention';
    else
      result := 'Unknown error code '+IntToStr(ErrorCode);
  end;
end;

procedure TSFMan.Check(ErrorCode: Integer);
begin
  if ErrorCode<>0 then
    Error(GetErrorText(ErrorCode));
end;

function TSFMan.GetNumDevs: Integer;
begin
  result := 0;
  Check(SFMan^.SF_GetNumDevs(@result));
end;

function TSFMan.DevName(ID: Integer): String;
var CO: CSFCapsObject;
begin
  result := '';
  if (ID<0) or (ID>=GetNumDevs) then
    Check(SFERR_DEVICE_NOT_FOUND);
  FillChar(CO, SizeOf(CO), 0);
  Check(SFMan^.SF_GetDevCaps(ID, @CO));
  result := CO.m_DevName;
end;

procedure TSFMan.Open(ID: Integer);
begin
  if (ID<0) or (ID>=GetNumDevs) then
    Check(SFERR_DEVICE_NOT_FOUND);
  if Opened then Close;
  Check(SFMan^.SF_Open(ID));
  Opened := true;
  DevID := ID;
end;

procedure TSFMan.Close;
begin
  if not Opened then Exit;
  Check(SFMan^.SF_Close(DevID));
  Opened := false;
end;

function TSFMan.IsMIDIBankUsed(BankNo: Byte): Word;
var Index: Word;
begin
  Index := BankNo;
  Check(SFMan^.SF_IsMIDIBankUsed(DevID, @Index));
  result := Index;
end;

procedure TSFMan.LoadBank(BankNo: Byte; FileName: String);
var ML: CSFMIDILocation;
    Buf: CSFBufferObject;
    TextBuf: array[0..1023] of char;
begin
  if not Opened then Error('Device not opened');
  if Length(FileName)>1023 then FileName := Copy(FileName, 1, 1023);
  Move(FileName[1], TextBuf, Length(FileName));
  TextBuf[Length(FileName)] := #0;
  ML.m_BankIndex := BankNo;
  Buf.m_Flag := Byte(SFFLAG_OPER_FILE);
  Buf.m_Buffer := @TextBuf;
  Buf.m_Size := Length(FileName)+1;
  Check(SFMan^.SF_LoadBank(DevID, @ML, @Buf));
end;

procedure TSFMan.LoadBankMem(BankNo: Byte; P: Pointer; Size: Integer);
var ML: CSFMIDILocation;
    Buf: CSFBufferObject;
begin
  if not Opened then Error('Device not opened');
  ML.m_BankIndex := BankNo;
  Buf.m_Flag := Byte(SFFLAG_OPER_MEMORY);
  Buf.m_Buffer := P;
  Buf.m_Size := Size;
  Check(SFMan^.SF_LoadBank(DevID, @ML, @Buf));
end;

procedure TSFMan.ClearBank(BankNo: Byte);
var ML:CSFMIDILocation;
    Done:boolean;
begin
  if not Opened then Error('Device not opened');
  ML.m_BankIndex := BankNo;
  repeat
    Check(SFMan^.SF_ClearLoadedBank(DevID, @ML));
    Done := IsMIDIBankUsed(BankNo)<>BankNo;
  until Done;
end;

procedure TSFMan.ClearAllBanks;
var i: Byte;
begin
  if not Opened then Error('Device not opened');
  for i := 1 to 127 do
    if IsMIDIBankUsed(i)=i then ClearBank(i);
end;

function TSFMan.BankName(BankNo: Byte): String;
var ML:CSFMIDILocation;
    Buf: CSFBufferObject;
    TextBuf: array[0..1023] of char;
begin
  result := '';
  FillChar(TextBuf, SizeOf(TextBuf), 0);
  if not Opened then Error('Device not opened');
  ML.m_BankIndex := BankNo;
  Buf.m_Flag := Byte(SFFLAG_OPER_MEMORY);
  Buf.m_Buffer := @TextBuf;
  Buf.m_Size := SizeOf(TextBuf);
  Check(SFMan^.SF_GetLoadedBankDescriptor(DevID, @ML, @Buf));
  result := TextBuf;
end;

function TSFMan.BankFileName(BankNo: Byte): String;
var ML:CSFMIDILocation;
    Buf: CSFBufferObject;
    TextBuf: array[0..1023] of char;
begin
  result := '';
  FillChar(TextBuf, SizeOf(TextBuf), 0);
  if not Opened then Error('Device not opened');
  ML.m_BankIndex := BankNo;
  Buf.m_Flag := Byte(SFFLAG_OPER_MEMORY);
  Buf.m_Buffer := @TextBuf;
  Buf.m_Size := SizeOf(TextBuf);
  Check(SFMan^.SF_GetLoadedBankPathName(DevID, @ML, @Buf));
  result := TextBuf;
end;

function TSFMan.PatchName(BankNo, PatchNo: Byte): String;
var ML:CSFMIDILocation;
    Buf: CSFBufferObject;
    TextBuf: array[0..1023] of char;
begin
  result := '';
  FillChar(TextBuf, SizeOf(TextBuf), 0);
  if not Opened then Error('Device not opened');
  ML.m_BankIndex := BankNo;
  ML.m_PresetIndex := PatchNo;
  Buf.m_Flag := Byte(SFFLAG_OPER_MEMORY);
  Buf.m_Buffer := @TextBuf;
  Buf.m_Size := SizeOf(TextBuf);
  Check(SFMan^.SF_GetLoadedPresetDescriptor(DevID, @ML, @Buf));
  result := TextBuf;
end;

function TSFMan.PercPatchName(PatchNo: Byte): String;
var ML:CSFMIDILocation;
    Buf: CSFBufferObject;
    TextBuf: array[0..1023] of char;
begin
  result := '';
  FillChar(TextBuf, SizeOf(TextBuf), 0);
  if not Opened then Error('Device not opened');
  ML.m_BankIndex := SFMAN_PERCUSSIVE_BANK;
  ML.m_PresetIndex := PatchNo;
  Buf.m_Flag := Byte(SFFLAG_OPER_MEMORY);
  Buf.m_Buffer := @TextBuf;
  Buf.m_Size := SizeOf(TextBuf);
  Check(SFMan^.SF_GetLoadedPresetDescriptor(DevID, @ML, @Buf));
  result := TextBuf;
end;

end.